/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.core.segment.processing.framework;

import com.google.common.base.Preconditions;
import java.io.File;
import java.util.Arrays;
import org.apache.commons.io.FileUtils;
import org.apache.pinot.common.utils.TarGzCompressionUtils;
import org.apache.pinot.core.indexsegment.generator.SegmentGeneratorConfig;
import org.apache.pinot.core.segment.creator.impl.SegmentIndexCreationDriverImpl;
import org.apache.pinot.core.segment.processing.framework.SegmentMapper;
import org.apache.pinot.core.segment.processing.framework.SegmentMapperConfig;
import org.apache.pinot.core.segment.processing.framework.SegmentProcessorConfig;
import org.apache.pinot.core.segment.processing.framework.SegmentReducer;
import org.apache.pinot.core.segment.processing.framework.SegmentReducerConfig;
import org.apache.pinot.spi.config.table.TableConfig;
import org.apache.pinot.spi.data.Schema;
import org.apache.pinot.spi.data.readers.FileFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SegmentProcessorFramework {
    private static final Logger LOGGER = LoggerFactory.getLogger(SegmentProcessorFramework.class);
    private final File _inputSegmentsDir;
    private final File _outputSegmentsDir;
    private final SegmentProcessorConfig _segmentProcessorConfig;
    private final Schema _pinotSchema;
    private final TableConfig _tableConfig;
    private final File _baseDir;
    private final File _mapperInputDir;
    private final File _mapperOutputDir;
    private final File _reducerOutputDir;

    public SegmentProcessorFramework(File inputSegmentsDir, SegmentProcessorConfig segmentProcessorConfig, File outputSegmentsDir) {
        LOGGER.info("Initializing SegmentProcessorFramework with input segments dir: {}, output segments dir: {} and segment processor config: {}", new Object[]{inputSegmentsDir.getAbsolutePath(), outputSegmentsDir.getAbsolutePath(), segmentProcessorConfig.toString()});
        this._inputSegmentsDir = inputSegmentsDir;
        Preconditions.checkState((this._inputSegmentsDir.exists() && this._inputSegmentsDir.isDirectory() ? 1 : 0) != 0, (String)"Input path: %s must be a directory with Pinot segments", (Object)this._inputSegmentsDir.getAbsolutePath());
        this._outputSegmentsDir = outputSegmentsDir;
        Preconditions.checkState((this._outputSegmentsDir.exists() && this._outputSegmentsDir.isDirectory() && this._outputSegmentsDir.list().length == 0 ? 1 : 0) != 0, (String)"Must provide existing empty output directory: %s", (Object)this._outputSegmentsDir.getAbsolutePath());
        this._segmentProcessorConfig = segmentProcessorConfig;
        this._pinotSchema = segmentProcessorConfig.getSchema();
        this._tableConfig = segmentProcessorConfig.getTableConfig();
        this._baseDir = new File(FileUtils.getTempDirectory(), "segment_processor_" + System.currentTimeMillis());
        FileUtils.deleteQuietly((File)this._baseDir);
        Preconditions.checkState((boolean)this._baseDir.mkdirs(), (String)"Failed to create base directory: %s for SegmentProcessor", (Object)this._baseDir);
        this._mapperInputDir = new File(this._baseDir, "mapper_input");
        Preconditions.checkState((boolean)this._mapperInputDir.mkdirs(), (String)"Failed to create mapper input directory: %s for SegmentProcessor", (Object)this._mapperInputDir);
        this._mapperOutputDir = new File(this._baseDir, "mapper_output");
        Preconditions.checkState((boolean)this._mapperOutputDir.mkdirs(), (String)"Failed to create mapper output directory: %s for SegmentProcessor", (Object)this._mapperOutputDir);
        this._reducerOutputDir = new File(this._baseDir, "reducer_output");
        Preconditions.checkState((boolean)this._reducerOutputDir.mkdirs(), (String)"Failed to create reducer output directory: %s for SegmentProcessor", (Object)this._reducerOutputDir);
    }

    public void processSegments() throws Exception {
        File[] segmentFiles = this._inputSegmentsDir.listFiles();
        if (segmentFiles.length == 0) {
            throw new IllegalStateException("No segments found in input dir: " + this._inputSegmentsDir.getAbsolutePath() + ". Exiting SegmentProcessorFramework.");
        }
        LOGGER.info("Beginning mapper phase. Processing segments: {}", (Object)Arrays.toString(this._inputSegmentsDir.list()));
        for (File segment : segmentFiles) {
            String fileName = segment.getName();
            File mapperInput = segment;
            if (!segment.isDirectory()) {
                if (fileName.endsWith(".tar.gz") || fileName.endsWith(".tgz")) {
                    mapperInput = (File)TarGzCompressionUtils.untar((File)segment, (File)this._mapperInputDir).get(0);
                } else {
                    throw new IllegalStateException("Unsupported segment format: " + segment.getAbsolutePath());
                }
            }
            SegmentMapperConfig mapperConfig = new SegmentMapperConfig(this._pinotSchema, this._segmentProcessorConfig.getRecordTransformerConfig(), this._segmentProcessorConfig.getRecordFilterConfig(), this._segmentProcessorConfig.getPartitionerConfigs());
            SegmentMapper mapper = new SegmentMapper(mapperInput.getName(), mapperInput, mapperConfig, this._mapperOutputDir);
            mapper.map();
            mapper.cleanup();
        }
        File[] mapperOutputFiles = this._mapperOutputDir.listFiles();
        if (mapperOutputFiles.length == 0) {
            throw new IllegalStateException("No files found in mapper output directory: " + this._mapperOutputDir.getAbsolutePath() + ". Exiting SegmentProcessorFramework.");
        }
        LOGGER.info("Beginning reducer phase. Processing files: {}", (Object)Arrays.toString(this._mapperOutputDir.list()));
        for (File partDir : mapperOutputFiles) {
            SegmentReducerConfig reducerConfig = new SegmentReducerConfig(this._pinotSchema, this._segmentProcessorConfig.getCollectorConfig(), this._segmentProcessorConfig.getSegmentConfig().getMaxNumRecordsPerSegment());
            SegmentReducer reducer = new SegmentReducer(partDir.getName(), partDir, reducerConfig, this._reducerOutputDir);
            reducer.reduce();
            reducer.cleanup();
        }
        File[] reducerOutputFiles = this._reducerOutputDir.listFiles();
        if (reducerOutputFiles.length == 0) {
            throw new IllegalStateException("No files found in reducer output directory: " + this._reducerOutputDir.getAbsolutePath() + ". Exiting SegmentProcessorFramework.");
        }
        LOGGER.info("Beginning segment generation phase. Processing files: {}", (Object)Arrays.toString(this._reducerOutputDir.list()));
        int segmentNum = 0;
        for (File resultFile : reducerOutputFiles) {
            SegmentGeneratorConfig segmentGeneratorConfig = new SegmentGeneratorConfig(this._tableConfig, this._pinotSchema);
            segmentGeneratorConfig.setTableName(this._tableConfig.getTableName());
            segmentGeneratorConfig.setOutDir(this._outputSegmentsDir.getAbsolutePath());
            segmentGeneratorConfig.setInputFilePath(resultFile.getAbsolutePath());
            segmentGeneratorConfig.setFormat(FileFormat.AVRO);
            segmentGeneratorConfig.setSequenceId(segmentNum++);
            SegmentIndexCreationDriverImpl driver = new SegmentIndexCreationDriverImpl();
            driver.init(segmentGeneratorConfig);
            driver.build();
        }
        LOGGER.info("Successfully converted segments from: {} to {}", (Object)this._inputSegmentsDir, (Object)Arrays.toString(this._outputSegmentsDir.list()));
    }

    public void cleanup() {
        FileUtils.deleteQuietly((File)this._baseDir);
    }
}

