/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.core.query.scheduler;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import com.google.common.util.concurrent.RateLimiter;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.LongAccumulator;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.pinot.common.exception.QueryException;
import org.apache.pinot.common.metrics.AbstractMetrics;
import org.apache.pinot.common.metrics.ServerMeter;
import org.apache.pinot.common.metrics.ServerMetrics;
import org.apache.pinot.common.metrics.ServerQueryPhase;
import org.apache.pinot.common.metrics.ServerTimer;
import org.apache.pinot.common.response.ProcessingException;
import org.apache.pinot.common.utils.DataTable;
import org.apache.pinot.core.common.datatable.DataTableImplV2;
import org.apache.pinot.core.query.executor.QueryExecutor;
import org.apache.pinot.core.query.request.ServerQueryRequest;
import org.apache.pinot.core.query.request.context.TimerContext;
import org.apache.pinot.core.query.scheduler.resources.ResourceManager;
import org.apache.pinot.spi.env.PinotConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class QueryScheduler {
    private static final Logger LOGGER = LoggerFactory.getLogger(QueryScheduler.class);
    private static final String INVALID_NUM_SCANNED = "-1";
    private static final String INVALID_SEGMENTS_COUNT = "-1";
    private static final String INVALID_FRESHNESS_MS = "-1";
    private static final String QUERY_LOG_MAX_RATE_KEY = "query.log.maxRatePerSecond";
    private static final double DEFAULT_QUERY_LOG_MAX_RATE = 10000.0;
    private final RateLimiter queryLogRateLimiter;
    private final RateLimiter numDroppedLogRateLimiter;
    private final AtomicInteger numDroppedLogCounter;
    protected final ServerMetrics serverMetrics;
    protected final QueryExecutor queryExecutor;
    protected final ResourceManager resourceManager;
    protected final LongAccumulator latestQueryTime;
    protected volatile boolean isRunning = false;

    public QueryScheduler(@Nonnull PinotConfiguration config, @Nonnull QueryExecutor queryExecutor, @Nonnull ResourceManager resourceManager, @Nonnull ServerMetrics serverMetrics, @Nonnull LongAccumulator latestQueryTime) {
        Preconditions.checkNotNull((Object)config);
        Preconditions.checkNotNull((Object)queryExecutor);
        Preconditions.checkNotNull((Object)resourceManager);
        Preconditions.checkNotNull((Object)serverMetrics);
        this.serverMetrics = serverMetrics;
        this.resourceManager = resourceManager;
        this.queryExecutor = queryExecutor;
        this.latestQueryTime = latestQueryTime;
        this.queryLogRateLimiter = RateLimiter.create((double)config.getProperty(QUERY_LOG_MAX_RATE_KEY, 10000.0));
        this.numDroppedLogRateLimiter = RateLimiter.create((double)1.0);
        this.numDroppedLogCounter = new AtomicInteger(0);
        LOGGER.info("Query log max rate: {}", (Object)this.queryLogRateLimiter.getRate());
    }

    @Nonnull
    public abstract ListenableFuture<byte[]> submit(@Nonnull ServerQueryRequest var1);

    public abstract String name();

    public void start() {
        this.isRunning = true;
    }

    public void stop() {
        this.isRunning = false;
    }

    protected ListenableFutureTask<byte[]> createQueryFutureTask(@Nonnull ServerQueryRequest queryRequest, @Nonnull ExecutorService executorService) {
        return ListenableFutureTask.create(() -> this.processQueryAndSerialize(queryRequest, executorService));
    }

    @Nullable
    protected byte[] processQueryAndSerialize(@Nonnull ServerQueryRequest queryRequest, @Nonnull ExecutorService executorService) {
        DataTable dataTable;
        this.latestQueryTime.accumulate(System.currentTimeMillis());
        try {
            dataTable = this.queryExecutor.processQuery(queryRequest, executorService);
        }
        catch (Exception e) {
            LOGGER.error("Encountered exception while processing requestId {} from broker {}", new Object[]{queryRequest.getRequestId(), queryRequest.getBrokerId(), e});
            this.serverMetrics.addMeteredGlobalValue((AbstractMetrics.Meter)ServerMeter.UNCAUGHT_EXCEPTIONS, 1L);
            dataTable = new DataTableImplV2();
            dataTable.addException(QueryException.getException((ProcessingException)QueryException.INTERNAL_ERROR, (Exception)e));
        }
        long requestId = queryRequest.getRequestId();
        Map dataTableMetadata = dataTable.getMetadata();
        dataTableMetadata.put("requestId", Long.toString(requestId));
        byte[] responseData = this.serializeDataTable(queryRequest, dataTable);
        String tableNameWithType = queryRequest.getTableNameWithType();
        long numDocsScanned = Long.parseLong(dataTableMetadata.getOrDefault("numDocsScanned", "-1"));
        long numEntriesScannedInFilter = Long.parseLong(dataTableMetadata.getOrDefault("numEntriesScannedInFilter", "-1"));
        long numEntriesScannedPostFilter = Long.parseLong(dataTableMetadata.getOrDefault("numEntriesScannedPostFilter", "-1"));
        long numSegmentsProcessed = Long.parseLong(dataTableMetadata.getOrDefault("numSegmentsProcessed", "-1"));
        long numSegmentsMatched = Long.parseLong(dataTableMetadata.getOrDefault("numSegmentsMatched", "-1"));
        long numSegmentsConsuming = Long.parseLong(dataTableMetadata.getOrDefault("numConsumingSegmentsProcessed", "-1"));
        long minConsumingFreshnessMs = Long.parseLong(dataTableMetadata.getOrDefault("minConsumingFreshnessTimeMs", "-1"));
        if (numDocsScanned > 0L) {
            this.serverMetrics.addMeteredTableValue(tableNameWithType, (AbstractMetrics.Meter)ServerMeter.NUM_DOCS_SCANNED, numDocsScanned);
        }
        if (numEntriesScannedInFilter > 0L) {
            this.serverMetrics.addMeteredTableValue(tableNameWithType, (AbstractMetrics.Meter)ServerMeter.NUM_ENTRIES_SCANNED_IN_FILTER, numEntriesScannedInFilter);
        }
        if (numEntriesScannedPostFilter > 0L) {
            this.serverMetrics.addMeteredTableValue(tableNameWithType, (AbstractMetrics.Meter)ServerMeter.NUM_ENTRIES_SCANNED_POST_FILTER, numEntriesScannedPostFilter);
        }
        TimerContext timerContext = queryRequest.getTimerContext();
        int numSegmentsQueried = queryRequest.getSegmentsToQuery().size();
        long schedulerWaitMs = timerContext.getPhaseDurationMs(ServerQueryPhase.SCHEDULER_WAIT);
        if (this.queryLogRateLimiter.tryAcquire() || this.forceLog(schedulerWaitMs, numDocsScanned)) {
            int numDroppedLog;
            LOGGER.info("Processed requestId={},table={},segments(queried/processed/matched/consuming)={}/{}/{}/{},schedulerWaitMs={},reqDeserMs={},totalExecMs={},resSerMs={},totalTimeMs={},minConsumingFreshnessMs={},broker={},numDocsScanned={},scanInFilter={},scanPostFilter={},sched={}", new Object[]{requestId, tableNameWithType, numSegmentsQueried, numSegmentsProcessed, numSegmentsMatched, numSegmentsConsuming, schedulerWaitMs, timerContext.getPhaseDurationMs(ServerQueryPhase.REQUEST_DESERIALIZATION), timerContext.getPhaseDurationMs(ServerQueryPhase.QUERY_PROCESSING), timerContext.getPhaseDurationMs(ServerQueryPhase.RESPONSE_SERIALIZATION), timerContext.getPhaseDurationMs(ServerQueryPhase.TOTAL_QUERY_TIME), minConsumingFreshnessMs, queryRequest.getBrokerId(), numDocsScanned, numEntriesScannedInFilter, numEntriesScannedPostFilter, this.name()});
            if (this.numDroppedLogRateLimiter.tryAcquire() && (numDroppedLog = this.numDroppedLogCounter.get()) > 0) {
                LOGGER.info("{} logs were dropped. (log max rate per second: {})", (Object)numDroppedLog, (Object)this.queryLogRateLimiter.getRate());
                this.numDroppedLogCounter.set(0);
            }
        } else {
            this.numDroppedLogCounter.incrementAndGet();
        }
        if (minConsumingFreshnessMs > -1L) {
            this.serverMetrics.addTimedTableValue(tableNameWithType, (AbstractMetrics.Timer)ServerTimer.FRESHNESS_LAG_MS, System.currentTimeMillis() - minConsumingFreshnessMs, TimeUnit.MILLISECONDS);
        }
        this.serverMetrics.addMeteredTableValue(tableNameWithType, (AbstractMetrics.Meter)ServerMeter.NUM_SEGMENTS_QUERIED, (long)numSegmentsQueried);
        this.serverMetrics.addMeteredTableValue(tableNameWithType, (AbstractMetrics.Meter)ServerMeter.NUM_SEGMENTS_PROCESSED, numSegmentsProcessed);
        this.serverMetrics.addMeteredTableValue(tableNameWithType, (AbstractMetrics.Meter)ServerMeter.NUM_SEGMENTS_MATCHED, numSegmentsMatched);
        return responseData;
    }

    private boolean forceLog(long schedulerWaitMs, long numDocsScanned) {
        if (schedulerWaitMs > 100L) {
            return true;
        }
        return numDocsScanned > 1000000L;
    }

    @Nullable
    private byte[] serializeDataTable(@Nonnull ServerQueryRequest queryRequest, @Nonnull DataTable dataTable) {
        TimerContext timerContext = queryRequest.getTimerContext();
        TimerContext.Timer responseSerializationTimer = timerContext.startNewPhaseTimer(ServerQueryPhase.RESPONSE_SERIALIZATION);
        byte[] responseByte = null;
        try {
            responseByte = dataTable.toBytes();
        }
        catch (Exception e) {
            this.serverMetrics.addMeteredGlobalValue((AbstractMetrics.Meter)ServerMeter.RESPONSE_SERIALIZATION_EXCEPTIONS, 1L);
            LOGGER.error("Caught exception while serializing response for requestId: {}, brokerId: {}", new Object[]{queryRequest.getRequestId(), queryRequest.getBrokerId(), e});
        }
        responseSerializationTimer.stopAndRecord();
        timerContext.startNewPhaseTimer(ServerQueryPhase.TOTAL_QUERY_TIME, timerContext.getQueryArrivalTimeMs()).stopAndRecord();
        return responseByte;
    }

    protected ListenableFuture<byte[]> immediateErrorResponse(ServerQueryRequest queryRequest, ProcessingException error) {
        DataTableImplV2 result = new DataTableImplV2();
        result.addException(error);
        return Futures.immediateFuture((Object)this.serializeDataTable(queryRequest, result));
    }
}

