/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.core.data.table;

import com.google.common.base.Preconditions;
import it.unimi.dsi.fastutil.objects.ObjectOpenHashSet;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import org.apache.pinot.common.utils.DataSchema;
import org.apache.pinot.core.data.table.Key;
import org.apache.pinot.core.data.table.Record;
import org.apache.pinot.core.query.aggregation.function.AggregationFunction;
import org.apache.pinot.core.query.postaggregation.PostAggregationFunction;
import org.apache.pinot.core.query.request.context.ExpressionContext;
import org.apache.pinot.core.query.request.context.FunctionContext;
import org.apache.pinot.core.query.request.context.OrderByExpressionContext;
import org.apache.pinot.core.query.request.context.QueryContext;
import org.apache.pinot.spi.utils.ByteArray;

public class TableResizer {
    private final DataSchema _dataSchema;
    private final int _numGroupByExpressions;
    private final Map<ExpressionContext, Integer> _groupByExpressionIndexMap;
    private final AggregationFunction[] _aggregationFunctions;
    private final Map<FunctionContext, Integer> _aggregationFunctionIndexMap;
    private final int _numOrderByExpressions;
    private final OrderByValueExtractor[] _orderByValueExtractors;
    private final Comparator<IntermediateRecord> _intermediateRecordComparator;

    public TableResizer(DataSchema dataSchema, QueryContext queryContext) {
        this._dataSchema = dataSchema;
        List<ExpressionContext> groupByExpressions = queryContext.getGroupByExpressions();
        assert (groupByExpressions != null);
        this._numGroupByExpressions = groupByExpressions.size();
        this._groupByExpressionIndexMap = new HashMap<ExpressionContext, Integer>();
        for (int i = 0; i < this._numGroupByExpressions; ++i) {
            this._groupByExpressionIndexMap.put(groupByExpressions.get(i), i);
        }
        this._aggregationFunctions = queryContext.getAggregationFunctions();
        assert (this._aggregationFunctions != null);
        this._aggregationFunctionIndexMap = queryContext.getAggregationFunctionIndexMap();
        assert (this._aggregationFunctionIndexMap != null);
        List<OrderByExpressionContext> orderByExpressions = queryContext.getOrderByExpressions();
        assert (orderByExpressions != null);
        this._numOrderByExpressions = orderByExpressions.size();
        this._orderByValueExtractors = new OrderByValueExtractor[this._numOrderByExpressions];
        Comparator[] comparators = new Comparator[this._numOrderByExpressions];
        for (int i = 0; i < this._numOrderByExpressions; ++i) {
            OrderByExpressionContext orderByExpression = orderByExpressions.get(i);
            this._orderByValueExtractors[i] = this.getOrderByValueExtractor(orderByExpression.getExpression());
            comparators[i] = orderByExpression.isAsc() ? Comparator.naturalOrder() : Comparator.reverseOrder();
        }
        this._intermediateRecordComparator = (o1, o2) -> {
            for (int i = 0; i < this._numOrderByExpressions; ++i) {
                int result = comparators[i].compare(o1._values[i], o2._values[i]);
                if (result == 0) continue;
                return result;
            }
            return 0;
        };
    }

    private OrderByValueExtractor getOrderByValueExtractor(ExpressionContext expression) {
        if (expression.getType() == ExpressionContext.Type.LITERAL) {
            return new LiteralExtractor(expression.getLiteral());
        }
        Integer groupByExpressionIndex = this._groupByExpressionIndexMap.get(expression);
        if (groupByExpressionIndex != null) {
            return new GroupByExpressionExtractor(groupByExpressionIndex);
        }
        FunctionContext function = expression.getFunction();
        Preconditions.checkState((function != null ? 1 : 0) != 0, (String)"Failed to find ORDER-BY expression: %s in the GROUP-BY clause", (Object)expression);
        if (function.getType() == FunctionContext.Type.AGGREGATION) {
            return new AggregationFunctionExtractor(this._aggregationFunctionIndexMap.get(function));
        }
        return new PostAggregationFunctionExtractor(function);
    }

    private IntermediateRecord getIntermediateRecord(Key key, Record record) {
        Comparable[] intermediateRecordValues = new Comparable[this._numOrderByExpressions];
        for (int i = 0; i < this._numOrderByExpressions; ++i) {
            intermediateRecordValues[i] = this._orderByValueExtractors[i].extract(record);
        }
        return new IntermediateRecord(key, intermediateRecordValues);
    }

    public void resizeRecordsMap(Map<Key, Record> recordsMap, int trimToSize) {
        int numRecordsToEvict = recordsMap.size() - trimToSize;
        if (numRecordsToEvict > 0) {
            if (numRecordsToEvict < trimToSize) {
                PriorityQueue<IntermediateRecord> priorityQueue = this.convertToIntermediateRecordsPQ(recordsMap, numRecordsToEvict, this._intermediateRecordComparator);
                for (IntermediateRecord evictRecord : priorityQueue) {
                    recordsMap.remove(evictRecord._key);
                }
            } else {
                Comparator<IntermediateRecord> comparator = this._intermediateRecordComparator.reversed();
                PriorityQueue<IntermediateRecord> priorityQueue = this.convertToIntermediateRecordsPQ(recordsMap, trimToSize, comparator);
                ObjectOpenHashSet keysToRetain = new ObjectOpenHashSet(priorityQueue.size());
                for (IntermediateRecord retainRecord : priorityQueue) {
                    keysToRetain.add((Object)retainRecord._key);
                }
                recordsMap.keySet().retainAll((Collection<?>)keysToRetain);
            }
        }
    }

    private PriorityQueue<IntermediateRecord> convertToIntermediateRecordsPQ(Map<Key, Record> recordsMap, int size, Comparator<IntermediateRecord> comparator) {
        PriorityQueue<IntermediateRecord> priorityQueue = new PriorityQueue<IntermediateRecord>(size, comparator);
        for (Map.Entry<Key, Record> entry : recordsMap.entrySet()) {
            IntermediateRecord intermediateRecord = this.getIntermediateRecord(entry.getKey(), entry.getValue());
            if (priorityQueue.size() < size) {
                priorityQueue.offer(intermediateRecord);
                continue;
            }
            IntermediateRecord peek = priorityQueue.peek();
            if (comparator.compare(peek, intermediateRecord) >= 0) continue;
            priorityQueue.poll();
            priorityQueue.offer(intermediateRecord);
        }
        return priorityQueue;
    }

    private List<Record> sortRecordsMap(Map<Key, Record> recordsMap) {
        int numRecords = recordsMap.size();
        ArrayList<Record> sortedRecords = new ArrayList<Record>(numRecords);
        ArrayList<IntermediateRecord> intermediateRecords = new ArrayList<IntermediateRecord>(numRecords);
        for (Map.Entry<Key, Record> entry : recordsMap.entrySet()) {
            intermediateRecords.add(this.getIntermediateRecord(entry.getKey(), entry.getValue()));
        }
        intermediateRecords.sort(this._intermediateRecordComparator);
        for (IntermediateRecord intermediateRecord : intermediateRecords) {
            sortedRecords.add(recordsMap.get(intermediateRecord._key));
        }
        return sortedRecords;
    }

    public List<Record> resizeAndSortRecordsMap(Map<Key, Record> recordsMap, int trimToSize) {
        int numRecords = recordsMap.size();
        if (numRecords == 0) {
            return Collections.emptyList();
        }
        int numRecordsToRetain = Math.min(numRecords, trimToSize);
        int numRecordsToEvict = numRecords - numRecordsToRetain;
        if (numRecordsToEvict < numRecordsToRetain) {
            if (numRecordsToEvict > 0) {
                PriorityQueue<IntermediateRecord> priorityQueue = this.convertToIntermediateRecordsPQ(recordsMap, numRecordsToEvict, this._intermediateRecordComparator);
                for (IntermediateRecord evictRecord : priorityQueue) {
                    recordsMap.remove(evictRecord._key);
                }
            }
            return this.sortRecordsMap(recordsMap);
        }
        PriorityQueue<IntermediateRecord> priorityQueue = this.convertToIntermediateRecordsPQ(recordsMap, numRecordsToRetain, this._intermediateRecordComparator.reversed());
        Record[] sortedArray = new Record[numRecordsToRetain];
        ObjectOpenHashSet keysToRetain = new ObjectOpenHashSet(numRecordsToRetain);
        while (!priorityQueue.isEmpty()) {
            IntermediateRecord intermediateRecord = priorityQueue.poll();
            keysToRetain.add((Object)intermediateRecord._key);
            Record record = recordsMap.get(intermediateRecord._key);
            sortedArray[--numRecordsToRetain] = record;
        }
        recordsMap.keySet().retainAll((Collection<?>)keysToRetain);
        return Arrays.asList(sortedArray);
    }

    private class PostAggregationFunctionExtractor
    implements OrderByValueExtractor {
        final Object[] _arguments;
        final OrderByValueExtractor[] _argumentExtractors;
        final PostAggregationFunction _postAggregationFunction;

        PostAggregationFunctionExtractor(FunctionContext function) {
            assert (function.getType() == FunctionContext.Type.TRANSFORM);
            List<ExpressionContext> arguments = function.getArguments();
            int numArguments = arguments.size();
            this._arguments = new Object[numArguments];
            this._argumentExtractors = new OrderByValueExtractor[numArguments];
            DataSchema.ColumnDataType[] argumentTypes = new DataSchema.ColumnDataType[numArguments];
            for (int i = 0; i < numArguments; ++i) {
                OrderByValueExtractor argumentExtractor;
                this._argumentExtractors[i] = argumentExtractor = TableResizer.this.getOrderByValueExtractor(arguments.get(i));
                argumentTypes[i] = argumentExtractor.getValueType();
            }
            this._postAggregationFunction = new PostAggregationFunction(function.getFunctionName(), argumentTypes);
        }

        @Override
        public DataSchema.ColumnDataType getValueType() {
            return this._postAggregationFunction.getResultType();
        }

        @Override
        public Comparable extract(Record record) {
            int numArguments = this._arguments.length;
            for (int i = 0; i < numArguments; ++i) {
                this._arguments[i] = this._argumentExtractors[i].extract(record);
            }
            Object result = this._postAggregationFunction.invoke(this._arguments);
            if (this._postAggregationFunction.getResultType() == DataSchema.ColumnDataType.BYTES) {
                return new ByteArray((byte[])result);
            }
            return (Comparable)result;
        }
    }

    private class AggregationFunctionExtractor
    implements OrderByValueExtractor {
        final int _index;
        final AggregationFunction _aggregationFunction;

        AggregationFunctionExtractor(int aggregationFunctionIndex) {
            this._index = aggregationFunctionIndex + TableResizer.this._numGroupByExpressions;
            this._aggregationFunction = TableResizer.this._aggregationFunctions[aggregationFunctionIndex];
        }

        @Override
        public DataSchema.ColumnDataType getValueType() {
            return this._aggregationFunction.getFinalResultColumnType();
        }

        @Override
        public Comparable extract(Record record) {
            return this._aggregationFunction.extractFinalResult(record.getValues()[this._index]);
        }
    }

    private class GroupByExpressionExtractor
    implements OrderByValueExtractor {
        final int _index;

        GroupByExpressionExtractor(int groupByExpressionIndex) {
            this._index = groupByExpressionIndex;
        }

        @Override
        public DataSchema.ColumnDataType getValueType() {
            return TableResizer.this._dataSchema.getColumnDataType(this._index);
        }

        @Override
        public Comparable extract(Record record) {
            return (Comparable)record.getValues()[this._index];
        }
    }

    private static class LiteralExtractor
    implements OrderByValueExtractor {
        final String _literal;

        LiteralExtractor(String literal) {
            this._literal = literal;
        }

        @Override
        public DataSchema.ColumnDataType getValueType() {
            return DataSchema.ColumnDataType.STRING;
        }

        public String extract(Record record) {
            return this._literal;
        }
    }

    private static interface OrderByValueExtractor {
        public DataSchema.ColumnDataType getValueType();

        public Comparable extract(Record var1);
    }

    private static class IntermediateRecord {
        final Key _key;
        final Comparable[] _values;

        IntermediateRecord(Key key, Comparable[] values) {
            this._key = key;
            this._values = values;
        }
    }
}

