/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.exchange.hdfs;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.errorprone.annotations.ThreadSafe;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import com.google.inject.Inject;
import io.airlift.slice.InputStreamSliceInput;
import io.airlift.slice.SizeOf;
import io.airlift.slice.Slice;
import io.trino.annotation.NotThreadSafe;
import io.trino.hadoop.ConfigurationInstantiator;
import io.trino.plugin.exchange.filesystem.ExchangeSourceFile;
import io.trino.plugin.exchange.filesystem.ExchangeStorageReader;
import io.trino.plugin.exchange.filesystem.ExchangeStorageWriter;
import io.trino.plugin.exchange.filesystem.FileStatus;
import io.trino.plugin.exchange.filesystem.FileSystemExchangeStorage;
import io.trino.plugin.exchange.hdfs.ExchangeHdfsConfig;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.net.URI;
import java.util.ArrayDeque;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Queue;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;

public class HadoopFileSystemExchangeStorage
implements FileSystemExchangeStorage {
    private final int blockSize;
    private final FileSystem fileSystem;

    @Inject
    public HadoopFileSystemExchangeStorage(ExchangeHdfsConfig config) throws IOException {
        Configuration hdfsConfig = ConfigurationInstantiator.newEmptyConfiguration();
        for (File resourcePath : config.getResourceConfigFiles()) {
            Preconditions.checkArgument((boolean)resourcePath.exists(), (String)"File does not exist: %s", (Object)resourcePath);
            hdfsConfig.addResource(new Path(resourcePath.getPath()));
        }
        this.fileSystem = FileSystem.get((Configuration)hdfsConfig);
        this.blockSize = Math.toIntExact(config.getHdfsStorageBlockSize().toBytes());
    }

    public void createDirectories(URI dir) throws IOException {
        this.fileSystem.mkdirs(new Path(dir));
    }

    public ExchangeStorageReader createExchangeStorageReader(List<ExchangeSourceFile> sourceFiles, int maxPageStorageSize) {
        return new HadoopExchangeStorageReader(this.fileSystem, sourceFiles, this.blockSize);
    }

    public ExchangeStorageWriter createExchangeStorageWriter(URI file) {
        return new HadoopExchangeStorageWriter(this.fileSystem, file);
    }

    public ListenableFuture<Void> createEmptyFile(URI file) {
        try {
            this.fileSystem.createNewFile(new Path(file));
        }
        catch (IOException e) {
            return Futures.immediateFailedFuture((Throwable)e);
        }
        return Futures.immediateVoidFuture();
    }

    public ListenableFuture<Void> deleteRecursively(List<URI> directories) {
        for (URI dir : directories) {
            try {
                this.fileSystem.delete(new Path(dir), true);
            }
            catch (IOException | RuntimeException e) {
                return Futures.immediateFailedFuture((Throwable)e);
            }
        }
        return Futures.immediateVoidFuture();
    }

    public ListenableFuture<List<FileStatus>> listFilesRecursively(URI dir) {
        ImmutableList.Builder builder = ImmutableList.builder();
        try {
            RemoteIterator fileStatusListIterator = this.fileSystem.listFiles(new Path(dir), true);
            while (fileStatusListIterator.hasNext()) {
                LocatedFileStatus fileStatus = (LocatedFileStatus)fileStatusListIterator.next();
                builder.add((Object)new FileStatus(fileStatus.getPath().toString(), fileStatus.getLen()));
            }
        }
        catch (IOException e) {
            return Futures.immediateFailedFuture((Throwable)e);
        }
        return Futures.immediateFuture((Object)builder.build());
    }

    public int getWriteBufferSize() {
        return this.blockSize;
    }

    public void close() {
    }

    @ThreadSafe
    private static class HadoopExchangeStorageReader
    implements ExchangeStorageReader {
        private static final int INSTANCE_SIZE = SizeOf.instanceSize(HadoopExchangeStorageReader.class);
        private final FileSystem fileSystem;
        @GuardedBy(value="this")
        private final Queue<ExchangeSourceFile> sourceFiles;
        private final int blockSize;
        @GuardedBy(value="this")
        private InputStreamSliceInput sliceInput;
        @GuardedBy(value="this")
        private boolean closed;

        public HadoopExchangeStorageReader(FileSystem fileSystem, List<ExchangeSourceFile> sourceFiles, int blockSize) {
            this.fileSystem = Objects.requireNonNull(fileSystem, "fileSystem is null");
            this.sourceFiles = new ArrayDeque<ExchangeSourceFile>((Collection)Objects.requireNonNull(sourceFiles, "sourceFiles is null"));
            this.blockSize = blockSize;
        }

        public synchronized Slice read() throws IOException {
            if (this.closed) {
                return null;
            }
            if (this.sliceInput != null && this.sliceInput.isReadable()) {
                return this.sliceInput.readSlice(this.sliceInput.readInt());
            }
            ExchangeSourceFile sourceFile = this.sourceFiles.poll();
            if (sourceFile == null) {
                this.close();
                return null;
            }
            this.sliceInput = this.getSliceInput(sourceFile);
            return this.sliceInput.readSlice(this.sliceInput.readInt());
        }

        public ListenableFuture<Void> isBlocked() {
            return Futures.immediateVoidFuture();
        }

        public synchronized long getRetainedSize() {
            return (long)INSTANCE_SIZE + (this.sliceInput == null ? 0L : this.sliceInput.getRetainedSize());
        }

        public synchronized boolean isFinished() {
            return this.closed;
        }

        public synchronized void close() {
            if (this.closed) {
                return;
            }
            this.closed = true;
            if (this.sliceInput != null) {
                this.sliceInput.close();
                this.sliceInput = null;
            }
        }

        private InputStreamSliceInput getSliceInput(ExchangeSourceFile sourceFile) throws IOException {
            Path fileURL = new Path(sourceFile.getFileUri());
            return new InputStreamSliceInput((InputStream)this.fileSystem.open(fileURL), this.blockSize);
        }
    }

    @NotThreadSafe
    private static class HadoopExchangeStorageWriter
    implements ExchangeStorageWriter {
        private static final int INSTANCE_SIZE = SizeOf.instanceSize(HadoopExchangeStorageReader.class);
        private final OutputStream outputStream;

        public HadoopExchangeStorageWriter(FileSystem fileSystem, URI file) {
            try {
                this.outputStream = fileSystem.create(new Path(file), true);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }

        public ListenableFuture<Void> write(Slice slice) {
            try {
                this.outputStream.write(slice.getBytes());
            }
            catch (IOException | RuntimeException e) {
                return Futures.immediateFailedFuture((Throwable)e);
            }
            return Futures.immediateVoidFuture();
        }

        public ListenableFuture<Void> finish() {
            try {
                this.outputStream.close();
            }
            catch (IOException | RuntimeException e) {
                return Futures.immediateFailedFuture((Throwable)e);
            }
            return Futures.immediateVoidFuture();
        }

        public ListenableFuture<Void> abort() {
            try {
                this.outputStream.close();
            }
            catch (IOException | RuntimeException e) {
                return Futures.immediateFailedFuture((Throwable)e);
            }
            return Futures.immediateVoidFuture();
        }

        public long getRetainedSize() {
            return INSTANCE_SIZE;
        }
    }
}

