/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.elasticsearch.client;

import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.auth.DefaultAWSCredentialsProviderChain;
import com.amazonaws.auth.STSAssumeRoleSessionCredentialsProvider;
import com.amazonaws.services.securitytoken.AWSSecurityTokenService;
import com.amazonaws.services.securitytoken.AWSSecurityTokenServiceClientBuilder;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.NullNode;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.google.inject.Inject;
import io.airlift.concurrent.Threads;
import io.airlift.json.JsonCodec;
import io.airlift.json.ObjectMapperProvider;
import io.airlift.log.Logger;
import io.airlift.stats.TimeStat;
import io.airlift.units.Duration;
import io.trino.plugin.base.ssl.SslUtils;
import io.trino.plugin.elasticsearch.AwsSecurityConfig;
import io.trino.plugin.elasticsearch.ElasticsearchConfig;
import io.trino.plugin.elasticsearch.ElasticsearchErrorCode;
import io.trino.plugin.elasticsearch.PasswordConfig;
import io.trino.plugin.elasticsearch.client.AwsRequestSigner;
import io.trino.plugin.elasticsearch.client.BackpressureRestHighLevelClient;
import io.trino.plugin.elasticsearch.client.CountResponse;
import io.trino.plugin.elasticsearch.client.ElasticsearchNode;
import io.trino.plugin.elasticsearch.client.IndexMetadata;
import io.trino.plugin.elasticsearch.client.NodesResponse;
import io.trino.plugin.elasticsearch.client.SearchShardsResponse;
import io.trino.plugin.elasticsearch.client.Shard;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.TrinoException;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.nio.client.HttpAsyncClientBuilder;
import org.apache.http.impl.nio.reactor.IOReactorConfig;
import org.apache.http.message.BasicHeader;
import org.apache.http.util.EntityUtils;
import org.elasticsearch.ElasticsearchStatusException;
import org.elasticsearch.action.search.ClearScrollRequest;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.search.SearchScrollRequest;
import org.elasticsearch.action.search.SearchType;
import org.elasticsearch.client.Response;
import org.elasticsearch.client.ResponseException;
import org.elasticsearch.client.RestClient;
import org.elasticsearch.client.RestClientBuilder;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.weakref.jmx.Managed;
import org.weakref.jmx.Nested;

public class ElasticsearchClient {
    private static final Logger LOG = Logger.get(ElasticsearchClient.class);
    private static final JsonCodec<SearchShardsResponse> SEARCH_SHARDS_RESPONSE_CODEC = JsonCodec.jsonCodec(SearchShardsResponse.class);
    private static final JsonCodec<NodesResponse> NODES_RESPONSE_CODEC = JsonCodec.jsonCodec(NodesResponse.class);
    private static final JsonCodec<CountResponse> COUNT_RESPONSE_CODEC = JsonCodec.jsonCodec(CountResponse.class);
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapperProvider().get();
    private static final Pattern ADDRESS_PATTERN = Pattern.compile("((?<cname>[^/]+)/)?(?<ip>.+):(?<port>\\d+)");
    private static final Set<String> NODE_ROLES = ImmutableSet.of((Object)"data", (Object)"data_content", (Object)"data_hot", (Object)"data_warm", (Object)"data_cold", (Object)"data_frozen", (Object[])new String[0]);
    private final BackpressureRestHighLevelClient client;
    private final int scrollSize;
    private final Duration scrollTimeout;
    private final AtomicReference<Set<ElasticsearchNode>> nodes = new AtomicReference<ImmutableSet>(ImmutableSet.of());
    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(Threads.daemonThreadsNamed((String)"NodeRefresher"));
    private final AtomicBoolean started = new AtomicBoolean();
    private final Duration refreshInterval;
    private final boolean tlsEnabled;
    private final boolean ignorePublishAddress;
    private final TimeStat searchStats = new TimeStat(TimeUnit.MILLISECONDS);
    private final TimeStat nextPageStats = new TimeStat(TimeUnit.MILLISECONDS);
    private final TimeStat countStats = new TimeStat(TimeUnit.MILLISECONDS);
    private final TimeStat backpressureStats = new TimeStat(TimeUnit.MILLISECONDS);

    @Inject
    public ElasticsearchClient(ElasticsearchConfig config, Optional<AwsSecurityConfig> awsSecurityConfig, Optional<PasswordConfig> passwordConfig) {
        this.client = ElasticsearchClient.createClient(config, awsSecurityConfig, passwordConfig, this.backpressureStats);
        this.ignorePublishAddress = config.isIgnorePublishAddress();
        this.scrollSize = config.getScrollSize();
        this.scrollTimeout = config.getScrollTimeout();
        this.refreshInterval = config.getNodeRefreshInterval();
        this.tlsEnabled = config.isTlsEnabled();
    }

    @PostConstruct
    public void initialize() {
        if (!this.started.getAndSet(true)) {
            this.refreshNodes();
            this.executor.scheduleWithFixedDelay(this::refreshNodes, this.refreshInterval.toMillis(), this.refreshInterval.toMillis(), TimeUnit.MILLISECONDS);
        }
    }

    @PreDestroy
    public void close() throws IOException {
        this.executor.shutdownNow();
        this.client.close();
    }

    private void refreshNodes() {
        try {
            Set<ElasticsearchNode> nodes = this.fetchNodes();
            HttpHost[] hosts = (HttpHost[])nodes.stream().map(ElasticsearchNode::getAddress).filter(Optional::isPresent).map(Optional::get).map(address -> HttpHost.create((String)String.format("%s://%s", this.tlsEnabled ? "https" : "http", address))).toArray(HttpHost[]::new);
            if (hosts.length > 0 && !this.ignorePublishAddress) {
                this.client.getLowLevelClient().setHosts(hosts);
            }
            this.nodes.set(nodes);
        }
        catch (Throwable e) {
            LOG.error(e, "Error refreshing nodes");
        }
    }

    private static BackpressureRestHighLevelClient createClient(ElasticsearchConfig config, Optional<AwsSecurityConfig> awsSecurityConfig, Optional<PasswordConfig> passwordConfig, TimeStat backpressureStats) {
        RestClientBuilder builder = RestClient.builder((HttpHost[])((HttpHost[])config.getHosts().stream().map(httpHost -> new HttpHost(httpHost, config.getPort(), config.isTlsEnabled() ? "https" : "http")).toArray(HttpHost[]::new))).setMaxRetryTimeoutMillis(StrictMath.toIntExact(config.getMaxRetryTime().toMillis()));
        builder.setHttpClientConfigCallback(ignored -> {
            RequestConfig requestConfig = RequestConfig.custom().setConnectTimeout(StrictMath.toIntExact(config.getConnectTimeout().toMillis())).setSocketTimeout(StrictMath.toIntExact(config.getRequestTimeout().toMillis())).build();
            IOReactorConfig reactorConfig = IOReactorConfig.custom().setIoThreadCount(config.getHttpThreadCount()).build();
            HttpAsyncClientBuilder clientBuilder = HttpAsyncClientBuilder.create().setDefaultRequestConfig(requestConfig).setDefaultIOReactorConfig(reactorConfig).setMaxConnPerRoute(config.getMaxHttpConnections()).setMaxConnTotal(config.getMaxHttpConnections());
            if (config.isTlsEnabled()) {
                ElasticsearchClient.buildSslContext(config.getKeystorePath(), config.getKeystorePassword(), config.getTrustStorePath(), config.getTruststorePassword()).ifPresent(arg_0 -> ((HttpAsyncClientBuilder)clientBuilder).setSSLContext(arg_0));
                if (config.isVerifyHostnames()) {
                    clientBuilder.setSSLHostnameVerifier((HostnameVerifier)NoopHostnameVerifier.INSTANCE);
                }
            }
            passwordConfig.ifPresent(securityConfig -> {
                BasicCredentialsProvider credentials = new BasicCredentialsProvider();
                credentials.setCredentials(AuthScope.ANY, (Credentials)new UsernamePasswordCredentials(securityConfig.getUser(), securityConfig.getPassword()));
                clientBuilder.setDefaultCredentialsProvider((CredentialsProvider)credentials);
            });
            awsSecurityConfig.ifPresent(securityConfig -> clientBuilder.addInterceptorLast((HttpRequestInterceptor)new AwsRequestSigner(securityConfig.getRegion(), ElasticsearchClient.getAwsCredentialsProvider(securityConfig))));
            return clientBuilder;
        });
        return new BackpressureRestHighLevelClient(builder, config, backpressureStats);
    }

    private static AWSCredentialsProvider getAwsCredentialsProvider(AwsSecurityConfig config) {
        DefaultAWSCredentialsProviderChain credentialsProvider = DefaultAWSCredentialsProviderChain.getInstance();
        if (config.getAccessKey().isPresent() && config.getSecretKey().isPresent()) {
            credentialsProvider = new AWSStaticCredentialsProvider((AWSCredentials)new BasicAWSCredentials(config.getAccessKey().get(), config.getSecretKey().get()));
        }
        if (config.getIamRole().isPresent()) {
            STSAssumeRoleSessionCredentialsProvider.Builder credentialsProviderBuilder = new STSAssumeRoleSessionCredentialsProvider.Builder(config.getIamRole().get(), "trino-session").withStsClient((AWSSecurityTokenService)((AWSSecurityTokenServiceClientBuilder)((AWSSecurityTokenServiceClientBuilder)AWSSecurityTokenServiceClientBuilder.standard().withRegion(config.getRegion())).withCredentials((AWSCredentialsProvider)credentialsProvider)).build());
            config.getExternalId().ifPresent(arg_0 -> ((STSAssumeRoleSessionCredentialsProvider.Builder)credentialsProviderBuilder).withExternalId(arg_0));
            credentialsProvider = credentialsProviderBuilder.build();
        }
        return credentialsProvider;
    }

    private static Optional<SSLContext> buildSslContext(Optional<File> keyStorePath, Optional<String> keyStorePassword, Optional<File> trustStorePath, Optional<String> trustStorePassword) {
        if (keyStorePath.isEmpty() && trustStorePath.isEmpty()) {
            return Optional.empty();
        }
        try {
            return Optional.of(SslUtils.createSSLContext(keyStorePath, keyStorePassword, trustStorePath, trustStorePassword));
        }
        catch (IOException | GeneralSecurityException e) {
            throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_SSL_INITIALIZATION_FAILURE, (Throwable)e);
        }
    }

    private Set<ElasticsearchNode> fetchNodes() {
        NodesResponse nodesResponse = this.doRequest("/_nodes/http", arg_0 -> NODES_RESPONSE_CODEC.fromJson(arg_0));
        ImmutableSet.Builder result = ImmutableSet.builder();
        for (Map.Entry<String, NodesResponse.Node> entry : nodesResponse.getNodes().entrySet()) {
            String nodeId = entry.getKey();
            NodesResponse.Node node = entry.getValue();
            if (Sets.intersection(node.getRoles(), NODE_ROLES).isEmpty()) continue;
            Optional<String> address = node.getAddress().flatMap(ElasticsearchClient::extractAddress);
            result.add((Object)new ElasticsearchNode(nodeId, address));
        }
        return result.build();
    }

    public Set<ElasticsearchNode> getNodes() {
        return this.nodes.get();
    }

    public List<Shard> getSearchShards(String index) {
        Map nodeById = (Map)this.getNodes().stream().collect(ImmutableMap.toImmutableMap(ElasticsearchNode::getId, Function.identity()));
        SearchShardsResponse shardsResponse = this.doRequest(String.format("/%s/_search_shards", index), arg_0 -> SEARCH_SHARDS_RESPONSE_CODEC.fromJson(arg_0));
        ImmutableList.Builder shards = ImmutableList.builder();
        ImmutableList nodes = ImmutableList.copyOf(nodeById.values());
        for (List<SearchShardsResponse.Shard> shardGroup : shardsResponse.getShardGroups()) {
            ElasticsearchNode node;
            SearchShardsResponse.Shard chosen;
            Optional<SearchShardsResponse.Shard> candidate = shardGroup.stream().filter(shard -> shard.getNode() != null && nodeById.containsKey(shard.getNode())).min(this::shardPreference);
            if (candidate.isEmpty()) {
                chosen = (SearchShardsResponse.Shard)shardGroup.stream().min(this::shardPreference).get();
                node = (ElasticsearchNode)nodes.get(chosen.getShard() % nodes.size());
            } else {
                chosen = candidate.get();
                node = (ElasticsearchNode)nodeById.get(chosen.getNode());
            }
            shards.add((Object)new Shard(chosen.getIndex(), chosen.getShard(), node.getAddress()));
        }
        return shards.build();
    }

    private int shardPreference(SearchShardsResponse.Shard left, SearchShardsResponse.Shard right) {
        if (left.isPrimary() == right.isPrimary()) {
            return 0;
        }
        return left.isPrimary() ? 1 : -1;
    }

    public boolean indexExists(String index) {
        String path = String.format("/%s/_mappings", index);
        try {
            Response response = this.client.getLowLevelClient().performRequest("GET", path, new Header[0]);
            return response.getStatusLine().getStatusCode() == 200;
        }
        catch (ResponseException e) {
            if (e.getResponse().getStatusLine().getStatusCode() == 404) {
                return false;
            }
            throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_CONNECTION_ERROR, (Throwable)e);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_CONNECTION_ERROR, (Throwable)e);
        }
    }

    public List<String> getIndexes() {
        return (List)this.doRequest("/_cat/indices?h=index,docs.count,docs.deleted&format=json&s=index:asc", body -> {
            try {
                ImmutableList.Builder result = ImmutableList.builder();
                JsonNode root = OBJECT_MAPPER.readTree(body);
                for (int i = 0; i < root.size(); ++i) {
                    String index = root.get(i).get("index").asText();
                    int docsCount = root.get(i).get("docs.count").asInt();
                    int deletedDocsCount = root.get(i).get("docs.deleted").asInt();
                    if (docsCount == 0 && deletedDocsCount == 0 && this.getIndexMetadata(index).getSchema().getFields().isEmpty()) continue;
                    result.add((Object)index);
                }
                return result.build();
            }
            catch (IOException e) {
                throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_INVALID_RESPONSE, (Throwable)e);
            }
        });
    }

    public Map<String, List<String>> getAliases() {
        return (Map)this.doRequest("/_aliases", body -> {
            try {
                ImmutableMap.Builder result = ImmutableMap.builder();
                JsonNode root = OBJECT_MAPPER.readTree(body);
                Iterator elements = root.fields();
                while (elements.hasNext()) {
                    Map.Entry element = (Map.Entry)elements.next();
                    JsonNode aliases = ((JsonNode)element.getValue()).get("aliases");
                    Iterator aliasNames = aliases.fieldNames();
                    if (!aliasNames.hasNext()) continue;
                    result.put((Object)((String)element.getKey()), (Object)ImmutableList.copyOf((Iterator)aliasNames));
                }
                return result.buildOrThrow();
            }
            catch (IOException e) {
                throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_INVALID_RESPONSE, (Throwable)e);
            }
        });
    }

    public IndexMetadata getIndexMetadata(String index) {
        String path = String.format("/%s/_mappings", index);
        return this.doRequest(path, body -> {
            try {
                JsonNode mappings = ((JsonNode)OBJECT_MAPPER.readTree(body).elements().next()).get("mappings");
                if (!mappings.elements().hasNext()) {
                    return new IndexMetadata(new IndexMetadata.ObjectType((List<IndexMetadata.Field>)ImmutableList.of()));
                }
                if (!mappings.has("properties") && !(mappings = (JsonNode)mappings.elements().next()).has("properties")) {
                    return new IndexMetadata(new IndexMetadata.ObjectType((List<IndexMetadata.Field>)ImmutableList.of()));
                }
                JsonNode metaNode = this.nullSafeNode(mappings, "_meta");
                JsonNode metaProperties = this.nullSafeNode(metaNode, "trino");
                if (metaProperties.isNull()) {
                    metaProperties = this.nullSafeNode(metaNode, "presto");
                }
                return new IndexMetadata(this.parseType(mappings.get("properties"), metaProperties));
            }
            catch (IOException e) {
                throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_INVALID_RESPONSE, (Throwable)e);
            }
        });
    }

    private IndexMetadata.ObjectType parseType(JsonNode properties, JsonNode metaProperties) {
        Iterator entries = properties.fields();
        ImmutableList.Builder result = ImmutableList.builder();
        block11: while (entries.hasNext()) {
            boolean asRawJson;
            JsonNode metaNode;
            Map.Entry field = (Map.Entry)entries.next();
            String name = (String)field.getKey();
            JsonNode value = (JsonNode)field.getValue();
            String type = "object";
            if (value.has("type")) {
                type = value.get("type").asText();
            }
            boolean isArray = !(metaNode = this.nullSafeNode(metaProperties, name)).isNull() && metaNode.has("isArray") && metaNode.get("isArray").asBoolean();
            boolean bl = asRawJson = !metaNode.isNull() && metaNode.has("asRawJson") && metaNode.get("asRawJson").asBoolean();
            if (isArray && asRawJson) {
                throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_INVALID_METADATA, String.format("A column, (%s) cannot be declared as a Trino array and also be rendered as json.", name));
            }
            switch (type) {
                case "date": {
                    Object formats = ImmutableList.of();
                    if (value.has("format")) {
                        formats = Arrays.asList(value.get("format").asText().split("\\|\\|"));
                    }
                    result.add((Object)new IndexMetadata.Field(asRawJson, isArray, name, new IndexMetadata.DateTimeType((List<String>)formats)));
                    continue block11;
                }
                case "scaled_float": {
                    result.add((Object)new IndexMetadata.Field(asRawJson, isArray, name, new IndexMetadata.ScaledFloatType(value.get("scaling_factor").asDouble())));
                    continue block11;
                }
                case "nested": 
                case "object": {
                    if (value.has("properties")) {
                        result.add((Object)new IndexMetadata.Field(asRawJson, isArray, name, this.parseType(value.get("properties"), metaNode)));
                        continue block11;
                    }
                    LOG.debug("Ignoring empty object field: %s", new Object[]{name});
                    continue block11;
                }
            }
            result.add((Object)new IndexMetadata.Field(asRawJson, isArray, name, new IndexMetadata.PrimitiveType(type)));
        }
        return new IndexMetadata.ObjectType((List<IndexMetadata.Field>)result.build());
    }

    private JsonNode nullSafeNode(JsonNode jsonNode, String name) {
        if (jsonNode == null || jsonNode.isNull() || jsonNode.get(name) == null) {
            return NullNode.getInstance();
        }
        return jsonNode.get(name);
    }

    public String executeQuery(String index, String query) {
        String body;
        Response response;
        String path = String.format("/%s/_search", index);
        try {
            response = this.client.getLowLevelClient().performRequest("GET", path, (Map<String, String>)ImmutableMap.of(), (HttpEntity)new ByteArrayEntity(query.getBytes(StandardCharsets.UTF_8)), new Header[]{new BasicHeader("Content-Type", "application/json"), new BasicHeader("Accept-Encoding", "application/json")});
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_CONNECTION_ERROR, (Throwable)e);
        }
        try {
            body = EntityUtils.toString((HttpEntity)response.getEntity());
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_INVALID_RESPONSE, (Throwable)e);
        }
        return body;
    }

    public SearchResponse beginSearch(String index, int shard, QueryBuilder query, Optional<List<String>> fields, List<String> documentFields, Optional<String> sort, OptionalLong limit) {
        SearchSourceBuilder sourceBuilder = SearchSourceBuilder.searchSource().query(query);
        if (limit.isPresent() && limit.getAsLong() < (long)this.scrollSize) {
            sourceBuilder.size(StrictMath.toIntExact(limit.getAsLong()));
        } else {
            sourceBuilder.size(this.scrollSize);
        }
        sort.ifPresent(arg_0 -> ((SearchSourceBuilder)sourceBuilder).sort(arg_0));
        fields.ifPresent(values -> {
            if (values.isEmpty()) {
                sourceBuilder.fetchSource(false);
            } else {
                sourceBuilder.fetchSource(values.toArray(new String[0]), null);
            }
        });
        documentFields.forEach(arg_0 -> ((SearchSourceBuilder)sourceBuilder).docValueField(arg_0));
        LOG.debug("Begin search: %s:%s, query: %s", new Object[]{index, shard, sourceBuilder});
        SearchRequest request = new SearchRequest(new String[]{index}).searchType(SearchType.QUERY_THEN_FETCH).preference("_shards:" + shard).scroll(new TimeValue(this.scrollTimeout.toMillis())).source(sourceBuilder);
        long start = System.nanoTime();
        try {
            SearchResponse searchResponse = this.client.search(request, new Header[0]);
            return searchResponse;
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_CONNECTION_ERROR, (Throwable)e);
        }
        catch (ElasticsearchStatusException e) {
            Throwable cause;
            Throwable[] suppressed = e.getSuppressed();
            if (suppressed.length > 0 && (cause = suppressed[0]) instanceof ResponseException) {
                throw ElasticsearchClient.propagate((ResponseException)cause);
            }
            throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_CONNECTION_ERROR, (Throwable)e);
        }
        finally {
            this.searchStats.add(Duration.nanosSince((long)start));
        }
    }

    public SearchResponse nextPage(String scrollId) {
        LOG.debug("Next page: %s", new Object[]{scrollId});
        SearchScrollRequest request = new SearchScrollRequest(scrollId).scroll(new TimeValue(this.scrollTimeout.toMillis()));
        long start = System.nanoTime();
        try {
            SearchResponse searchResponse = this.client.searchScroll(request, new Header[0]);
            return searchResponse;
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_CONNECTION_ERROR, (Throwable)e);
        }
        finally {
            this.nextPageStats.add(Duration.nanosSince((long)start));
        }
    }

    public long count(String index, int shard, QueryBuilder query) {
        SearchSourceBuilder sourceBuilder = SearchSourceBuilder.searchSource().query(query);
        LOG.debug("Count: %s:%s, query: %s", new Object[]{index, shard, sourceBuilder});
        long start = System.nanoTime();
        try {
            Response response;
            try {
                response = this.client.getLowLevelClient().performRequest("GET", String.format("/%s/_count?preference=_shards:%s", index, shard), (Map<String, String>)ImmutableMap.of(), (HttpEntity)new StringEntity(sourceBuilder.toString()), new Header[]{new BasicHeader("Content-Type", "application/json")});
            }
            catch (ResponseException e) {
                throw ElasticsearchClient.propagate(e);
            }
            catch (IOException e) {
                throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_CONNECTION_ERROR, (Throwable)e);
            }
            try {
                long e = ((CountResponse)COUNT_RESPONSE_CODEC.fromJson(EntityUtils.toByteArray((HttpEntity)response.getEntity()))).getCount();
                return e;
            }
            catch (IOException e) {
                throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_INVALID_RESPONSE, (Throwable)e);
            }
        }
        finally {
            this.countStats.add(Duration.nanosSince((long)start));
        }
    }

    public void clearScroll(String scrollId) {
        ClearScrollRequest request = new ClearScrollRequest();
        request.addScrollId(scrollId);
        try {
            this.client.clearScroll(request, new Header[0]);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_CONNECTION_ERROR, (Throwable)e);
        }
    }

    @Managed
    @Nested
    public TimeStat getSearchStats() {
        return this.searchStats;
    }

    @Managed
    @Nested
    public TimeStat getNextPageStats() {
        return this.nextPageStats;
    }

    @Managed
    @Nested
    public TimeStat getCountStats() {
        return this.countStats;
    }

    @Managed
    @Nested
    public TimeStat getBackpressureStats() {
        return this.backpressureStats;
    }

    private <T> T doRequest(String path, ResponseHandler<T> handler) {
        String body;
        Response response;
        Preconditions.checkArgument((boolean)path.startsWith("/"), (Object)"path must be an absolute path");
        try {
            response = this.client.getLowLevelClient().performRequest("GET", path, new Header[0]);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_CONNECTION_ERROR, (Throwable)e);
        }
        try {
            body = EntityUtils.toString((HttpEntity)response.getEntity());
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_INVALID_RESPONSE, (Throwable)e);
        }
        return handler.process(body);
    }

    private static TrinoException propagate(ResponseException exception) {
        HttpEntity entity = exception.getResponse().getEntity();
        if (entity != null && entity.getContentType() != null) {
            try {
                JsonNode reason = OBJECT_MAPPER.readTree(entity.getContent()).path("error").path("root_cause").path(0).path("reason");
                if (!reason.isMissingNode()) {
                    throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_QUERY_FAILURE, reason.asText(), (Throwable)exception);
                }
            }
            catch (IOException e) {
                TrinoException result = new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_QUERY_FAILURE, (Throwable)exception);
                result.addSuppressed((Throwable)e);
                throw result;
            }
        }
        throw new TrinoException((ErrorCodeSupplier)ElasticsearchErrorCode.ELASTICSEARCH_QUERY_FAILURE, (Throwable)exception);
    }

    @VisibleForTesting
    static Optional<String> extractAddress(String address) {
        Matcher matcher = ADDRESS_PATTERN.matcher(address);
        if (!matcher.matches()) {
            return Optional.empty();
        }
        String cname = matcher.group("cname");
        String ip = matcher.group("ip");
        String port = matcher.group("port");
        if (cname != null) {
            return Optional.of(cname + ":" + port);
        }
        return Optional.of(ip + ":" + port);
    }

    private static interface ResponseHandler<T> {
        public T process(String var1);
    }
}

