/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.jdbc;

import com.google.common.base.Preconditions;
import io.trino.plugin.jdbc.ConnectionFactory;
import io.trino.plugin.jdbc.ForwardingConnection;
import io.trino.plugin.jdbc.ReusableConnectionFactory;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.security.ConnectorIdentity;
import io.trino.testing.TestingConnectorSession;
import java.sql.Connection;
import java.time.Duration;
import java.util.Objects;
import org.assertj.core.api.Assertions;
import org.testng.annotations.Test;

public class TestReusableConnectionFactory {
    private static final ConnectorSession ALICE = TestingConnectorSession.builder().setIdentity(ConnectorIdentity.ofUser((String)"alice")).build();
    private static final ConnectorSession BOB = TestingConnectorSession.builder().setIdentity(ConnectorIdentity.ofUser((String)"bob")).build();
    private static final Duration FOREVER = Duration.ofSeconds(5L);
    private static final int HUGE_SIZE = 1000;

    @Test
    public void testConnectionIsReusedSingleUser() throws Exception {
        try (MockConnectionFactory mockConnectionFactory = new MockConnectionFactory();
             ReusableConnectionFactory connectionFactory = new ReusableConnectionFactory((ConnectionFactory)mockConnectionFactory, FOREVER, 1000L);){
            connectionFactory.openConnection(ALICE).close();
            connectionFactory.openConnection(ALICE).close();
            connectionFactory.openConnection(ALICE).close();
            Assertions.assertThat((int)mockConnectionFactory.openedConnections).isEqualTo(1);
        }
    }

    @Test
    public void testSingleUserCanCreateMultipleConnections() throws Exception {
        try (MockConnectionFactory mockConnectionFactory = new MockConnectionFactory();
             ReusableConnectionFactory connectionFactory = new ReusableConnectionFactory((ConnectionFactory)mockConnectionFactory, FOREVER, 1000L);){
            Connection connection1 = connectionFactory.openConnection(ALICE);
            Assertions.assertThat((int)mockConnectionFactory.openedConnections).isEqualTo(1);
            Connection connection2 = connectionFactory.openConnection(ALICE);
            Assertions.assertThat((int)mockConnectionFactory.openedConnections).isEqualTo(2);
            connection1.close();
            Assertions.assertThat((int)mockConnectionFactory.closedConnections).isEqualTo(0);
            connection2.close();
            Assertions.assertThat((int)mockConnectionFactory.closedConnections).isEqualTo(1);
        }
    }

    @Test
    public void testConnectionIsReusedTwoUsers() throws Exception {
        try (MockConnectionFactory mockConnectionFactory = new MockConnectionFactory();
             ReusableConnectionFactory connectionFactory = new ReusableConnectionFactory((ConnectionFactory)mockConnectionFactory, FOREVER, 1000L);){
            connectionFactory.openConnection(ALICE).close();
            connectionFactory.openConnection(BOB).close();
            connectionFactory.openConnection(BOB).close();
            connectionFactory.openConnection(ALICE).close();
            Assertions.assertThat((int)mockConnectionFactory.openedConnections).isEqualTo(2);
        }
    }

    @Test
    public void testConnectionIsNotShared() throws Exception {
        try (MockConnectionFactory mockConnectionFactory = new MockConnectionFactory();
             ReusableConnectionFactory connectionFactory = new ReusableConnectionFactory((ConnectionFactory)mockConnectionFactory, FOREVER, 1000L);
             Connection ignored = connectionFactory.openConnection(ALICE);){
            Assertions.assertThat((int)mockConnectionFactory.openedConnections).isEqualTo(1);
            connectionFactory.openConnection(ALICE).close();
            Assertions.assertThat((int)mockConnectionFactory.openedConnections).isEqualTo(2);
        }
    }

    @Test
    public void testConnectionIsEvicted() throws Exception {
        try (MockConnectionFactory mockConnectionFactory = new MockConnectionFactory();
             ReusableConnectionFactory connectionFactory = new ReusableConnectionFactory((ConnectionFactory)mockConnectionFactory, Duration.ofMillis(50L), 1000L);){
            connectionFactory.openConnection(ALICE).close();
            Thread.sleep(100L);
            connectionFactory.openConnection(ALICE).close();
            Assertions.assertThat((int)mockConnectionFactory.openedConnections).isEqualTo(2);
        }
    }

    @Test
    public void testNoNewConnectionIsReusedWhenCacheIsFull() throws Exception {
        try (MockConnectionFactory mockConnectionFactory = new MockConnectionFactory();
             ReusableConnectionFactory connectionFactory = new ReusableConnectionFactory((ConnectionFactory)mockConnectionFactory, FOREVER, 1L);){
            connectionFactory.openConnection(ALICE).close();
            connectionFactory.openConnection(BOB).close();
            connectionFactory.openConnection(ALICE).close();
            connectionFactory.openConnection(BOB).close();
            connectionFactory.openConnection(ALICE).close();
            Assertions.assertThat((int)mockConnectionFactory.openedConnections).isEqualTo(5);
        }
    }

    @Test
    public void testConnectionIsNotReusedWhenSetToReadOnly() throws Exception {
        try (MockConnectionFactory mockConnectionFactory = new MockConnectionFactory();
             ReusableConnectionFactory connectionFactory = new ReusableConnectionFactory((ConnectionFactory)mockConnectionFactory, FOREVER, 1000L);){
            Connection connection = connectionFactory.openConnection(ALICE);
            connection.setReadOnly(true);
            connection.close();
            connectionFactory.openConnection(ALICE).close();
            Assertions.assertThat((int)mockConnectionFactory.openedConnections).isEqualTo(2);
        }
    }

    @Test
    public void testConnectionIsNotReusedWhenDelegateIsClosed() throws Exception {
        MockConnectionFactory mockConnectionFactory = new MockConnectionFactory();
        ReusableConnectionFactory connectionFactory = new ReusableConnectionFactory((ConnectionFactory)mockConnectionFactory, FOREVER, 1000L);
        Connection connection = connectionFactory.openConnection(ALICE);
        Connection delegate = ((ReusableConnectionFactory.CachedConnection)connection).delegate();
        delegate.close();
        connection.close();
        Connection secondConnection = connectionFactory.openConnection(ALICE);
        Connection secondDelegate = ((ReusableConnectionFactory.CachedConnection)secondConnection).delegate();
        Assertions.assertThat((Object)delegate).isNotEqualTo((Object)secondDelegate);
        secondConnection.close();
        Assertions.assertThat((int)mockConnectionFactory.openedConnections).isEqualTo(2);
        Assertions.assertThat((int)mockConnectionFactory.closedConnections).isEqualTo(1);
    }

    @Test
    public void testConnectionIsNotReusedWhenAutoCommitDisabled() throws Exception {
        try (MockConnectionFactory mockConnectionFactory = new MockConnectionFactory();
             ReusableConnectionFactory connectionFactory = new ReusableConnectionFactory((ConnectionFactory)mockConnectionFactory, FOREVER, 1000L);){
            Connection connection = connectionFactory.openConnection(ALICE);
            connection.setAutoCommit(false);
            connection.close();
            connectionFactory.openConnection(ALICE).close();
            Assertions.assertThat((int)mockConnectionFactory.openedConnections).isEqualTo(2);
        }
    }

    private static final class MockConnectionFactory
    implements ConnectionFactory {
        private int openedConnections;
        private int closedConnections;

        private MockConnectionFactory() {
        }

        public Connection openConnection(ConnectorSession session) {
            ++this.openedConnections;
            return new MockConnection(() -> ++this.closedConnections);
        }

        public void close() {
            Assertions.assertThat((int)this.openedConnections).isEqualTo(this.closedConnections);
        }
    }

    private static final class MockConnection
    extends ForwardingConnection {
        private final Runnable onClose;
        private volatile boolean closed;

        MockConnection(Runnable onClose) {
            this.onClose = Objects.requireNonNull(onClose, "onClose is null");
        }

        protected Connection delegate() {
            throw new UnsupportedOperationException();
        }

        public boolean isClosed() {
            return this.closed;
        }

        public void setAutoCommit(boolean autoCommit) {
        }

        public void setReadOnly(boolean readOnly) {
        }

        public void close() {
            Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"Connection is already closed");
            this.closed = true;
            this.onClose.run();
        }
    }
}

