/*
 * Copyright 2022 Salesforce, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.netflix.spinnaker.kork.test.log;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.Logger;
import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.Appender;
import ch.qos.logback.core.Layout;
import ch.qos.logback.core.OutputStreamAppender;
import ch.qos.logback.core.encoder.LayoutWrappingEncoder;
import ch.qos.logback.core.read.ListAppender;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import org.slf4j.LoggerFactory;

/**
 * Store log events to facilitate assertions. Inspired by
 * https://www.baeldung.com/junit-asserting-logs.
 */
public class MemoryAppender extends ListAppender<ILoggingEvent> {

  /** This makes it possible to operate on the contents of actual, fully laid out log messages. */
  private static final Layout<ILoggingEvent> layout = getRootLayout();

  /** Capture log events generated by this logger */
  private final Logger logger;

  /**
   * Construct a MemoryAppender and start capturing debug (and more severe) log messages.
   *
   * @param loggerClass capture log events generated by this class
   */
  public MemoryAppender(Class loggerClass) {
    this(loggerClass, Level.DEBUG);
  }

  /**
   * Construct a MemoryAppender and start capturing logs.
   *
   * @param loggerClass capture log events generated by this class
   * @param level capture log events up to this level
   */
  public MemoryAppender(Class loggerClass, Level level) {
    super();
    this.logger = (Logger) LoggerFactory.getLogger(Objects.requireNonNull(loggerClass));
    this.setContext((LoggerContext) LoggerFactory.getILoggerFactory());
    this.logger.addAppender(this);
    this.logger.setLevel(level);
    start();
  }

  /** Clear captured log messages */
  public void reset() {
    this.list.clear();
  }

  /** Count log messages at a particular level */
  public long countEventsForLevel(Level level) {
    return this.list.stream().filter(event -> event.getLevel() == level).count();
  }

  /**
   * Return laid out log messages at a particular level that contain a string
   *
   * @param string the string to look for
   * @param level the log level
   * @return all log messages that satisfy the criteria
   */
  public List<String> layoutSearch(String string, Level level) {

    return this.list.stream()
        .filter(event -> event.getLevel().equals(level))
        .map(layout::doLayout)
        .filter(message -> message.contains(string))
        .collect(Collectors.toList());
  }

  /**
   * Return log messages at a particular level that contain a string
   *
   * @param string the string to look for
   * @param level the log level
   * @return all log messages that satisfy the criteria
   */
  public List<String> search(String string, Level level) {
    return this.list.stream()
        .filter(event -> event.getLevel().equals(level))
        .map(Object::toString)
        .filter(message -> message.contains(string))
        .collect(Collectors.toList());
  }

  /** Accessor for the root layout for the root logger. */
  private static Layout<ILoggingEvent> getRootLayout() {
    Logger rootLogger = (Logger) LoggerFactory.getLogger(Logger.ROOT_LOGGER_NAME);

    final Iterator<Appender<ILoggingEvent>> appenderIterator = rootLogger.iteratorForAppenders();

    // Assume the root logger has exactly one appender, from which we grab its
    // layout.
    final Appender<ILoggingEvent> appender = appenderIterator.next();

    OutputStreamAppender<ILoggingEvent> rootAppender = null;
    if (appender instanceof OutputStreamAppender) {
      // And, assume it's an OutputStreamAppender.  We need this to get at its
      // encoder...
      rootAppender = (OutputStreamAppender<ILoggingEvent>) appender;
    } else {
      throw new IllegalStateException("no root OutputStreamAppender");
    }

    // Verify the one encoder assumption.
    if (appenderIterator.hasNext()) {
      throw new IllegalStateException("multiple encoders -- can't determine layout");
    }

    // Finally, grab the layout.

    // Look for LayoutWrappingEncoder since it's the class that introduces
    // Layout.  See http://logback.qos.ch/manual/encoders.html.
    if (!(rootAppender.getEncoder() instanceof LayoutWrappingEncoder)) {
      throw new IllegalStateException("no LayoutWrappingEncoder -- can't determine layout");
    }

    final LayoutWrappingEncoder<ILoggingEvent> rootEncoder =
        (LayoutWrappingEncoder<ILoggingEvent>) rootAppender.getEncoder();

    return rootEncoder.getLayout();
  }
}
