/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.stork.servicediscovery.dns;

import io.smallrye.mutiny.Multi;
import io.smallrye.mutiny.Uni;
import io.smallrye.mutiny.subscription.MultiEmitter;
import io.smallrye.mutiny.tuples.Tuple2;
import io.smallrye.stork.api.Metadata;
import io.smallrye.stork.api.ServiceInstance;
import io.smallrye.stork.impl.CachingServiceDiscovery;
import io.smallrye.stork.impl.DefaultServiceInstance;
import io.smallrye.stork.servicediscovery.dns.DnsConfiguration;
import io.smallrye.stork.servicediscovery.dns.DnsMetadataKey;
import io.smallrye.stork.servicediscovery.dns.DnsRecordType;
import io.smallrye.stork.utils.DurationUtils;
import io.smallrye.stork.utils.HostAndPort;
import io.smallrye.stork.utils.ServiceInstanceIds;
import io.smallrye.stork.utils.ServiceInstanceUtils;
import io.smallrye.stork.utils.StorkAddressUtils;
import io.vertx.core.Vertx;
import io.vertx.core.dns.DnsClient;
import io.vertx.core.dns.DnsClientOptions;
import io.vertx.core.dns.SrvRecord;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DnsServiceDiscovery
extends CachingServiceDiscovery {
    private static final Logger log = LoggerFactory.getLogger(DnsServiceDiscovery.class);
    private final String serviceName;
    private final String hostname;
    private final DnsRecordType recordType;
    private final boolean secure;
    private final Integer port;
    private final boolean failOnError;
    private final long dnsTimeoutMs;
    private final boolean recursionDesired;
    private final boolean resolveSrv;
    final Map<String, DnsClient> dnsClients = new HashMap<String, DnsClient>();

    DnsServiceDiscovery(String serviceName, DnsConfiguration config, Vertx vertx) {
        super(config.getRefreshPeriod());
        this.serviceName = serviceName;
        this.secure = this.isSecure(config);
        this.recordType = this.recordType(config.getRecordType());
        this.failOnError = Boolean.parseBoolean(config.getFailOnError());
        this.recursionDesired = Boolean.parseBoolean(config.getRecursionDesired());
        this.resolveSrv = Boolean.parseBoolean(config.getResolveSrv());
        this.dnsTimeoutMs = DurationUtils.parseDuration((String)config.getDnsTimeout(), (String)"DNS timeout").toMillis();
        String dnsServersString = config.getDnsServers();
        if (dnsServersString != null && !dnsServersString.isBlank() && !"none".equalsIgnoreCase(dnsServersString)) {
            for (String dnsServer : dnsServersString.split(",")) {
                HostAndPort hostAndPort = StorkAddressUtils.parseToHostAndPort((String)dnsServer, (int)53, (String)("DNS server address for service " + serviceName));
                this.dnsClients.put(dnsServer, this.createClient(vertx, hostAndPort));
            }
        }
        if (this.dnsClients.isEmpty()) {
            this.dnsClients.put("Default DNS", this.createClient(vertx, null));
        }
        this.hostname = config.getHostname() == null ? serviceName : config.getHostname();
        try {
            this.port = config.getPort() == null ? null : Integer.valueOf(Integer.parseInt(config.getPort()));
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException("Invalid port for service " + serviceName, e);
        }
        if (this.port == null && this.recordType != DnsRecordType.SRV) {
            throw new IllegalArgumentException("DNS service discovery for record types different than SRV require service instance port to be specified");
        }
    }

    private DnsRecordType recordType(String recordType) {
        String recordTypeString = recordType.toUpperCase(Locale.ROOT);
        try {
            return DnsRecordType.valueOf(recordTypeString);
        }
        catch (IllegalArgumentException e) {
            throw new IllegalArgumentException("Invalid DNS record type '" + recordType + "' for service " + this.serviceName + ". The available types are " + Arrays.toString((Object[])DnsRecordType.values()), e);
        }
    }

    private DnsClient createClient(Vertx vertx, HostAndPort hostAndPort) {
        DnsClientOptions options = new DnsClientOptions().setQueryTimeout(this.dnsTimeoutMs).setRecursionDesired(this.recursionDesired);
        if (hostAndPort != null) {
            options.setHost(hostAndPort.host).setPort(hostAndPort.port);
        }
        return vertx.createDnsClient(options);
    }

    public Uni<List<ServiceInstance>> fetchNewServiceInstances(List<ServiceInstance> previousInstances) {
        AtomicInteger queriesLeft = new AtomicInteger(this.dnsClients.size());
        AtomicBoolean successRecorded = new AtomicBoolean();
        if (this.recordType == DnsRecordType.SRV) {
            return this.resolveSRV(previousInstances);
        }
        return this.resolveAorAAA(previousInstances, queriesLeft, successRecorded);
    }

    private Uni<List<ServiceInstance>> resolveAorAAA(List<ServiceInstance> previousInstances, AtomicInteger queriesLeft, AtomicBoolean successRecorded) {
        Multi serviceInstances = Multi.createFrom().emitter(em -> {
            block4: for (Map.Entry<String, DnsClient> dnsClient : this.dnsClients.entrySet()) {
                DnsClient client = dnsClient.getValue();
                switch (this.recordType) {
                    case A: {
                        client.resolveA(this.hostname).onFailure(error -> this.handleResolutionFailure((Throwable)error, queriesLeft, (MultiEmitter<?>)em, (String)dnsClient.getKey())).onSuccess(lst -> this.handleStringResolution((List<String>)lst, (MultiEmitter<? super ServiceInstance>)em, queriesLeft, previousInstances, successRecorded));
                        continue block4;
                    }
                    case AAAA: {
                        client.resolveAAAA(this.hostname).onFailure(error -> this.handleResolutionFailure((Throwable)error, queriesLeft, (MultiEmitter<?>)em, (String)dnsClient.getKey())).onSuccess(lst -> this.handleStringResolution((List<String>)lst, (MultiEmitter<? super ServiceInstance>)em, queriesLeft, previousInstances, successRecorded));
                        continue block4;
                    }
                }
                em.fail((Throwable)new IllegalStateException("Unsupported DNS record type " + this.recordType));
            }
        });
        return this.collectResults(successRecorded, (Multi<ServiceInstance>)serviceInstances);
    }

    private Uni<List<ServiceInstance>> collectResults(AtomicBoolean successRecorded, Multi<ServiceInstance> serviceInstances) {
        return serviceInstances.collect().asList().onItem().transformToUni(result -> {
            if (successRecorded.get()) {
                return Uni.createFrom().item(result);
            }
            return Uni.createFrom().failure((Throwable)new RuntimeException("No DNS server was able to resolve '" + this.hostname + "'"));
        });
    }

    private Uni<List<ServiceInstance>> resolveSRV(List<ServiceInstance> previousInstances) {
        AtomicInteger queriesLeft = new AtomicInteger(this.dnsClients.size());
        AtomicBoolean successRecorded = new AtomicBoolean();
        Multi records = Multi.createFrom().emitter(em -> {
            for (Map.Entry<String, DnsClient> clientEntry : this.dnsClients.entrySet()) {
                DnsClient client = clientEntry.getValue();
                client.resolveSRV(this.hostname).onFailure(error -> this.handleResolutionFailure((Throwable)error, queriesLeft, (MultiEmitter<?>)em, (String)clientEntry.getKey())).onSuccess(lst -> {
                    successRecorded.set(true);
                    for (SrvRecord record : lst) {
                        em.emit((Object)Tuple2.of((Object)client, (Object)record));
                    }
                    if (queriesLeft.decrementAndGet() == 0) {
                        em.complete();
                    }
                });
            }
        });
        if (!this.resolveSrv) {
            Multi targets = records.onItem().transformToUni(record -> {
                String target = ((SrvRecord)record.getItem2()).target();
                return Uni.createFrom().item((Object)this.toStorkServiceInstance(target, ((SrvRecord)record.getItem2()).port(), ((SrvRecord)record.getItem2()).weight(), previousInstances));
            }).concatenate();
            return this.collectResults(successRecorded, (Multi<ServiceInstance>)targets);
        }
        Multi instances = records.onItem().transformToUni(record -> {
            String target = ((SrvRecord)record.getItem2()).target();
            DnsClient client = (DnsClient)record.getItem1();
            Uni aInstances = Uni.createFrom().emitter(em -> client.resolveA(target, addresses -> {
                if (addresses.failed()) {
                    log.warn("Failed to lookup the address retrieved from DNS: " + target, addresses.cause());
                    em.complete(Collections.emptyList());
                } else {
                    em.complete(addresses.result());
                }
            }));
            Uni aaaaInstances = Uni.createFrom().emitter(em -> client.resolveAAAA(target, addresses -> {
                if (addresses.failed()) {
                    log.warn("Failed to lookup the address retrieved from DNS: " + target, addresses.cause());
                    em.complete(Collections.emptyList());
                } else {
                    em.complete(addresses.result());
                }
            }));
            return Uni.combine().all().unis(aInstances, aaaaInstances).combinedWith((strings, strings2) -> {
                ArrayList result = new ArrayList(strings);
                result.addAll(strings2);
                if (result.isEmpty()) {
                    log.warn("Failed to resolve ip address for target from SRV request: " + target);
                }
                return result;
            }).onItem().transform(addresses -> addresses.stream().map(address -> this.toStorkServiceInstance((String)address, ((SrvRecord)record.getItem2()).port(), ((SrvRecord)record.getItem2()).weight(), previousInstances)).collect(Collectors.toList()));
        }).concatenate().onItem().transformToMulti(l -> Multi.createFrom().iterable((Iterable)l)).concatenate();
        return this.collectResults(successRecorded, (Multi<ServiceInstance>)instances);
    }

    private void handleStringResolution(List<String> lst, MultiEmitter<? super ServiceInstance> em, AtomicInteger queriesLeft, List<ServiceInstance> previousInstances, AtomicBoolean successRecorded) {
        for (String target : lst) {
            em.emit((Object)this.toStorkServiceInstance(target, this.port, 1, previousInstances));
        }
        successRecorded.set(true);
        if (queriesLeft.decrementAndGet() == 0) {
            em.complete();
        }
    }

    private void handleResolutionFailure(Throwable error, AtomicInteger queriesLeft, MultiEmitter<?> em, String dnsServer) {
        String message = "Failure resolving name " + this.hostname + " with " + dnsServer;
        log.warn(message, error);
        if (this.failOnError) {
            em.fail((Throwable)new RuntimeException(message, error));
        }
        if (queriesLeft.decrementAndGet() == 0) {
            em.complete();
        }
    }

    private ServiceInstance toStorkServiceInstance(String target, int port, int weight, List<ServiceInstance> previousInstances) {
        if (this.port != null) {
            port = this.port;
        }
        Metadata<DnsMetadataKey> dnsMetadata = this.createDnsMetadata(this.hostname, weight);
        ServiceInstance matching = ServiceInstanceUtils.findMatching(previousInstances, (String)target, (int)port);
        if (matching == null) {
            return new DefaultServiceInstance(ServiceInstanceIds.next().longValue(), target, port, this.secure, dnsMetadata);
        }
        return matching;
    }

    private Metadata<DnsMetadataKey> createDnsMetadata(String hostname, int weight) {
        Metadata dnsMetadata = Metadata.of(DnsMetadataKey.class);
        dnsMetadata = dnsMetadata.with((Enum)DnsMetadataKey.DNS_NAME, (Object)hostname);
        dnsMetadata = dnsMetadata.with((Enum)DnsMetadataKey.DNS_WEIGHT, (Object)weight);
        return dnsMetadata;
    }

    private boolean isSecure(DnsConfiguration config) {
        return config.getSecure() != null && Boolean.parseBoolean(config.getSecure());
    }
}

