/*
 * This file is part of ToolFactory JVM driver.
 *
 * Hosted at: https://github.com/toolfactory/jvm-driver
 *
 * --
 *
 * The MIT License (MIT)
 *
 * Copyright (c) 2019-2021 Luke Hutchison, Roberto Gentili
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
 * documentation files (the "Software"), to deal in the Software without restriction, including without
 * limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
 * LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
 * EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
 * AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
 * OR OTHER DEALINGS IN THE SOFTWARE.
 */
package io.github.toolfactory.jvm.function.util;


import java.nio.ByteBuffer;

import io.github.toolfactory.jvm.function.template.Consumer;
import io.github.toolfactory.jvm.function.template.Function;


public class JavaClass {
	private String classNameSlashed;
	private String className;

	private JavaClass(String className, ByteBuffer byteCode) {
		this.classNameSlashed = className;
	}

	private JavaClass(ByteBuffer byteCode) {
		this(Classes.retrieveName(byteCode), BufferHandler.shareContent(byteCode));
	}

	public static JavaClass create(ByteBuffer byteCode) {
		return new JavaClass(byteCode);
	}

	public static void use(ByteBuffer byteCode, Consumer<JavaClass> javaClassConsumer) {
		javaClassConsumer.accept(JavaClass.create(byteCode));
	}

	public static <T, E extends Throwable> T extractByUsing(ByteBuffer byteCode, Function<JavaClass, T> javaClassConsumer) throws E {
		return javaClassConsumer.apply(JavaClass.create(byteCode));
	}

	private  String _getPackageName() {
		return classNameSlashed.contains("/") ?
			classNameSlashed.substring(0, classNameSlashed.lastIndexOf("/")) :
			null;
	}

	private String _getSimpleName() {
		return classNameSlashed.contains("/") ?
			classNameSlashed.substring(classNameSlashed.lastIndexOf("/") + 1) :
			classNameSlashed;
	}

	public String getPackageName() {
		String pckgName = _getPackageName();
		if (pckgName != null) {
			pckgName = pckgName.replace("/", ".");
		}
		return pckgName;
	}

	public String getSimpleName() {
		return _getSimpleName();
	}


	public String getName() {
		if (className == null) {
			String packageName = getPackageName();
			String classSimpleName = getSimpleName();
			String name = null;
			if (packageName != null) {
				name = packageName;
			}
			if (classSimpleName != null) {
				if (packageName == null) {
					name = "";
				} else {
					name += ".";
				}
				name += classSimpleName;
			}
			className = name;
		}
		return className;
	}

}