package io.github.effiban.scala2java.test.utils.integration.runner

import io.github.effiban.scala2java.core.Scala2JavaTranslator.translate
import io.github.effiban.scala2java.core.entities.FileExtensions
import io.github.effiban.scala2java.test.utils.integration.matchers.FileMatchers.equalContentsOf
import org.scalatest.funsuite.AnyFunSuiteLike
import org.scalatest.matchers.should.Matchers
import org.scalatest.{BeforeAndAfterAll, OptionValues}

import java.nio.file.{Files, Path, Paths}
import scala.collection.Set
import scala.jdk.CollectionConverters._
import scala.jdk.StreamConverters._
import scala.util.Using

/**
 * ===Overview===
 * A convenience runner for executing integration tests of the Scala2Java tool and its extensions.<br>
 * The runner will automatically discover test data files, and will then generate and execute test methods for them.<br>
 *
 * ===Usage===
 * To use the runner, implement this trait and provide the following:
 *
 *   - (mandatory) The base path containing the test data (explained in detail below)
 *   - (optional) A selected path for filtering the tests to be executed, or the empty string (default) for executing all the tests.
 *
 * To run the tests, simply run your implementation class and it will run all tests satisfying the filter.<br>
 * If a test fails, a nicely-formatted error message will be generated comparing the expected and actual contents side-by-side,
 * making it easy to investigate the failure.
 *
 * ===Test Data===
 *
 * The test data files '''must''' satisfy the following requirements:
 *   - The files must all be under a single base path
 *   - The files for each test must be in a separate directory
 *   - There must be exactly two files for each test:
 *     - Input Scala file with a `.scala` extension, containing the code to be translated
 *     - Expected output Java file which must have:
 *       - A filename, with a `.java` extension, which is the name expected to be generated by the tool
 *         (it should be the same as the Scala file name, unless it is supposed to be transformed - see [[io.github.effiban.scala2java.spi.transformers.FileNameTransformer]])
 *       - Contents which are the expected Java code matching the input Scala code
 *
 *  For example, suppose the following directory tree exists for the base path 'testdata':
 *
 *  - testdata
 *    - Scenario1
 *      - Class1.scala
 *      - Class1.java
 *    - Scenario2
 *      - Class2.scala
 *      - Class2.java
 *    - Scenario3
 *      - Class3.scala
 *      - Class3.java
 *
 * In this case, the runner will automatically generate and execute three test methods, as follows:
 *   1.
 *     - test name: "translate [Scenario1]"
 *     - action:    Executes tool on `Class1.scala`, expecting output file with same name + contents as `Class1.java`
 *
 *   1.
 *     - test name: "translate [Scenario2]"
 *     - action:    Executes tool on `Class1.scala`, expecting output file with same name + contents as `Class2.java`
 *
 *   1.
 *     - test name: "translate [Scenario3]"
 *     - action:    Executes tool on `Class3.scala`, expecting output file with same name + contents as `Class3.java`
 *
 * '''NOTES'''
 *   1. Since each pair of test files has its own directory, it is perfectly valid to reuse the same class name for all
 *   (the test name will contain the path without the file names, so it will not matter).
 *   1. The directory structure can be nested to any depth you like, provided all test files share the same base path.
 *   This way, you can use the directory names to describe the scenario you are testing.
 *   For example, you might create the following directory tree (considering the previous note as well):
 *
 *   - testdata
 *    - Method
 *      - ReturningInt
 *        - MyClass.scala
 *        - MyClass.java
 *      - ReturningString
 *        - MyClass.scala
 *        - MyClass.java
 *    - LocalVariable
 *      - OfType
 *        - List
 *          - MyClass.scala
 *          - MyClass.java
 *        - Set
 *          - MyClass.scala
 *          - MyClass.java
 *      - DataMember
 *        - OfType
 *          - Set
 *            - MyClass.scala
 *            - MyClass.java
 *
 *   This would generate test methods named:
 *   - "translate [Method/ReturningInt]"
 *   - "translate [LocalVariable/OfType/List]"
 *   - "translate [LocalVariable/OfType/Set]"
 *   - "translate [DataMember/OfType/Set]"
 */
trait IntegrationTestRunner
  extends Matchers
  with OptionValues
  with BeforeAndAfterAll { this: AnyFunSuiteLike =>

  private val testFilesBasePath: Path = resolveTestFilesBasePath()
  private val selectedTestPath: String = resolveSelectedTestPath()

  private var outputJavaBasePath: Path = _

  /** Resolves the absolute base path for the test data files.<br>
   * Typically, it will lookup some environment variable or system property.<br>
   * This method '''must''' be overriden and must return a valid absolute path containing test files,
   * which satisfy the structure described in the top-level documentation of this trait.
   *
   * @return the absolute base path containing all test data files
   */
  protected def resolveTestFilesBasePath(): Path

  /** Resolves the selected path for filtering the test data files to be included.<br>
   * This can be a string which is a full or partial match to a valid path containing test files.<br>
   * Typically, it will lookup some environment variable or system property.<br>
   * This method is optional - override only if you wish to be able to filter the tests being executed.
   *
   * @return if overriden - the selected test path
   *         otherwise - an empty string, meaning that all tests will be executed
   */
  protected def resolveSelectedTestPath(): String = ""

  override protected def beforeAll(): Unit = {
    super.beforeAll()

    outputJavaBasePath = Files.createTempDirectory("outputjava")
  }

  override protected def afterAll(): Unit = {
    super.afterAll()

    val outputJavaBaseDir = outputJavaBasePath.toFile
    outputJavaBaseDir.listFiles().foreach(_.delete())
    outputJavaBaseDir.delete()
  }

  Using(Files.walk(testFilesBasePath)) { stream =>
    stream.toScala(LazyList)
      .filter(path => selectedTestPath.isBlank || path.toString.contains(selectedTestPath))
      .filter(Files.isDirectory(_))
      .map(regularPathsInside)
      .map(toPathPair)
      .collect { case Some(pathPair) => pathPair }
      .foreach(test)
  }

  private def test(pathPair: PathPair): Unit = {
    import pathPair._
    val relativePath = testFilesBasePath.relativize(scalaPath.getParent)
    test(s"translate [$relativePath]") {
      val javaFileName = javaPath.getFileName.toString
      val expectedJavaPath = pathOf(testFilesBasePath, relativePath, javaFileName)
      val outputJavaPath = pathOf(outputJavaBasePath, relativePath, javaFileName)

      translate(scalaPath, Some(outputJavaPath.getParent))

      withClue(s"Output java file not found at $outputJavaPath") {
        outputJavaPath.toFile.exists() shouldBe true
      }
      outputJavaPath should equalContentsOf(expectedJavaPath)
    }
  }

  private def pathOf(basePath: Path, relativePath: Path, fileName: String) = Paths.get(basePath.toString, relativePath.toString, fileName)

  private def regularPathsInside(dir: Path) = {
    java.util.List.of(dir.toFile.listFiles(): _*).asScala
      .filterNot(_.isDirectory)
      .map(_.toPath)
      .toList
  }

  private def toPathPair(paths: List[Path]): Option[PathPair] = {
    val pathMap = paths.map(path => (extractFileExtension(path), path)).toMap
    if (pathMap.keys.toSet == Set(FileExtensions.Scala, FileExtensions.Java)) {
      Some(PathPair(scalaPath = pathMap(FileExtensions.Scala), javaPath = pathMap(FileExtensions.Java)))
    } else {
      None
    }
  }

  private def extractFileExtension(path: Path) = path.getFileName.toString.split("\\.").lastOption.getOrElse("")

  private case class PathPair(scalaPath: Path, javaPath: Path)
}
