package io.github.effiban.scala2java.spi

import io.github.effiban.scala2java.spi.predicates.ExtendedPredicates
import io.github.effiban.scala2java.spi.providers.ExtendedProviders
import io.github.effiban.scala2java.spi.transformers.ExtendedTransformers
import io.github.effiban.scala2java.spi.typeinferrers.ExtendedTypeInferrers

import scala.meta.Term

/** The trait which must be implemented by any extension of the Scala2Java tool.
 * The purpose of an extension is to provide additional customization for the Java code generated by the Scala2Java core tool.
 * Typically, it will apply modifications which are required by certain frameworks having different syntax in Scala and Java.
 *
 * This trait includes the following:
 *   - A mandatory method `shouldBeAppliedIfContains`, which is the filter indicating whether the extension should be applied.
 *   - A collection of 'hook provider' methods, each providing an implementation which can modify some element of the Scala code.<br>
 *     '''NOTE that''':
 *      - All hooks are optional. If a provider is not implemented - a default implementation will be returned, having no effect.
 *      - A hook will always be applied __before__ any core logic of the tool operates on the same Scala code.
 *
 * Extensions are designed to be loaded dynamically by the tool at runtime.
 * To achieve this, an extension '''must''' make itself discoverable by a [[java.util.ServiceLoader]] - see there for details.
 */
trait Scala2JavaExtension
  extends ExtendedPredicates
    with ExtendedProviders
    with ExtendedTransformers
    with ExtendedTypeInferrers {

  /** Indicates whether this extension should be applied if the given [[scala.meta.Term.Select]] (qualified name) appears in the Scala source file.<br>
   * The tool will invoke this method for every qualified name in the file, and if at least one returns `true` - the extension as a whole will be applied.
   *
   * @param termSelect the qualified name to evaluate
   * @return `true` if the extension should be applied, `false` otherwise
   */
  def shouldBeAppliedIfContains(termSelect: Term.Select): Boolean
}
