/* Quark 1.0.452 run at 2016-10-24 14:33:24.783025 */
package mdk;

/**
 * A session provides a lightweight sequential context that a
 * microservice can use in the context of any application
 * framework in order to manage its interactions with other
 * microservices. It provides simple APIs for service
 * resolution, distributed tracing, and circuit breakers.
 *
 * A microservices architecture enables small self contained
 * units of business logic to be implemented by separate teams
 * working on isolated services based on the languages and
 * frameworks best suited for their problem domain.
 *
 * Any given microservice will contain sequential business logic
 * implemented in a variety of ways depending on the application
 * framework chosen. For example it may be a long running
 * thread, a simple blocking request handler, or a chained
 * series of reactive handlers in an async environment.
 *
 * For the most part this business logic can be implemented
 * exactly as prescribed by the application framework of choice,
 * however in a microservices architecture, some special care
 * needs to be taken when this business logic interacts with
 * other microservices.
 *
 * Because microservices are updated with much higher frequency
 * than normal web applications, the interactions between them
 * form key points that require extra care beyond normal web
 * interactions in order to avoid creating a system that is both
 * extremely fragile, unreliable, and opaque.
 *
 * Realtime service resolution, distributed tracing, and
 * resilience heuristics such as circuit breakers provide the
 * foundational behavior required at these interaction
 * points. These capabilites must be combined with the defensive
 * coding practice of intelligent fallback behavior when remote
 * services are unavailable or misbehaving, in order to build a
 * robust microservice application.
 *
 * Because of this, a session is expected to be created and made
 * available to all business logic within a given microservice,
 * e.g. on a per request basis, as a thread local, part of a
 * context object, etc depending on the application framework of
 * choice.
 *
 */
public interface Session {
    public static quark.reflect.Class mdk_Session_ref = datawire_mdk_md.Root.mdk_Session_md;
    /**
     * Grabs the encoded context.
     */
     String inject();
    /**
     * Returns an externalized representation of the distributed session.
     */
     String externalize();
    /**
     * Record a log entry at the CRITICAL logging level.
     */
     void critical(String category, String text);
    /**
     * Record a log entry at the ERROR logging level.
     */
     void error(String category, String text);
    /**
     * Record a log entry at the WARN logging level.
     */
     void warn(String category, String text);
    /**
     * Record a log entry at the INFO logging level.
     */
     void info(String category, String text);
    /**
     * Record a log entry at the DEBUG logging level.
     */
     void debug(String category, String text);
    /**
     * EXPERIMENTAL: Set the logging level for the session.
     */
     void trace(String level);
    /**
     * EXPERIMENTAL; requires MDK_EXPERIMENTAL=1 environment variable to
     * function.
     *
     * Override service resolution for the current distributed
     * session. All attempts to resolve *service*, *version*
     * will be replaced with an attempt to resolve *target*,
     * *targetVersion*. This effect will be propogated to any
     * downstream services involved in the distributed session.
     *
     */
     void route(String service, String version, String target, String targetVersion);
    /**
     * Locate a compatible service instance.
     *
     * Uses a minimum of 10 seconds and the timeout set on the session.
     *
     */
     mdk_discovery.Node resolve(String service, String version);
    /**
     * Locate a compatible service instance with a non-default timeout.
     *
     */
     mdk_discovery.Node resolve_until(String service, String version, Double timeout);
    /**
     * Locate a compatible service instance asynchronously. The result is returned as a promise.
     *
     */
     Object resolve_async(String service, String version);
    /**
     * Start an interaction with a remote service.
     *
     * The session tracks any nodes resolved during an
     * interactin with a remote service.
     *
     * The service resolution API permits a compatible instance
     * of the service to be located. In addition, it tracks
     * which exact instances are in use during any
     * interaction. Should the interaction fail, circuit breaker
     * state is updated for those nodes, and all involved
     * instances involved are reported to the tracing services.
     *
     * This permits realtime reporting of integration issues
     * when services are updated, and also allows circuit
     * breakers to mitigate the impact of any such issues.
     *
     */
     void start_interaction();
    /**
     * Record an interaction as failed.
     *
     * This will update circuit breaker state for the remote
     * nodes, as well as reporting all nodes involved to the
     * tracing system.
     *
     */
     void fail_interaction(String message);
    /**
     * Finish an interaction.
     *
     * This marks an interaction as completed.
     *
     */
     void finish_interaction();
    /**
     * This is a convenience API that will perform
     * start_interaction() followed by callable(ssn) followed by
     * finish_interaction().
     *
     *
     */
     void interact(quark.UnaryCallable callable);
    /**
     * Set how many seconds the session is expected to live from this point.
     *
     * If a timeout has previously been set the new timeout will only be
     * used if it is lower than the existing timeout.
     *
     * The MDK will not enforce the timeout. Rather, it provides the
     * information to any process or server in the same session (even if
     * they are on different machines). By passing this timeout to
     * blocking APIs you can ensure timeouts are enforced across a whole
     * distributed session.
     *
     */
     void setDeadline(Double seconds);
    /**
     * DEPRECATED, use setDeadline().
     */
     void setTimeout(Double seconds);
    /**
     * Return how many seconds until the session ought to end.
     *
     * This will only be accurate across multiple servers insofar as their
     * clocks are in sync.
     *
     * If a timeout has not been set the result will be null.
     *
     */
     Double getRemainingTime();
    /**
     * Return the value of a property from the distributed session.
     */
     Object getProperty(String property);
    /**
     * Set a property on the distributed session.
     *
     * The key should be prefixed with a namespace so that it doesn't conflict
     * with built-in properties, e.g. 'examplenamespace:myproperty' instead of
     * 'myproperty'.
     *
     * The value should be JSON serializable.
     *
     */
     void setProperty(String property, Object value);
    /**
     * Return whether the distributed session has a property.
     */
     Boolean hasProperty(String property);
}
