/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.util;

import io.confluent.ksql.schema.ksql.types.SqlDecimal;
import io.confluent.ksql.schema.ksql.types.SqlTypes;
import io.confluent.ksql.util.KsqlException;
import io.confluent.ksql.util.KsqlPreconditions;
import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;
import java.text.DecimalFormat;
import org.apache.kafka.connect.data.Decimal;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.data.SchemaBuilder;
import org.apache.kafka.connect.errors.DataException;

public final class DecimalUtil {
    private static final String PRECISION_FIELD = "connect.decimal.precision";

    private DecimalUtil() {
    }

    public static boolean isDecimal(Schema schema) {
        return schema.type() == Schema.Type.BYTES && "org.apache.kafka.connect.data.Decimal".equals(schema.name());
    }

    public static void requireDecimal(Schema schema) {
        KsqlPreconditions.checkArgument(DecimalUtil.isDecimal(schema), String.format("Expected schema of type DECIMAL but got a schema of type %s and name %s", schema.type(), schema.name()));
    }

    public static SchemaBuilder builder(Schema schema) {
        DecimalUtil.requireDecimal(schema);
        return DecimalUtil.builder(DecimalUtil.precision(schema), DecimalUtil.scale(schema));
    }

    public static SchemaBuilder builder(int precision, int scale) {
        DecimalUtil.validateParameters(precision, scale);
        return Decimal.builder((int)scale).optional().parameter(PRECISION_FIELD, Integer.toString(precision));
    }

    public static int scale(Schema schema) {
        DecimalUtil.requireDecimal(schema);
        String scaleString = (String)schema.parameters().get("scale");
        if (scaleString == null) {
            throw new DataException("Invalid Decimal schema: scale parameter not found.");
        }
        try {
            return Integer.parseInt(scaleString);
        }
        catch (NumberFormatException e) {
            throw new DataException("Invalid scale parameter found in Decimal schema: ", (Throwable)e);
        }
    }

    public static int precision(Schema schema) {
        DecimalUtil.requireDecimal(schema);
        String precisionString = (String)schema.parameters().get(PRECISION_FIELD);
        if (precisionString == null) {
            throw new DataException("Invalid Decimal schema: precision parameter not found.");
        }
        try {
            return Integer.parseInt(precisionString);
        }
        catch (NumberFormatException e) {
            throw new DataException("Invalid precision parameter found in Decimal schema: ", (Throwable)e);
        }
    }

    public static String format(int precision, int scale, BigDecimal value) {
        DecimalFormat format = new DecimalFormat();
        format.setMinimumIntegerDigits(precision - scale);
        format.setMinimumFractionDigits(scale);
        return format.format(value);
    }

    public static BigDecimal ensureFit(BigDecimal value, Schema schema) {
        if (value == null) {
            return null;
        }
        int precision = DecimalUtil.precision(schema);
        int scale = DecimalUtil.scale(schema);
        DecimalUtil.validateParameters(precision, scale);
        DecimalUtil.ensureMax(value, precision, scale);
        try {
            return value.setScale(scale, RoundingMode.UNNECESSARY);
        }
        catch (ArithmeticException e) {
            throw new KsqlException(String.format("Cannot fit decimal '%s' into DECIMAL(%d, %d) without rounding. (Requires %d,%d)", value.toPlainString(), precision, scale, value.precision(), value.scale()));
        }
    }

    public static SqlDecimal toSqlDecimal(Schema schema) {
        switch (schema.type()) {
            case BYTES: {
                DecimalUtil.requireDecimal(schema);
                return SqlDecimal.of(DecimalUtil.precision(schema), DecimalUtil.scale(schema));
            }
            case INT32: {
                return SqlTypes.INT_UPCAST_TO_DECIMAL;
            }
            case INT64: {
                return SqlTypes.BIGINT_UPCAST_TO_DECIMAL;
            }
        }
        throw new KsqlException("Cannot convert schema of type " + schema.type() + " to decimal.");
    }

    public static BigDecimal cast(long value, int precision, int scale) {
        DecimalUtil.validateParameters(precision, scale);
        BigDecimal decimal = new BigDecimal(value, new MathContext(precision));
        DecimalUtil.ensureMax(decimal, precision, scale);
        return decimal.setScale(scale, RoundingMode.UNNECESSARY);
    }

    public static BigDecimal cast(double value, int precision, int scale) {
        DecimalUtil.validateParameters(precision, scale);
        BigDecimal decimal = new BigDecimal(value, new MathContext(precision));
        DecimalUtil.ensureMax(decimal, precision, scale);
        return decimal.setScale(scale, RoundingMode.HALF_UP);
    }

    public static BigDecimal cast(BigDecimal value, int precision, int scale) {
        if (precision == value.precision() && scale == value.scale()) {
            return value;
        }
        DecimalUtil.validateParameters(precision, scale);
        DecimalUtil.ensureMax(value, precision, scale);
        return value.setScale(scale, RoundingMode.HALF_UP);
    }

    public static BigDecimal cast(String value, int precision, int scale) {
        DecimalUtil.validateParameters(precision, scale);
        BigDecimal decimal = new BigDecimal(value);
        DecimalUtil.ensureMax(decimal, precision, scale);
        return decimal.setScale(scale, RoundingMode.HALF_UP);
    }

    private static void ensureMax(BigDecimal value, int precision, int scale) {
        int digits = precision - scale;
        BigDecimal maxValue = new BigDecimal(Math.pow(10.0, digits));
        if (maxValue.compareTo(value.abs()) < 1) {
            throw new ArithmeticException(String.format("Numeric field overflow: A field with precision %d and scale %d must round to an absolute value less than 10^%d. Got %s", precision, scale, digits, value.toPlainString()));
        }
    }

    public static void validateParameters(int precision, int scale) {
        KsqlPreconditions.checkArgument(precision > 0, String.format("DECIMAL precision must be >= 1: DECIMAL(%d,%d)", precision, scale));
        KsqlPreconditions.checkArgument(scale >= 0, String.format("DECIMAL scale must be >= 0: DECIMAL(%d,%d)", precision, scale));
        KsqlPreconditions.checkArgument(precision >= scale, String.format("DECIMAL precision must be >= scale: DECIMAL(%d,%d)", precision, scale));
    }
}

