/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.compress;

import com.yahoo.path.Path;
import com.yahoo.yolean.Exceptions;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.util.Objects;
import java.util.function.Predicate;
import java.util.zip.GZIPInputStream;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.ArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveInputStream;

public class ArchiveStreamReader
implements AutoCloseable {
    private final ArchiveInputStream<? extends ArchiveEntry> archiveInputStream;
    private final Options options;
    private long totalRead = 0L;
    private long entriesRead = 0L;

    private <T extends ArchiveEntry> ArchiveStreamReader(ArchiveInputStream<T> archiveInputStream, Options options) {
        this.archiveInputStream = Objects.requireNonNull(archiveInputStream);
        this.options = Objects.requireNonNull(options);
    }

    public static ArchiveStreamReader ofTarGzip(InputStream inputStream, Options options) {
        return new ArchiveStreamReader(new TarArchiveInputStream((InputStream)Exceptions.uncheck(() -> new GZIPInputStream(inputStream))), options);
    }

    public static ArchiveStreamReader ofZip(InputStream inputStream, Options options) {
        return new ArchiveStreamReader(new ZipArchiveInputStream(inputStream), options);
    }

    public ArchiveFile readNextTo(OutputStream outputStream) {
        try {
            ArchiveEntry entry;
            while ((entry = this.archiveInputStream.getNextEntry()) != null) {
                int read;
                Path path = Path.fromString(ArchiveStreamReader.requireNormalized(entry.getName(), this.options.allowDotSegment));
                if (ArchiveStreamReader.isSymlink(entry)) {
                    throw new IllegalArgumentException(this.archiveType() + " entry " + String.valueOf(path) + " is a symbolic link, which is unsupported");
                }
                if (entry.isDirectory() || !this.options.pathPredicate.test(path.toString())) continue;
                if (++this.entriesRead > this.options.maxEntries) {
                    throw new IllegalArgumentException(this.archiveType() + " contains more files than the limit of " + this.options.maxEntries);
                }
                long size = 0L;
                byte[] buffer = new byte[2048];
                while ((read = this.archiveInputStream.read(buffer)) != -1) {
                    this.totalRead += (long)read;
                    size += (long)read;
                    if (this.totalRead > this.options.maxSize) {
                        throw new IllegalArgumentException("Total size of " + this.archiveType() + " exceeds size limit of " + this.options.maxSize + " bytes");
                    }
                    if ((long)read > this.options.maxEntrySize) {
                        if (this.options.truncateEntry) continue;
                        throw new IllegalArgumentException("Size of " + this.archiveType() + " entry " + String.valueOf(path) + " exceeded entry size limit of " + this.options.maxEntrySize + " bytes");
                    }
                    outputStream.write(buffer, 0, read);
                }
                return new ArchiveFile(path, size);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return null;
    }

    @Override
    public void close() {
        Exceptions.uncheck(() -> this.archiveInputStream.close());
    }

    private String archiveType() {
        return this.archiveInputStream instanceof TarArchiveInputStream ? "TAR" : "ZIP";
    }

    private static boolean isSymlink(ArchiveEntry entry) {
        if (entry instanceof ZipArchiveEntry) {
            ZipArchiveEntry zipEntry = (ZipArchiveEntry)entry;
            return zipEntry.isUnixSymlink();
        }
        if (entry instanceof TarArchiveEntry) {
            TarArchiveEntry tarEntry = (TarArchiveEntry)entry;
            return tarEntry.isSymbolicLink();
        }
        throw new IllegalArgumentException("Unsupported archive entry " + entry.getClass().getSimpleName() + ", cannot check for symbolic link");
    }

    private static String requireNormalized(String name, boolean allowDotSegment) {
        for (String part : name.split("/")) {
            if (!part.isEmpty() && (allowDotSegment || !part.equals(".")) && !part.equals("..")) continue;
            throw new IllegalArgumentException("Unexpected non-normalized path found in zip content: '" + name + "'");
        }
        return name;
    }

    private static long requireNonNegative(String field, long n) {
        if (n < 0L) {
            throw new IllegalArgumentException(field + " cannot be negative, got " + n);
        }
        return n;
    }

    public static class Options {
        private long maxSize = 8L * (long)Math.pow(1024.0, 3.0);
        private long maxEntrySize = Long.MAX_VALUE;
        private long maxEntries = Long.MAX_VALUE;
        private boolean truncateEntry = false;
        private boolean allowDotSegment = false;
        private Predicate<String> pathPredicate = path -> true;

        private Options() {
        }

        public static Options standard() {
            return new Options();
        }

        public Options maxSize(long size) {
            this.maxSize = ArchiveStreamReader.requireNonNegative("size", size);
            return this;
        }

        public Options maxEntrySize(long size) {
            this.maxEntrySize = ArchiveStreamReader.requireNonNegative("size", size);
            return this;
        }

        public Options maxEntries(long count) {
            this.maxEntries = ArchiveStreamReader.requireNonNegative("count", count);
            return this;
        }

        public Options truncateEntry(boolean truncate) {
            this.truncateEntry = truncate;
            return this;
        }

        public Options pathPredicate(Predicate<String> predicate) {
            this.pathPredicate = predicate;
            return this;
        }

        public Options allowDotSegment(boolean allow) {
            this.allowDotSegment = allow;
            return this;
        }
    }

    public static class ArchiveFile {
        private final Path path;
        private final long size;

        public ArchiveFile(Path name, long size) {
            this.path = Objects.requireNonNull(name);
            this.size = ArchiveStreamReader.requireNonNegative("size", size);
        }

        public Path path() {
            return this.path;
        }

        public long size() {
            return this.size;
        }
    }
}

