/*
 * Copyright 2000-2022 Vaadin Ltd.
 *
 * Licensed under the Commercial Vaadin Developer License version 4.0 (CVDLv4);
 * you may not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * https://vaadin.com/license/cvdl-4.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.shared.ui.richtextarea;

import java.io.Serializable;

/**
 * Supporting data structure for snippets feature.
 * 
 * @since 8.16
 */
public class Snippet implements Serializable {

    private static final int MAX_TITLE_LENGTH = 20;

    public String title;
    public String text;

    /**
     * Creates an empty snippet
     */
    public Snippet() {
    	this(null);
    }

    /**
     * Creates a snippet with a specified body text. Title is automatically
     * generated from the first 20 characters in the text, with ellipsis added
     * if the length exceeds 20 characters.
     * 
     * @param text
     *            body text, may contain HTML markup
     */
    public Snippet(String text) {
    	this(null,text);
    }

    /**
     * Creates a snippet with a specified title and text. The title can be of
     * arbitrary length. The body text may contain HTML markup.
     * 
     * @param title
     *            title of the Snippet. May be of arbitrary length.
     * @param text
     *            body text, may contain HTML markup
     */
    public Snippet(String title, String text) {
        this.text = text != null ? text : "";
        this.title = title != null ? title : formatTextAsTitle(this.text);
    }

    /**
     * Get the set title text (either a user-specified string or up to 20
     * characters of the main snippet text.
     * 
     * @return a string
     */
    public String getTitle() {
        return title;
    }

    /**
     * Get the snippet content text.
     * 
     * @return a string
     */
    public String getText() {
        return text;
    }

    private static String formatTextAsTitle(String text) {
        return text.length() > MAX_TITLE_LENGTH
                ? text.substring(0, MAX_TITLE_LENGTH - 3) + "..."
                : text;
    }
}
