package com.smartystreets.api.us_street;

import com.smartystreets.api.Request;
import com.smartystreets.api.Response;
import com.smartystreets.api.Sender;
import com.smartystreets.api.Serializer;
import com.smartystreets.api.exceptions.SmartyException;

import java.io.IOException;

/**
 * This client sends lookups to the SmartyStreets US Street API, <br>
 *     and attaches the results to the appropriate Lookup objects.
 */
public class Client {
    private final Sender sender;
    private final Serializer serializer;

    public Client(Sender sender, Serializer serializer) {
        this.sender = sender;
        this.serializer = serializer;
    }

    public void send(Lookup lookup) throws SmartyException, IOException {
        Batch batch = new Batch();
        batch.add(lookup);
        this.send(batch);
    }

    /**
     *
     * @param batch Batch must contain between 1 and 100 Lookup objects
     * @throws SmartyException
     * @throws IOException
     */
    public void send(Batch batch) throws SmartyException, IOException {
        Request request = new Request();

        if (batch.size() == 0)
            throw new SmartyException("Batch must contain between 1 and 100 lookups");

        request.setPayload(this.serializer.serialize(batch.getAllLookups()));

        Response response = this.sender.send(request);

        Candidate[] candidates = this.serializer.deserialize(response.getPayload(), Candidate[].class);
        if (candidates == null)
            candidates = new Candidate[0];
        this.assignCandidatesToLookups(batch, candidates);
    }

    private void assignCandidatesToLookups(Batch batch, Candidate[] candidates) {
        for (Candidate candidate : candidates)
            batch.get(candidate.getInputIndex()).addToResult(candidate);
    }
}
