package com.krux.hyperion.objects

import aws.{AdpJsonSerializer, AdpShellCommandActivity, AdpRef,
  AdpDataNode, AdpActivity, AdpEc2Resource}
import com.krux.hyperion.HyperionContext

trait GoogleStorageActivity extends PipelineActivity

/**
 * Google Storage Download activity
 */
case class GoogleStorageDownloadActivity(
    id: String,
    runsOn: Ec2Resource,
    input: String = "",
    output: Option[S3DataNode] = None,
    botoConfigUrl: String = "",
    dependsOn: Seq[PipelineActivity] = Seq()
  )(
    implicit val hc: HyperionContext
  ) extends GoogleStorageActivity {
  def dependsOn(activities: PipelineActivity*) = this.copy(dependsOn = activities)
  def forClient(client: String) = this.copy(id = s"${id}_${client}")

  def withBotoConfigUrl(url: String) = this.copy(botoConfigUrl = url)
  def withInput(path: String) = this.copy(input = path)
  def withOutput(out: S3DataNode) = this.copy(output = Some(out))

  override def objects: Iterable[PipelineObject] = Seq(runsOn) ++ output ++ dependsOn

  def serialize = AdpShellCommandActivity(
      id,
      Some(id),
      None,
      Some(s"${hc.scriptUri}gsutil/gsutil_download.sh"),
      Some(Seq(botoConfigUrl, input)),
      None,
      output.map(out => AdpRef[AdpDataNode](out.id)),
      "true",
      None,
      None,
      dependsOn match {
        case Seq() => None
        case deps => Some(deps.map(act => AdpRef[AdpActivity](act.id)))
      },
      AdpRef[AdpEc2Resource](runsOn.id)
    )
}

/**
 * Google Storage Upload activity
 */
case class GoogleStorageUploadActivity(
    id: String,
    runsOn: Ec2Resource,
    input: Option[S3DataNode] = None,
    output: String = "",
    botoConfigUrl: String = "",
    dependsOn: Seq[PipelineActivity] = Seq()
  )(
    implicit val hc: HyperionContext
  ) extends GoogleStorageActivity {
  def dependsOn(activities: PipelineActivity*) = this.copy(dependsOn = activities)
  def forClient(client: String) = this.copy(id = s"${id}_${client}")

  def withBotoConfigUrl(url: String) = this.copy(botoConfigUrl = url)
  def withInput(in: S3DataNode) = this.copy(input = Some(in))
  def withOutput(path: String) = this.copy(output = path)

  override def objects: Iterable[PipelineObject] = Seq(runsOn) ++ input ++ dependsOn

  def serialize = AdpShellCommandActivity(
      id,
      Some(id),
      None,
      Some(s"${hc.scriptUri}gsutil/gsutil_upload.sh"),
      Some(Seq(botoConfigUrl, output)),
      input.map(in => AdpRef[AdpDataNode](in.id)),
      None,
      "true",
      None,
      None,
      dependsOn match {
        case Seq() => None
        case deps => Some(deps.map(act => AdpRef[AdpActivity](act.id)))
      },
      AdpRef[AdpEc2Resource](runsOn.id)
    )
}
