/*
 * Copyright (c) 2008-2016, Hazelcast, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hazelcast.query;

import com.hazelcast.nio.ObjectDataInput;
import com.hazelcast.nio.ObjectDataOutput;
import com.hazelcast.nio.serialization.IdentifiedDataSerializable;
import com.hazelcast.query.impl.predicates.PredicateDataSerializerHook;

import java.io.IOException;
import java.util.Map;

import static com.hazelcast.util.Preconditions.checkNotNull;

/**
 * A {@link Predicate} that restricts the execution of a Predicate to a single Partition.
 *
 * This can help to speed up query execution since only a single instead of all partitions needs to be queried.
 *
 * This predicate can only be used as an outer predicate.
 *
 * @param <K> type of the entry key
 * @param <V> type of the entry value
 */
public class PartitionPredicate<K, V> implements Predicate<K, V>, IdentifiedDataSerializable {

    private Object partitionKey;
    private Predicate<K, V> target;

    // should only be used for deserialization
    public PartitionPredicate() {
    }

    /**
     * Creates a new PartitionPredicate.
     *
     * @param partitionKey the partition key
     * @param target       the target {@link Predicate}
     * @throws IllegalArgumentException if partitionId smaller than zero
     * @throws NullPointerException     if target Predicate is {@code null}
     */
    public PartitionPredicate(Object partitionKey, Predicate<K, V> target) {
        this.partitionKey = checkNotNull(partitionKey, "partitionKey can't be null");
        this.target = checkNotNull(target, "target predicate can't be null");
    }

    /**
     * Returns the partition key that determines the partition the target {@link Predicate} is going to execute on.
     *
     * @return the partition id
     */
    public Object getPartitionKey() {
        return partitionKey;
    }

    /**
     * Returns the target {@link Predicate}.
     *
     * @return the target {@link Predicate}.
     */
    public Predicate<K, V> getTarget() {
        return target;
    }

    @Override
    public boolean apply(Map.Entry<K, V> mapEntry) {
        throw new UnsupportedOperationException();
    }

    @Override
    public int getFactoryId() {
        return PredicateDataSerializerHook.F_ID;
    }

    @Override
    public int getId() {
        return PredicateDataSerializerHook.PARTITION_PREDICATE;
    }

    @Override
    public void writeData(ObjectDataOutput out) throws IOException {
        out.writeObject(partitionKey);
        out.writeObject(target);
    }

    @Override
    public void readData(ObjectDataInput in) throws IOException {
        this.partitionKey = in.readObject();
        this.target = in.readObject();
    }

    @Override
    public String toString() {
        return "PartitionPredicate{"
                + "partitionKey=" + partitionKey
                + ", target=" + target
                + '}';
    }
}
