/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.gcsio;

import com.google.cloud.ReadChannel;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageExceptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageItemInfo;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageOptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageReadOptions;
import com.google.cloud.hadoop.gcsio.StorageResourceId;
import com.google.cloud.hadoop.util.ErrorTypeExtractor;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.Storage;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.flogger.GoogleLogger;
import java.io.ByteArrayInputStream;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import java.util.ArrayList;

@VisibleForTesting
class GoogleCloudStorageClientReadChannel
implements SeekableByteChannel {
    private static final GoogleLogger logger = GoogleLogger.forEnclosingClass();
    private final StorageResourceId resourceId;
    private final GoogleCloudStorageReadOptions readOptions;
    private final GoogleCloudStorageOptions storageOptions;
    private final Storage storage;
    private final long objectSize;
    private final ErrorTypeExtractor errorExtractor;
    private ContentReadChannel contentReadChannel;
    private boolean open = true;
    private long currentPosition = 0L;

    public GoogleCloudStorageClientReadChannel(Storage storage, GoogleCloudStorageItemInfo itemInfo, GoogleCloudStorageReadOptions readOptions, ErrorTypeExtractor errorExtractor, GoogleCloudStorageOptions storageOptions) throws IOException {
        GoogleCloudStorageClientReadChannel.validate(itemInfo);
        this.storage = storage;
        this.errorExtractor = errorExtractor;
        this.resourceId = new StorageResourceId(itemInfo.getBucketName(), itemInfo.getObjectName(), itemInfo.getContentGeneration());
        this.readOptions = readOptions;
        this.storageOptions = storageOptions;
        this.objectSize = itemInfo.getSize();
        this.contentReadChannel = new ContentReadChannel(readOptions, this.resourceId);
    }

    @Override
    public int read(ByteBuffer dst) throws IOException {
        this.throwIfNotOpen();
        if (dst.remaining() == 0) {
            return 0;
        }
        ((GoogleLogger.Api)logger.atFiner()).log("Reading %d bytes at %d position from '%s'", (Object)dst.remaining(), (Object)this.currentPosition, (Object)this.resourceId);
        if (this.currentPosition == this.objectSize) {
            return -1;
        }
        return this.contentReadChannel.readContent(dst);
    }

    @Override
    public int write(ByteBuffer src) throws IOException {
        throw new UnsupportedOperationException("Cannot mutate read-only channel");
    }

    @Override
    public long position() throws IOException {
        return this.currentPosition;
    }

    @Override
    public SeekableByteChannel position(long newPosition) throws IOException {
        this.throwIfNotOpen();
        if (newPosition == this.currentPosition) {
            return this;
        }
        this.validatePosition(newPosition);
        ((GoogleLogger.Api)logger.atFiner()).log("Seek from %s to %s position for '%s'", (Object)this.currentPosition, (Object)newPosition, (Object)this.resourceId);
        this.currentPosition = newPosition;
        return this;
    }

    @Override
    public long size() throws IOException {
        return this.objectSize;
    }

    @Override
    public SeekableByteChannel truncate(long size) throws IOException {
        throw new UnsupportedOperationException("Cannot mutate read-only channel");
    }

    @Override
    public boolean isOpen() {
        return this.open;
    }

    @Override
    public void close() throws IOException {
        if (this.open) {
            try {
                ((GoogleLogger.Api)logger.atFiner()).log("Closing channel for '%s'", (Object)this.resourceId);
                this.contentReadChannel.closeContentChannel();
            }
            catch (Exception e) {
                throw new IOException(String.format("Exception occurred while closing channel '%s'", this.resourceId), e);
            }
            finally {
                this.contentReadChannel = null;
                this.open = false;
            }
        }
    }

    @VisibleForTesting
    boolean randomAccessStatus() {
        return this.contentReadChannel.randomAccess;
    }

    private static void validate(GoogleCloudStorageItemInfo itemInfo) throws IOException {
        Preconditions.checkNotNull((Object)itemInfo, (Object)"itemInfo cannot be null");
        StorageResourceId resourceId = itemInfo.getResourceId();
        Preconditions.checkArgument((boolean)resourceId.isStorageObject(), (String)"Can not open a non-file object for read: %s", (Object)resourceId);
        if (!itemInfo.exists()) {
            throw new FileNotFoundException(String.format("Item not found: %s", resourceId));
        }
        String contentEncoding = itemInfo.getContentEncoding();
        if (contentEncoding != null && contentEncoding.contains("gzip")) {
            throw new IOException(String.format("Cannot read GZIP-encoded file (%s) (not supported via gRPC API): %s", contentEncoding, resourceId));
        }
    }

    private IOException convertError(Exception error) {
        String msg = String.format("Error reading '%s'", this.resourceId);
        switch (this.errorExtractor.getErrorType(error)) {
            case NOT_FOUND: {
                return GoogleCloudStorageExceptions.createFileNotFoundException(this.resourceId.getBucketName(), this.resourceId.getObjectName(), new IOException(msg, error));
            }
            case OUT_OF_RANGE: {
                return (IOException)new EOFException(msg).initCause(error);
            }
        }
        return new IOException(msg, error);
    }

    private void validatePosition(long position) throws IOException {
        if (position < 0L) {
            throw new EOFException(String.format("Invalid seek offset: position value (%d) must be >= 0 for '%s'", position, this.resourceId));
        }
        if (this.objectSize >= 0L && position >= this.objectSize) {
            throw new EOFException(String.format("Invalid seek offset: position value (%d) must be between 0 and %d for '%s'", position, this.objectSize, this.resourceId));
        }
    }

    private void throwIfNotOpen() throws IOException {
        if (!this.isOpen()) {
            throw new ClosedChannelException();
        }
    }

    private class ContentReadChannel {
        private static final int SKIP_BUFFER_SIZE = 8192;
        private final BlobId blobId;
        private long contentChannelCurrentPosition = -1L;
        private long contentChannelEnd = -1L;
        private byte[] footerContent;
        private byte[] skipBuffer = null;
        private ReadableByteChannel byteChannel = null;
        private boolean randomAccess;

        public ContentReadChannel(GoogleCloudStorageReadOptions readOptions, StorageResourceId resourceId) {
            this.blobId = BlobId.of((String)resourceId.getBucketName(), (String)resourceId.getObjectName(), (Long)resourceId.getGenerationId());
            this.randomAccess = readOptions.getFadvise() == GoogleCloudStorageReadOptions.Fadvise.RANDOM;
        }

        public int readContent(ByteBuffer dst) throws IOException {
            this.performPendingSeeks();
            Preconditions.checkState((this.contentChannelCurrentPosition == GoogleCloudStorageClientReadChannel.this.currentPosition || this.byteChannel == null ? 1 : 0) != 0, (String)"contentChannelCurrentPosition (%s) should be equal to currentPosition (%s) after lazy seek, if channel is open", (long)this.contentChannelCurrentPosition, (long)GoogleCloudStorageClientReadChannel.this.currentPosition);
            int totalBytesRead = 0;
            while (dst.hasRemaining()) {
                int remainingBeforeRead = dst.remaining();
                try {
                    int bytesRead;
                    if (this.byteChannel == null) {
                        this.byteChannel = this.openByteChannel(dst.remaining());
                    }
                    if ((bytesRead = this.byteChannel.read(dst)) == 0) {
                        throw new IOException(String.format("Read 0 bytes without blocking from object: '%s'", GoogleCloudStorageClientReadChannel.this.resourceId));
                    }
                    if (bytesRead < 0) {
                        if (GoogleCloudStorageClientReadChannel.this.currentPosition != this.contentChannelEnd && GoogleCloudStorageClientReadChannel.this.currentPosition != GoogleCloudStorageClientReadChannel.this.objectSize) {
                            throw new IOException(String.format("Received end of stream result before all requestedBytes were received;EndOf stream signal received at offset: %d where as stream was suppose to end at: %d for resource: %s of size: %d", GoogleCloudStorageClientReadChannel.this.currentPosition, this.contentChannelEnd, GoogleCloudStorageClientReadChannel.this.resourceId, GoogleCloudStorageClientReadChannel.this.objectSize));
                        }
                        if (this.contentChannelEnd == GoogleCloudStorageClientReadChannel.this.objectSize || GoogleCloudStorageClientReadChannel.this.currentPosition != this.contentChannelEnd) break;
                        this.closeContentChannel();
                        continue;
                    }
                    totalBytesRead += bytesRead;
                    GoogleCloudStorageClientReadChannel.this.currentPosition = GoogleCloudStorageClientReadChannel.this.currentPosition + (long)bytesRead;
                    this.contentChannelCurrentPosition += (long)bytesRead;
                    Preconditions.checkState((this.contentChannelCurrentPosition == GoogleCloudStorageClientReadChannel.this.currentPosition ? 1 : 0) != 0, (String)"contentChannelPosition (%s) should be equal to currentPosition (%s) after successful read", (long)this.contentChannelCurrentPosition, (long)GoogleCloudStorageClientReadChannel.this.currentPosition);
                }
                catch (Exception e) {
                    int partialBytes = this.partiallyReadBytes(remainingBeforeRead, dst);
                    totalBytesRead += partialBytes;
                    GoogleCloudStorageClientReadChannel.this.currentPosition = GoogleCloudStorageClientReadChannel.this.currentPosition + (long)partialBytes;
                    ((GoogleLogger.Api)logger.atFine()).log("Closing contentChannel after %s exception for '%s'.", (Object)e.getMessage(), (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
                    this.closeContentChannel();
                    throw GoogleCloudStorageClientReadChannel.this.convertError(e);
                }
            }
            return totalBytesRead;
        }

        private int partiallyReadBytes(int remainingBeforeRead, ByteBuffer dst) {
            int partialReadBytes = 0;
            if (remainingBeforeRead != dst.remaining()) {
                partialReadBytes = remainingBeforeRead - dst.remaining();
            }
            return partialReadBytes;
        }

        private boolean shouldDetectRandomAccess() {
            return !this.randomAccess && GoogleCloudStorageClientReadChannel.this.readOptions.getFadvise() == GoogleCloudStorageReadOptions.Fadvise.AUTO;
        }

        private void setRandomAccess() {
            this.randomAccess = true;
        }

        private ReadableByteChannel openByteChannel(long bytesToRead) throws IOException {
            Preconditions.checkArgument((bytesToRead > 0L ? 1 : 0) != 0, (String)"bytesToRead should be greater than 0, but was %s", (long)bytesToRead);
            Preconditions.checkState((this.byteChannel == null && this.contentChannelEnd < 0L ? 1 : 0) != 0, (String)"contentChannel and contentChannelEnd should be not initialized yet for '%s'", (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
            if (this.footerContent != null && GoogleCloudStorageClientReadChannel.this.currentPosition >= GoogleCloudStorageClientReadChannel.this.objectSize - (long)this.footerContent.length) {
                return this.serveFooterContent();
            }
            this.contentChannelCurrentPosition = this.getRangeRequestStart();
            this.contentChannelEnd = this.getRangeRequestEnd(this.contentChannelCurrentPosition, bytesToRead);
            Preconditions.checkState((this.contentChannelEnd >= this.contentChannelCurrentPosition ? 1 : 0) != 0, (Object)String.format("Start position should be <= endPosition startPosition:%d, endPosition: %d", this.contentChannelCurrentPosition, this.contentChannelEnd));
            ReadableByteChannel readableByteChannel = this.getStorageReadChannel(this.contentChannelCurrentPosition, this.contentChannelEnd);
            if (this.contentChannelEnd == GoogleCloudStorageClientReadChannel.this.objectSize && this.contentChannelEnd - this.contentChannelCurrentPosition <= (long)GoogleCloudStorageClientReadChannel.this.readOptions.getMinRangeRequestSize()) {
                if (this.footerContent == null) {
                    this.cacheFooter(readableByteChannel);
                }
                return this.serveFooterContent();
            }
            Preconditions.checkState((this.contentChannelCurrentPosition == GoogleCloudStorageClientReadChannel.this.currentPosition ? 1 : 0) != 0, (Object)"position of read offset isn't in alignment with channel's read offset");
            return readableByteChannel;
        }

        private void cacheFooter(ReadableByteChannel readableByteChannel) throws IOException {
            int footerSize = Math.toIntExact(GoogleCloudStorageClientReadChannel.this.objectSize - this.contentChannelCurrentPosition);
            this.footerContent = new byte[footerSize];
            try (InputStream footerStream = Channels.newInputStream(readableByteChannel);){
                int bytesRead;
                int totalBytesRead = 0;
                do {
                    if ((bytesRead = footerStream.read(this.footerContent, totalBytesRead, footerSize - totalBytesRead)) < 0) continue;
                    totalBytesRead += bytesRead;
                } while (bytesRead >= 0 && totalBytesRead < footerSize);
                Preconditions.checkState((bytesRead >= 0 ? 1 : 0) != 0, (String)"footerStream shouldn't be empty before reading the footer of size %s, totalBytesRead %s, read via last call %s, for '%s'", (Object)footerSize, (Object)totalBytesRead, (Object)bytesRead, (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
                Preconditions.checkState((totalBytesRead == footerSize ? 1 : 0) != 0, (String)"totalBytesRead (%s) should equal footerSize (%s) for '%s'", (Object)totalBytesRead, (Object)footerSize, (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
            }
            catch (Exception e) {
                this.footerContent = null;
                throw e;
            }
            ((GoogleLogger.Api)logger.atFiner()).log("Prefetched %s bytes footer for '%s'", this.footerContent.length, (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
        }

        private ReadableByteChannel serveFooterContent() {
            this.contentChannelCurrentPosition = GoogleCloudStorageClientReadChannel.this.currentPosition;
            int offset = Math.toIntExact(GoogleCloudStorageClientReadChannel.this.currentPosition - (GoogleCloudStorageClientReadChannel.this.objectSize - (long)this.footerContent.length));
            int length = this.footerContent.length - offset;
            ((GoogleLogger.Api)logger.atFiner()).log("Opened channel (prefetched footer) from %d position for '%s'", GoogleCloudStorageClientReadChannel.this.currentPosition, (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
            return Channels.newChannel(new ByteArrayInputStream(this.footerContent, offset, length));
        }

        private long getRangeRequestStart() {
            if (GoogleCloudStorageClientReadChannel.this.readOptions.getFadvise() != GoogleCloudStorageReadOptions.Fadvise.SEQUENTIAL && this.isFooterRead()) {
                return Math.max(0L, GoogleCloudStorageClientReadChannel.this.objectSize - (long)GoogleCloudStorageClientReadChannel.this.readOptions.getMinRangeRequestSize());
            }
            return GoogleCloudStorageClientReadChannel.this.currentPosition;
        }

        public void closeContentChannel() {
            if (this.byteChannel != null) {
                ((GoogleLogger.Api)logger.atFiner()).log("Closing internal contentChannel for '%s'", (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
                try {
                    this.byteChannel.close();
                }
                catch (Exception e) {
                    ((GoogleLogger.Api)((GoogleLogger.Api)logger.atFine()).withCause((Throwable)e)).log("Got an exception on contentChannel.close() for '%s'; ignoring it.", (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
                }
                finally {
                    this.byteChannel = null;
                    this.reset();
                }
            }
        }

        private void reset() {
            Preconditions.checkState((this.byteChannel == null ? 1 : 0) != 0, (String)"contentChannel should be null for '%s'", (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
            this.contentChannelCurrentPosition = -1L;
            this.contentChannelEnd = -1L;
        }

        private boolean isInRangeSeek() {
            long seekDistance = GoogleCloudStorageClientReadChannel.this.currentPosition - this.contentChannelCurrentPosition;
            return this.byteChannel != null && seekDistance > 0L && seekDistance <= GoogleCloudStorageClientReadChannel.this.readOptions.getInplaceSeekLimit() && GoogleCloudStorageClientReadChannel.this.currentPosition < this.contentChannelEnd;
        }

        private void skipInPlace() {
            if (this.skipBuffer == null) {
                this.skipBuffer = new byte[8192];
            }
            long seekDistance = GoogleCloudStorageClientReadChannel.this.currentPosition - this.contentChannelCurrentPosition;
            while (seekDistance > 0L && this.byteChannel != null) {
                try {
                    int bufferSize = Math.toIntExact(Math.min((long)this.skipBuffer.length, seekDistance));
                    int bytesRead = this.byteChannel.read(ByteBuffer.wrap(this.skipBuffer, 0, bufferSize));
                    if (bytesRead < 0) {
                        ((GoogleLogger.Api)logger.atInfo()).log("Somehow read %d bytes trying to skip %d bytes to seek to position %d, size: %d", (Object)bytesRead, (Object)seekDistance, (Object)GoogleCloudStorageClientReadChannel.this.currentPosition, (Object)GoogleCloudStorageClientReadChannel.this.objectSize);
                        this.closeContentChannel();
                        continue;
                    }
                    seekDistance -= (long)bytesRead;
                    this.contentChannelCurrentPosition += (long)bytesRead;
                }
                catch (Exception e) {
                    ((GoogleLogger.Api)((GoogleLogger.Api)logger.atInfo()).withCause((Throwable)e)).log("Got an IO exception on contentChannel.read(), a lazy-seek will be pending for '%s'", (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
                    this.closeContentChannel();
                }
            }
            Preconditions.checkState((this.byteChannel == null || this.contentChannelCurrentPosition == GoogleCloudStorageClientReadChannel.this.currentPosition ? 1 : 0) != 0, (String)"contentChannelPosition (%s) should be equal to currentPosition (%s) after successful in-place skip", (long)this.contentChannelCurrentPosition, (long)GoogleCloudStorageClientReadChannel.this.currentPosition);
        }

        private void performPendingSeeks() {
            if (GoogleCloudStorageClientReadChannel.this.currentPosition == this.contentChannelCurrentPosition && this.byteChannel != null) {
                return;
            }
            ((GoogleLogger.Api)logger.atFiner()).log("Performing lazySeek from %s to %s position '%s'", (Object)this.contentChannelCurrentPosition, (Object)GoogleCloudStorageClientReadChannel.this.currentPosition, (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
            if (this.isInRangeSeek()) {
                this.skipInPlace();
            } else {
                if (this.isRandomAccessPattern()) {
                    this.setRandomAccess();
                }
                this.closeContentChannel();
            }
        }

        private boolean isRandomAccessPattern() {
            if (!this.shouldDetectRandomAccess()) {
                return false;
            }
            if (GoogleCloudStorageClientReadChannel.this.currentPosition < this.contentChannelCurrentPosition) {
                ((GoogleLogger.Api)logger.atFine()).log("Detected backward read from %s to %s position, switching to random IO for '%s'", (Object)this.contentChannelCurrentPosition, (Object)GoogleCloudStorageClientReadChannel.this.currentPosition, (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
                return true;
            }
            if (this.contentChannelCurrentPosition >= 0L && this.contentChannelCurrentPosition + GoogleCloudStorageClientReadChannel.this.readOptions.getInplaceSeekLimit() < GoogleCloudStorageClientReadChannel.this.currentPosition) {
                ((GoogleLogger.Api)logger.atFine()).log("Detected forward read from %s to %s position over %s threshold, switching to random IO for '%s'", (Object)this.contentChannelCurrentPosition, (Object)GoogleCloudStorageClientReadChannel.this.currentPosition, (Object)GoogleCloudStorageClientReadChannel.this.readOptions.getInplaceSeekLimit(), (Object)GoogleCloudStorageClientReadChannel.this.resourceId);
                return true;
            }
            return false;
        }

        private ReadableByteChannel getStorageReadChannel(long seek, long limit) throws IOException {
            ReadChannel readChannel = GoogleCloudStorageClientReadChannel.this.storage.reader(this.blobId, this.generateReadOptions(this.blobId));
            try {
                readChannel.seek(seek);
                readChannel.limit(limit);
                return readChannel;
            }
            catch (Exception e) {
                throw new IOException(String.format("Unable to update the boundaries/Range of contentChannel %s", GoogleCloudStorageClientReadChannel.this.resourceId.toString()), e);
            }
        }

        private Storage.BlobSourceOption[] generateReadOptions(BlobId blobId) {
            ArrayList<Storage.BlobSourceOption> blobReadOptions = new ArrayList<Storage.BlobSourceOption>();
            if (blobId.getGeneration() != null) {
                blobReadOptions.add(Storage.BlobSourceOption.generationMatch((long)blobId.getGeneration()));
            }
            if (GoogleCloudStorageClientReadChannel.this.storageOptions.getEncryptionKey() != null) {
                blobReadOptions.add(Storage.BlobSourceOption.decryptionKey((String)GoogleCloudStorageClientReadChannel.this.storageOptions.getEncryptionKey().value()));
            }
            return blobReadOptions.toArray(new Storage.BlobSourceOption[blobReadOptions.size()]);
        }

        private boolean isFooterRead() {
            return GoogleCloudStorageClientReadChannel.this.objectSize - GoogleCloudStorageClientReadChannel.this.currentPosition <= (long)GoogleCloudStorageClientReadChannel.this.readOptions.getMinRangeRequestSize();
        }

        private long getRangeRequestEnd(long startPosition, long bytesToRead) {
            long endPosition = GoogleCloudStorageClientReadChannel.this.objectSize;
            if (this.randomAccess) {
                endPosition = startPosition + Math.max(bytesToRead, (long)GoogleCloudStorageClientReadChannel.this.readOptions.getMinRangeRequestSize());
            }
            if (this.footerContent != null) {
                endPosition = Math.min(endPosition, GoogleCloudStorageClientReadChannel.this.objectSize - (long)this.footerContent.length);
            }
            return endPosition;
        }
    }
}

