/*
 * Decompiled with CFR 0.152.
 */
package com.google.appengine.tools.cloudstorage.oauth;

import com.google.appengine.api.urlfetch.FetchOptions;
import com.google.appengine.api.urlfetch.HTTPHeader;
import com.google.appengine.api.urlfetch.HTTPMethod;
import com.google.appengine.api.urlfetch.HTTPRequest;
import com.google.appengine.api.urlfetch.HTTPResponse;
import com.google.appengine.api.utils.FutureWrapper;
import com.google.appengine.tools.cloudstorage.BadRangeException;
import com.google.appengine.tools.cloudstorage.GcsFileMetadata;
import com.google.appengine.tools.cloudstorage.GcsFileOptions;
import com.google.appengine.tools.cloudstorage.GcsFilename;
import com.google.appengine.tools.cloudstorage.RawGcsService;
import com.google.appengine.tools.cloudstorage.oauth.OAuthURLFetchService;
import com.google.appengine.tools.cloudstorage.oauth.URLFetchUtils;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.logging.Logger;

final class OauthRawGcsService
implements RawGcsService {
    private static final String ACL = "x-goog-acl";
    private static final String CACHE_CONTROL = "Cache-Control";
    private static final String CONTENT_ENCODING = "Content-Encoding";
    private static final String CONTENT_DISPOSITION = "Content-Disposition";
    private static final String CONTENT_TYPE = "Content-Type";
    private static final String ETAG = "ETag";
    private final HTTPHeader versionHeader = new HTTPHeader("x-goog-api-version", "2");
    private static final Logger log = Logger.getLogger(OauthRawGcsService.class.getName());
    public static final List<String> OAUTH_SCOPES = ImmutableList.of((Object)"https://www.googleapis.com/auth/devstorage.read_write");
    private static final int READ_LIMIT_BYTES = 0x800000;
    static final int CHUNK_ALIGNMENT_BYTES = 262144;
    private final OAuthURLFetchService urlfetch;

    OauthRawGcsService(OAuthURLFetchService urlfetch) {
        this.urlfetch = (OAuthURLFetchService)Preconditions.checkNotNull((Object)urlfetch, (Object)"Null urlfetch");
    }

    public String toString() {
        return this.getClass().getSimpleName() + "(" + this.urlfetch + ")";
    }

    private static URL makeUrl(GcsFilename filename, String uploadId) {
        String encodedFileName;
        try {
            encodedFileName = URLEncoder.encode(filename.getObjectName(), "UTF-8");
        }
        catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
        String s = "https://storage.googleapis.com/" + filename.getBucketName() + "/" + encodedFileName + (uploadId == null ? "" : "?upload_id=" + uploadId);
        try {
            return new URL(s);
        }
        catch (MalformedURLException e) {
            throw new RuntimeException("Internal error: " + s, e);
        }
    }

    private static HTTPRequest makeRequest(GcsFilename filename, String uploadId, HTTPMethod method, long timeoutMillis) {
        return new HTTPRequest(OauthRawGcsService.makeUrl(filename, uploadId), method, FetchOptions.Builder.disallowTruncate().doNotFollowRedirects().validateCertificate().setDeadline(Double.valueOf((double)timeoutMillis / 1000.0)));
    }

    private static Error handleError(HTTPRequest req, HTTPResponse resp) throws IOException {
        int responseCode = resp.getResponseCode();
        switch (responseCode) {
            case 400: {
                throw new RuntimeException("Server replied with 400, probably bad request: " + URLFetchUtils.describeRequestAndResponse(req, resp, true));
            }
            case 401: {
                throw new RuntimeException("Server replied with 401, probably bad authentication: " + URLFetchUtils.describeRequestAndResponse(req, resp, true));
            }
            case 403: {
                throw new RuntimeException("Server replied with 403, check that ACLs are set correctly on the object and bucket: " + URLFetchUtils.describeRequestAndResponse(req, resp, true));
            }
        }
        if (responseCode >= 500 && responseCode < 600) {
            throw new IOException("Response code " + resp.getResponseCode() + ", retryable: " + URLFetchUtils.describeRequestAndResponse(req, resp, true));
        }
        throw new RuntimeException("Unexpected response code " + resp.getResponseCode() + ": " + URLFetchUtils.describeRequestAndResponse(req, resp, true));
    }

    @Override
    public RawGcsService.RawGcsCreationToken beginObjectCreation(GcsFilename filename, GcsFileOptions options, long timeoutMillis) throws IOException {
        HTTPResponse resp;
        HTTPRequest req = OauthRawGcsService.makeRequest(filename, null, HTTPMethod.POST, timeoutMillis);
        req.setHeader(new HTTPHeader("x-goog-resumable", "start"));
        req.setHeader(this.versionHeader);
        this.addOptionsHeaders(req, options);
        try {
            resp = this.urlfetch.fetch(req);
        }
        catch (IOException e) {
            throw new IOException("URLFetch threw IOException; request: " + URLFetchUtils.describeRequest(req), e);
        }
        if (resp.getResponseCode() == 201) {
            String location = URLFetchUtils.getSingleHeader(resp, "location");
            String marker = "?upload_id=";
            Preconditions.checkState((boolean)location.contains(marker), (String)"bad location without upload_id: %s", (Object[])new Object[]{location});
            Preconditions.checkState((!location.contains("&") ? 1 : 0) != 0, (String)"bad location with &: %s", (Object[])new Object[]{location});
            String uploadId = location.substring(location.indexOf(marker) + marker.length());
            return new GcsRestCreationToken(filename, uploadId, 0L);
        }
        throw OauthRawGcsService.handleError(req, resp);
    }

    private void addOptionsHeaders(HTTPRequest req, GcsFileOptions options) {
        if (options == null) {
            return;
        }
        if (options.getMimeType() != null) {
            req.setHeader(new HTTPHeader(CONTENT_TYPE, options.getMimeType()));
        }
        if (options.getAcl() != null) {
            req.setHeader(new HTTPHeader(ACL, options.getAcl()));
        }
        if (options.getCacheControl() != null) {
            req.setHeader(new HTTPHeader(CACHE_CONTROL, options.getCacheControl()));
        }
        if (options.getContentDisposition() != null) {
            req.setHeader(new HTTPHeader(CONTENT_DISPOSITION, options.getContentDisposition()));
        }
        if (options.getContentEncoding() != null) {
            req.setHeader(new HTTPHeader(CONTENT_ENCODING, options.getContentEncoding()));
        }
        for (Map.Entry<String, String> entry : options.getUserMetadata().entrySet()) {
            req.setHeader(new HTTPHeader("x-goog-meta-" + entry.getKey(), entry.getValue()));
        }
    }

    @Override
    public RawGcsService.RawGcsCreationToken continueObjectCreation(RawGcsService.RawGcsCreationToken x, ByteBuffer chunk, long timeoutMillis) throws IOException {
        GcsRestCreationToken token = (GcsRestCreationToken)x;
        int length = chunk.remaining();
        this.put(token, chunk, false, timeoutMillis);
        return new GcsRestCreationToken(token.filename, token.uploadId, token.offset + (long)length);
    }

    @Override
    public void finishObjectCreation(RawGcsService.RawGcsCreationToken x, ByteBuffer chunk, long timeoutMillis) throws IOException {
        GcsRestCreationToken token = (GcsRestCreationToken)x;
        this.put(token, chunk, true, timeoutMillis);
    }

    private void put(GcsRestCreationToken token, ByteBuffer chunk, boolean isFinalChunk, long timeoutMillis) throws IOException {
        HTTPResponse resp;
        int length = chunk.remaining();
        long offset = token.offset;
        Preconditions.checkArgument((offset % 262144L == 0L ? 1 : 0) != 0, (String)"%s: Offset not aligned; offset=%s, length=%s, token=%s", (Object[])new Object[]{this, offset, length, token});
        Preconditions.checkArgument((isFinalChunk || length % 262144 == 0 ? 1 : 0) != 0, (String)"%s: Chunk not final and not aligned: offset=%s, length=%s, token=%s", (Object[])new Object[]{this, offset, length, token});
        Preconditions.checkArgument((isFinalChunk || length > 0 ? 1 : 0) != 0, (String)"%s: Chunk empty and not final: offset=%s, length=%s, token=%s", (Object[])new Object[]{this, offset, length, token});
        if (log.isLoggable(Level.FINEST)) {
            log.finest(this + ": About to write to " + token + " " + String.format("0x%x", length) + " bytes at offset " + String.format("0x%x", offset) + "; isFinalChunk: " + isFinalChunk + ")");
        }
        long limit = offset + (long)length;
        HTTPRequest req = OauthRawGcsService.makeRequest(token.filename, token.uploadId, HTTPMethod.PUT, timeoutMillis);
        req.setHeader(this.versionHeader);
        req.setHeader(new HTTPHeader("Content-Range", "bytes " + (length == 0 ? "*" : offset + "-" + (limit - 1L)) + (isFinalChunk ? "/" + limit : "/*")));
        req.setPayload(OauthRawGcsService.peekBytes(chunk));
        try {
            resp = this.urlfetch.fetch(req);
        }
        catch (IOException e) {
            throw new IOException("URLFetch threw IOException; request: " + URLFetchUtils.describeRequest(req), e);
        }
        switch (resp.getResponseCode()) {
            case 200: {
                if (!isFinalChunk) {
                    throw new RuntimeException("Unexpected response code 200 on non-final chunk: " + URLFetchUtils.describeRequestAndResponse(req, resp, true));
                }
                chunk.position(chunk.limit());
                return;
            }
            case 308: {
                if (isFinalChunk) {
                    throw new RuntimeException("Unexpected response code 308 on final chunk: " + URLFetchUtils.describeRequestAndResponse(req, resp, true));
                }
                chunk.position(chunk.limit());
                return;
            }
        }
        throw OauthRawGcsService.handleError(req, resp);
    }

    private static byte[] peekBytes(ByteBuffer in) {
        if (in.hasArray() && in.position() == 0 && in.arrayOffset() == 0 && in.array().length == in.limit()) {
            return in.array();
        }
        int pos = in.position();
        byte[] buf = new byte[in.remaining()];
        in.get(buf);
        in.position(pos);
        return buf;
    }

    @Override
    public boolean deleteObject(GcsFilename filename, long timeoutMillis) throws IOException {
        HTTPResponse resp;
        HTTPRequest req = OauthRawGcsService.makeRequest(filename, null, HTTPMethod.DELETE, timeoutMillis);
        req.setHeader(this.versionHeader);
        try {
            resp = this.urlfetch.fetch(req);
        }
        catch (IOException e) {
            throw new IOException("URLFetch threw IOException; request: " + URLFetchUtils.describeRequest(req), e);
        }
        switch (resp.getResponseCode()) {
            case 204: {
                return true;
            }
            case 404: {
                return false;
            }
        }
        throw OauthRawGcsService.handleError(req, resp);
    }

    private long getLengthFromContentRange(HTTPResponse resp) {
        String range = URLFetchUtils.getSingleHeader(resp, "Content-Range");
        Preconditions.checkState((boolean)range.matches("bytes [0-9]+-[0-9]+/[0-9]+"), (String)"%s: unexpected Content-Range: %s", (Object[])new Object[]{this, range});
        return Long.parseLong(range.substring(range.indexOf("/") + 1));
    }

    private long getLengthFromContentLength(HTTPResponse resp) {
        return Long.parseLong(URLFetchUtils.getSingleHeader(resp, "Content-Length"));
    }

    @Override
    public Future<GcsFileMetadata> readObjectAsync(final ByteBuffer dst, final GcsFilename filename, long startOffsetBytes, long timeoutMillis) {
        Preconditions.checkArgument((startOffsetBytes >= 0L ? 1 : 0) != 0, (String)"%s: offset must be non-negative: %s", (Object[])new Object[]{this, startOffsetBytes});
        int n = dst.remaining();
        Preconditions.checkArgument((n > 0 ? 1 : 0) != 0, (String)"%s: dst full: %s", (Object[])new Object[]{this, dst});
        final int want = Math.min(0x800000, n);
        final HTTPRequest req = OauthRawGcsService.makeRequest(filename, null, HTTPMethod.GET, timeoutMillis);
        req.setHeader(this.versionHeader);
        req.setHeader(new HTTPHeader("Range", "bytes=" + startOffsetBytes + "-" + (startOffsetBytes + (long)want - 1L)));
        return new FutureWrapper<HTTPResponse, GcsFileMetadata>(this.urlfetch.fetchAsync(req)){

            protected GcsFileMetadata wrap(HTTPResponse resp) throws IOException {
                long totalLength;
                switch (resp.getResponseCode()) {
                    case 200: {
                        totalLength = OauthRawGcsService.this.getLengthFromContentLength(resp);
                        break;
                    }
                    case 206: {
                        totalLength = OauthRawGcsService.this.getLengthFromContentRange(resp);
                        break;
                    }
                    case 416: {
                        throw new BadRangeException("Requested Range not satisfiable; perhaps read past EOF? " + URLFetchUtils.describeRequestAndResponse(req, resp, true));
                    }
                    default: {
                        throw OauthRawGcsService.handleError(req, resp);
                    }
                }
                byte[] content = resp.getContent();
                Preconditions.checkState((content.length <= want ? 1 : 0) != 0, (String)"%s: got %s > wanted %s", (Object[])new Object[]{this, content.length, want});
                dst.put(content);
                return OauthRawGcsService.this.getMetadataFromResponse(filename, resp, totalLength);
            }

            protected Throwable convertException(Throwable e) {
                if (e instanceof IOException || e instanceof BadRangeException) {
                    return e;
                }
                return new IOException("URLFetch threw IOException; request: " + URLFetchUtils.describeRequest(req), e);
            }
        };
    }

    @Override
    public GcsFileMetadata getObjectMetadata(GcsFilename filename, long timeoutMillis) throws IOException {
        HTTPResponse resp;
        HTTPRequest req = OauthRawGcsService.makeRequest(filename, null, HTTPMethod.HEAD, timeoutMillis);
        req.setHeader(this.versionHeader);
        try {
            resp = this.urlfetch.fetch(req);
        }
        catch (IOException e) {
            throw new IOException("URLFetch threw IOException; request: " + URLFetchUtils.describeRequest(req), e);
        }
        int responseCode = resp.getResponseCode();
        if (responseCode == 404) {
            return null;
        }
        if (responseCode != 200) {
            throw OauthRawGcsService.handleError(req, resp);
        }
        return this.getMetadataFromResponse(filename, resp, this.getLengthFromContentLength(resp));
    }

    private GcsFileMetadata getMetadataFromResponse(GcsFilename filename, HTTPResponse resp, long length) {
        List headers = resp.getHeaders();
        GcsFileOptions.Builder optionsBuilder = new GcsFileOptions.Builder();
        String etag = null;
        for (HTTPHeader header : headers) {
            if (header.getName().startsWith("x-goog-meta-")) {
                String key = header.getName().replaceFirst("x-goog-meta-", "");
                String value = header.getValue();
                optionsBuilder.addUserMetadata(key, value);
            }
            if (header.getName().equals(ACL)) {
                optionsBuilder.acl(header.getValue());
            }
            if (header.getName().equals(CACHE_CONTROL)) {
                optionsBuilder.cacheControl(header.getValue());
            }
            if (header.getName().equals(CONTENT_ENCODING)) {
                optionsBuilder.contentEncoding(header.getValue());
            }
            if (header.getName().equals(CONTENT_DISPOSITION)) {
                optionsBuilder.contentDisposition(header.getValue());
            }
            if (header.getName().equals(CONTENT_TYPE)) {
                optionsBuilder.mimeType(header.getValue());
            }
            if (!header.getName().equals(ETAG)) continue;
            etag = header.getValue();
        }
        GcsFileOptions options = optionsBuilder.build();
        return new GcsFileMetadata(filename, options, etag, length);
    }

    @Override
    public int getChunkSizeBytes() {
        return 262144;
    }

    public static class GcsRestCreationToken
    implements RawGcsService.RawGcsCreationToken {
        private static final long serialVersionUID = 975106845036199413L;
        private final GcsFilename filename;
        private final String uploadId;
        private final long offset;

        GcsRestCreationToken(GcsFilename filename, String uploadId, long offset) {
            this.filename = (GcsFilename)Preconditions.checkNotNull((Object)filename, (Object)"Null filename");
            this.uploadId = (String)Preconditions.checkNotNull((Object)uploadId, (Object)"Null uploadId");
            this.offset = offset;
        }

        @Override
        public GcsFilename getFilename() {
            return this.filename;
        }

        public String toString() {
            return this.getClass().getSimpleName() + "(" + this.filename + ", " + this.uploadId + ")";
        }

        @Override
        public long getOffset() {
            return this.offset;
        }
    }
}

