/*
 * Decompiled with CFR 0.152.
 */
package com.github.cassandra.jdbc.internal.cassandra.io.util;

import com.github.cassandra.jdbc.internal.cassandra.concurrent.ScheduledExecutors;
import com.github.cassandra.jdbc.internal.cassandra.config.DatabaseDescriptor;
import com.github.cassandra.jdbc.internal.cassandra.db.BlacklistedDirectories;
import com.github.cassandra.jdbc.internal.cassandra.db.Keyspace;
import com.github.cassandra.jdbc.internal.cassandra.io.FSError;
import com.github.cassandra.jdbc.internal.cassandra.io.FSReadError;
import com.github.cassandra.jdbc.internal.cassandra.io.FSWriteError;
import com.github.cassandra.jdbc.internal.cassandra.io.sstable.CorruptSSTableException;
import com.github.cassandra.jdbc.internal.cassandra.service.StorageService;
import com.github.cassandra.jdbc.internal.cassandra.utils.JVMStabilityInspector;
import com.github.cassandra.jdbc.internal.cassandra.utils.Throwables;
import com.github.cassandra.jdbc.internal.slf4j.Logger;
import com.github.cassandra.jdbc.internal.slf4j.LoggerFactory;
import java.io.Closeable;
import java.io.DataInput;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.text.DecimalFormat;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import sun.nio.ch.DirectBuffer;

public class FileUtils {
    public static final Charset CHARSET = StandardCharsets.UTF_8;
    private static final Logger logger = LoggerFactory.getLogger(FileUtils.class);
    private static final double KB = 1024.0;
    private static final double MB = 1048576.0;
    private static final double GB = 1.073741824E9;
    private static final double TB = 1.099511627776E12;
    private static final DecimalFormat df = new DecimalFormat("#.##");
    private static final boolean canCleanDirectBuffers;

    public static void createHardLink(String from, String to) {
        FileUtils.createHardLink(new File(from), new File(to));
    }

    public static void createHardLink(File from, File to) {
        if (to.exists()) {
            throw new RuntimeException("Tried to create duplicate hard link to " + to);
        }
        if (!from.exists()) {
            throw new RuntimeException("Tried to hard link to file that does not exist " + from);
        }
        try {
            Files.createLink(to.toPath(), from.toPath());
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, to);
        }
    }

    public static File createTempFile(String prefix, String suffix, File directory) {
        try {
            return File.createTempFile(prefix, suffix, directory);
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, directory);
        }
    }

    public static File createTempFile(String prefix, String suffix) {
        return FileUtils.createTempFile(prefix, suffix, new File(System.getProperty("java.io.tmpdir")));
    }

    public static Throwable deleteWithConfirm(String filePath, boolean expect, Throwable accumulate) {
        return FileUtils.deleteWithConfirm(new File(filePath), expect, accumulate);
    }

    public static Throwable deleteWithConfirm(File file, boolean expect, Throwable accumulate) {
        boolean exists = file.exists();
        assert (exists || !expect) : "attempted to delete non-existing file " + file.getName();
        try {
            if (exists) {
                Files.delete(file.toPath());
            }
        }
        catch (Throwable t) {
            try {
                throw new FSWriteError(t, file);
            }
            catch (Throwable t2) {
                accumulate = Throwables.merge(accumulate, t2);
            }
        }
        return accumulate;
    }

    public static void deleteWithConfirm(String file) {
        FileUtils.deleteWithConfirm(new File(file));
    }

    public static void deleteWithConfirm(File file) {
        Throwables.maybeFail(FileUtils.deleteWithConfirm(file, true, null));
    }

    public static void renameWithOutConfirm(String from, String to) {
        block2: {
            try {
                FileUtils.atomicMoveWithFallback(new File(from).toPath(), new File(to).toPath());
            }
            catch (IOException e) {
                if (!logger.isTraceEnabled()) break block2;
                logger.trace("Could not move file " + from + " to " + to, e);
            }
        }
    }

    public static void renameWithConfirm(String from, String to) {
        FileUtils.renameWithConfirm(new File(from), new File(to));
    }

    public static void renameWithConfirm(File from, File to) {
        assert (from.exists());
        if (logger.isTraceEnabled()) {
            logger.trace(String.format("Renaming %s to %s", from.getPath(), to.getPath()));
        }
        try {
            FileUtils.atomicMoveWithFallback(from.toPath(), to.toPath());
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Failed to rename %s to %s", from.getPath(), to.getPath()), e);
        }
    }

    private static void atomicMoveWithFallback(Path from, Path to) throws IOException {
        try {
            Files.move(from, to, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (AtomicMoveNotSupportedException e) {
            logger.trace("Could not do an atomic move", e);
            Files.move(from, to, StandardCopyOption.REPLACE_EXISTING);
        }
    }

    public static void truncate(String path, long size) {
        try (FileChannel channel = FileChannel.open(Paths.get(path, new String[0]), StandardOpenOption.READ, StandardOpenOption.WRITE);){
            channel.truncate(size);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static void closeQuietly(Closeable c) {
        try {
            if (c != null) {
                c.close();
            }
        }
        catch (Exception e) {
            logger.warn("Failed closing {}", (Object)c, (Object)e);
        }
    }

    public static void closeQuietly(AutoCloseable c) {
        try {
            if (c != null) {
                c.close();
            }
        }
        catch (Exception e) {
            logger.warn("Failed closing {}", (Object)c, (Object)e);
        }
    }

    public static void close(Closeable ... cs) throws IOException {
        FileUtils.close(Arrays.asList(cs));
    }

    public static void close(Iterable<? extends Closeable> cs) throws IOException {
        IOException e = null;
        for (Closeable closeable : cs) {
            try {
                if (closeable == null) continue;
                closeable.close();
            }
            catch (IOException ex) {
                e = ex;
                logger.warn("Failed closing stream {}", (Object)closeable, (Object)ex);
            }
        }
        if (e != null) {
            throw e;
        }
    }

    public static void closeQuietly(Iterable<? extends AutoCloseable> cs) {
        for (AutoCloseable autoCloseable : cs) {
            try {
                if (autoCloseable == null) continue;
                autoCloseable.close();
            }
            catch (Exception ex) {
                logger.warn("Failed closing {}", (Object)autoCloseable, (Object)ex);
            }
        }
    }

    public static String getCanonicalPath(String filename) {
        try {
            return new File(filename).getCanonicalPath();
        }
        catch (IOException e) {
            throw new FSReadError((Throwable)e, filename);
        }
    }

    public static String getCanonicalPath(File file) {
        try {
            return file.getCanonicalPath();
        }
        catch (IOException e) {
            throw new FSReadError((Throwable)e, file);
        }
    }

    public static boolean isContained(File folder, File file) {
        String folderPath = FileUtils.getCanonicalPath(folder);
        String filePath = FileUtils.getCanonicalPath(file);
        return filePath.startsWith(folderPath);
    }

    public static String getRelativePath(String basePath, String path) {
        try {
            return Paths.get(basePath, new String[0]).relativize(Paths.get(path, new String[0])).toString();
        }
        catch (Exception ex) {
            String absDataPath = FileUtils.getCanonicalPath(basePath);
            return Paths.get(absDataPath, new String[0]).relativize(Paths.get(path, new String[0])).toString();
        }
    }

    public static boolean isCleanerAvailable() {
        return canCleanDirectBuffers;
    }

    public static void clean(ByteBuffer buffer) {
        DirectBuffer db;
        if (buffer == null) {
            return;
        }
        if (FileUtils.isCleanerAvailable() && buffer.isDirect() && (db = (DirectBuffer)((Object)buffer)).cleaner() != null) {
            db.cleaner().clean();
        }
    }

    public static void createDirectory(String directory) {
        FileUtils.createDirectory(new File(directory));
    }

    public static void createDirectory(File directory) {
        if (!directory.exists() && !directory.mkdirs()) {
            throw new FSWriteError((Throwable)new IOException("Failed to mkdirs " + directory), directory);
        }
    }

    public static boolean delete(String file) {
        File f = new File(file);
        return f.delete();
    }

    public static void delete(File ... files) {
        for (File file : files) {
            file.delete();
        }
    }

    public static void deleteAsync(final String file) {
        Runnable runnable = new Runnable(){

            @Override
            public void run() {
                FileUtils.deleteWithConfirm(new File(file));
            }
        };
        ScheduledExecutors.nonPeriodicTasks.execute(runnable);
    }

    public static String stringifyFileSize(double value) {
        if (value >= 1.099511627776E12) {
            double d = value / 1.099511627776E12;
            String val = df.format(d);
            return val + " TB";
        }
        if (value >= 1.073741824E9) {
            double d = value / 1.073741824E9;
            String val = df.format(d);
            return val + " GB";
        }
        if (value >= 1048576.0) {
            double d = value / 1048576.0;
            String val = df.format(d);
            return val + " MB";
        }
        if (value >= 1024.0) {
            double d = value / 1024.0;
            String val = df.format(d);
            return val + " KB";
        }
        String val = df.format(value);
        return val + " bytes";
    }

    public static void deleteRecursive(File dir) {
        if (dir.isDirectory()) {
            String[] children;
            for (String child : children = dir.list()) {
                FileUtils.deleteRecursive(new File(dir, child));
            }
        }
        FileUtils.deleteWithConfirm(dir);
    }

    public static void deleteRecursiveOnExit(File dir) {
        if (dir.isDirectory()) {
            String[] children;
            for (String child : children = dir.list()) {
                FileUtils.deleteRecursiveOnExit(new File(dir, child));
            }
        }
        logger.trace("Scheduling deferred deletion of file: " + dir);
        dir.deleteOnExit();
    }

    public static void handleCorruptSSTable(CorruptSSTableException e) {
        if (!StorageService.instance.isSetupCompleted()) {
            FileUtils.handleStartupFSError((Throwable)e);
        }
        JVMStabilityInspector.inspectThrowable((Throwable)e);
        switch (DatabaseDescriptor.getDiskFailurePolicy()) {
            case stop_paranoid: {
                StorageService.instance.stopTransports();
            }
        }
    }

    public static void handleFSError(FSError e) {
        if (!StorageService.instance.isSetupCompleted()) {
            FileUtils.handleStartupFSError((Throwable)e);
        }
        JVMStabilityInspector.inspectThrowable((Throwable)e);
        switch (DatabaseDescriptor.getDiskFailurePolicy()) {
            case stop_paranoid: 
            case stop: {
                StorageService.instance.stopTransports();
                break;
            }
            case best_effort: {
                File directory;
                BlacklistedDirectories.maybeMarkUnwritable((File)e.path);
                if (!(e instanceof FSReadError) || (directory = BlacklistedDirectories.maybeMarkUnreadable((File)e.path)) == null) break;
                Keyspace.removeUnreadableSSTables((File)directory);
                break;
            }
            case ignore: {
                break;
            }
            default: {
                throw new IllegalStateException();
            }
        }
    }

    private static void handleStartupFSError(Throwable t) {
        switch (DatabaseDescriptor.getDiskFailurePolicy()) {
            case stop_paranoid: 
            case stop: 
            case die: {
                logger.error("Exiting forcefully due to file system exception on startup, disk failure policy \"{}\"", (Object)DatabaseDescriptor.getDiskFailurePolicy(), (Object)t);
                JVMStabilityInspector.killCurrentJVM(t, true);
                break;
            }
        }
    }

    public static long folderSize(File folder) {
        final long[] sizeArr = new long[]{0L};
        try {
            Files.walkFileTree(folder.toPath(), (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    sizeArr[0] = sizeArr[0] + attrs.size();
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (IOException e) {
            logger.error("Error while getting {} folder size. {}", (Object)folder, (Object)e);
        }
        return sizeArr[0];
    }

    public static void copyTo(DataInput in, OutputStream out, int length) throws IOException {
        byte[] buffer = new byte[65536];
        int copiedBytes = 0;
        while (copiedBytes + buffer.length < length) {
            in.readFully(buffer);
            out.write(buffer);
            copiedBytes += buffer.length;
        }
        if (copiedBytes < length) {
            int left = length - copiedBytes;
            in.readFully(buffer, 0, left);
            out.write(buffer, 0, left);
        }
    }

    public static boolean isSubDirectory(File parent, File child) throws IOException {
        parent = parent.getCanonicalFile();
        for (File toCheck = child = child.getCanonicalFile(); toCheck != null; toCheck = toCheck.getParentFile()) {
            if (!parent.equals(toCheck)) continue;
            return true;
        }
        return false;
    }

    public static void append(File file, String ... lines) {
        if (file.exists()) {
            FileUtils.write(file, Arrays.asList(lines), StandardOpenOption.APPEND);
        } else {
            FileUtils.write(file, Arrays.asList(lines), StandardOpenOption.CREATE);
        }
    }

    public static void appendAndSync(File file, String ... lines) {
        if (file.exists()) {
            FileUtils.write(file, Arrays.asList(lines), StandardOpenOption.APPEND, StandardOpenOption.SYNC);
        } else {
            FileUtils.write(file, Arrays.asList(lines), StandardOpenOption.CREATE, StandardOpenOption.SYNC);
        }
    }

    public static void replace(File file, String ... lines) {
        FileUtils.write(file, Arrays.asList(lines), StandardOpenOption.TRUNCATE_EXISTING);
    }

    public static void write(File file, List<String> lines, StandardOpenOption ... options) {
        try {
            Files.write(file.toPath(), lines, CHARSET, (OpenOption[])options);
        }
        catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    public static List<String> readLines(File file) {
        try {
            return Files.readAllLines(file.toPath(), CHARSET);
        }
        catch (IOException ex) {
            if (ex instanceof NoSuchFileException) {
                return Collections.emptyList();
            }
            throw new RuntimeException(ex);
        }
    }

    static {
        boolean canClean = false;
        try {
            ByteBuffer buf = ByteBuffer.allocateDirect(1);
            ((DirectBuffer)((Object)buf)).cleaner().clean();
            canClean = true;
        }
        catch (Throwable t) {
            JVMStabilityInspector.inspectThrowable(t);
            logger.info("Cannot initialize un-mmaper.  (Are you using a non-Oracle JVM?)  Compacted data files will not be removed promptly.  Consider using an Oracle JVM or using standard disk access mode");
        }
        canCleanDirectBuffers = canClean;
    }
}

