package de.lessvoid.nifty.tools;

import javax.annotation.Nonnull;
import java.util.EnumMap;
import java.util.Map;

/**
 * This is a special implementation for a enumeration based storage. It stores values in relation to enum keys that
 * are generated by a factory once they are requested. This is handy in case the factory creates carrier objects that
 * are supposed to store data in relation to the enumeration keys.
 *
 * @param <K> the type of the keys, this has to be a enumeration
 * @param <V> the type of objects used as values, the factory has to generate those
 * @author Martin Karing &lt;nitram@illarion.org&gt;
 */
public class EnumStorage<K extends Enum<K>, V> {
  /**
   * This map is the storage system of this class.
   */
  @Nonnull
  private final Map<K, V> storage;

  /**
   * This is the factory used to create the entries stored.
   */
  @Nonnull
  private final Factory<V> valueFactory;

  /**
   * Create a new instance of this storage.
   *
   * @param enumClass the class of the enumeration that is used to provide the keys
   * @param factory   the factory instance that creates the values
   */
  public EnumStorage(@Nonnull final Class<K> enumClass, @Nonnull final Factory<V> factory) {
    storage = new EnumMap<K, V>(enumClass);
    valueFactory = factory;
  }

  /**
   * Check if a specific entry of the storage was used already.
   * <p/>
   * In case this is {@code false} the value assigned to the key was never requested.
   *
   * @param key the key
   * @return {@code true} in case the key is assigned to a value
   */
  public boolean isSet(@Nonnull final K key) {
    return storage.containsKey(key);
  }

  /**
   * Get the value assigned to the specific key.
   *
   * @param key the key
   * @return the value assigned to the key, this value is newly created upon the first request
   */
  @Nonnull
  public V get(@Nonnull final K key) {
    V value = storage.get(key);
    if (value == null) {
      value = valueFactory.createNew();
      storage.put(key, value);
    }
    return value;
  }
}
