package com.alibaba.schedulerx.worker.util;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.commons.lang.StringUtils;
import org.springframework.context.ApplicationContext;

/**
 *
 * @author xiaomeng.hxm
 */
public class SpringContext {
    public volatile static ApplicationContext context;

    private volatile static CountDownLatch LATCH = new CountDownLatch(1);

    private static final AtomicInteger LOCK_TIMES = new AtomicInteger(0);

    private static final AtomicInteger UNLOCK_TIMES = new AtomicInteger(0);

    /**
     * not thread safe
     *
     * @param clazz
     * @return
     */
    public static <T> T getBean(Class<T> clazz) throws Exception {
        return getBean(null, clazz);
    }

    /**
     * this method not thread safe
     * @param className className pattern classFullName[:beanId]
     * @param <T> target Class Type
     * @return spring bean
     */
    public static <T> T getBean(final String className) throws Exception {
        LATCH.await();

        String clazzName = className;
        String beanId = null;
        int beanIdIndex = className.indexOf(":");
        if (beanIdIndex > 0) {
            clazzName = className.substring(0, beanIdIndex);
            beanId = className.substring(beanIdIndex + 1);
        }
        Class<?> clazz = Class.forName(clazzName);

        return getBean(beanId, clazz);
    }

    /**
     * this method not thread safe
     * @param className className pattern classFullName[:beanId]
     * @param classLoader target classLoader
     * @param <T> target Class Type
     * @return spring bean
     */
    public static <T> T getBean(final String className, ClassLoader classLoader) throws Exception {
        LATCH.await();

        String clazzName = className;
        String beanId = null;
        int beanIdIndex = className.indexOf(":");
        if (beanIdIndex > 0) {
            clazzName = className.substring(0, beanIdIndex);
            beanId = className.substring(beanIdIndex + 1);
        }

        Class<?> clazz = null;
        if (classLoader == null) {
            clazz = Class.forName(clazzName);
        } else {
            clazz = classLoader.loadClass(clazzName);
        }

        return getBean(beanId, clazz);
    }

    /**
     * get bean throw error, if spring context is ConfigurableApplicationContext then try to refresh it
     * to fix ConfigurableApplicationContext modified but get bean error.
     *
     * @param beanId  bean id
     * @param clazz   clazz
     * @param <T>     return Type
     * @return spring bean
     */
    private static <T> T getBean(String beanId, Class<?> clazz) throws Exception {
        LATCH.await();

        if (StringUtils.isBlank(beanId)) {
            return (T)context.getBean(clazz);
        } else {
            return (T)context.getBean(beanId, clazz);
        }
    }
    
    public static Object getBeanByName(String beanName) throws Exception {
        LATCH.await();

        return context.getBean(beanName);
    }

    public static void unlock() {
        LATCH.countDown();
        UNLOCK_TIMES.incrementAndGet();
    }

    public static void lock() {
        LATCH = new CountDownLatch(1);
        LOCK_TIMES.incrementAndGet();
    }

    public static int lockTimes(){
        return LOCK_TIMES.get();
    }

    public static int unLockTimes(){
        return UNLOCK_TIMES.get();
    }
}
