package com.alibaba.schedulerx.worker.timer;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import com.alibaba.schedulerx.common.domain.JobInstanceInfo;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.UnirestUtil;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.alibaba.schedulerx.worker.master.TaskMaster;
import com.alibaba.schedulerx.worker.master.TaskMasterPool;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.ListUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;

/**
 * update running job instance workers
 * @author zhaibian
 * @version $Id: TaskMasterUpdateWorkersTimer.java, v 0.1 2019年03月06日 11:44 zhaibian Exp $
 */
    public class TaskMasterUpdateWorkersTimer extends AbstractTimerTask {
    private TaskMasterPool masterPool = TaskMasterPool.INSTANCE;
    protected static final Logger LOGGER = LogFactory.getLogger(TaskMasterUpdateWorkersTimer.class);

    @Override
    public String getName() {
        return "TaskMasterUpdateWorkersTimer";
    }

    @Override
    public long getInitialDelay() {
        return 60;
    }

    @Override
    public long getPeriod() {
        return 60;
    }

    @Override
    public void run() {
        try {
            Collection<TaskMaster> allTaskMaster = masterPool.getAllTaskMaster();
            if (CollectionUtils.isEmpty(allTaskMaster)) {
                return;
            }

            // jobId -> jobInstance list
            Map<Long, List<JobInstanceInfo>> jobInstanceMap = Maps.newHashMap();
            // appGroupId -> jobId list
            Map<Long, List<Long>> appGroupJob = Maps.newHashMap();
            for (TaskMaster taskMaster : allTaskMaster) {
                Long jobId = taskMaster.getJobInstanceInfo().getJobId();
                List<JobInstanceInfo> list = jobInstanceMap.get(jobId);
                if (list == null) {
                    list = Lists.newArrayList();
                    jobInstanceMap.put(jobId, list);
                }
                list.add(taskMaster.getJobInstanceInfo());

                Long appGroupId = taskMaster.getJobInstanceInfo().getAppGroupId();
                List<Long> agList = appGroupJob.get(appGroupId);
                if (agList == null) {
                    agList = Lists.newArrayList();
                    appGroupJob.put(appGroupId, agList);
                }
                agList.add(jobId);
            }

            // grep has designate jobIds
            Map<Long, List<Long>> hasNotDesignatedAppGroupJob = Maps.newHashMap();
            List<Long> hasDesignatedJobIds = Lists.newArrayList();
            for (Entry<Long, List<Long>> entry : appGroupJob.entrySet()) {
                // value maybe int type
                Set<Long> ids = grepHasDesignateJobIds(entry.getKey(), entry.getValue());
                if (CollectionUtils.isNotEmpty(ids)) {
                    Set<Long> idSet = new HashSet<>(ids.size());
                    for (Number n : ids) {
                        idSet.add(n.longValue());
                    }
                    hasDesignatedJobIds.addAll(idSet);
                    hasNotDesignatedAppGroupJob.put(entry.getKey(), ListUtils.removeAll(entry.getValue(), idSet));
                } else {
                    hasNotDesignatedAppGroupJob.put(entry.getKey(), entry.getValue());
                }
            }

            // update has not designated jobs workers
            if (MapUtils.isNotEmpty(hasNotDesignatedAppGroupJob)) {
                for (Entry<Long, List<Long>> entry : hasNotDesignatedAppGroupJob.entrySet()) {
                    Set<String> allWorkers = getAllWorkers(entry.getKey(), -1L);
                    for (Long jobId : entry.getValue()) {
                        updateWorkers(allWorkers, jobInstanceMap.get(jobId));
                    }
                }
            }

            // update has designated job workers
            for (Long jobId : hasDesignatedJobIds) {
                Long appGroupId = jobInstanceMap.get(jobId).get(0).getAppGroupId();
                Set<String> allWorkers = getAllWorkers(appGroupId, jobId);
                updateWorkers(allWorkers, jobInstanceMap.get(jobId));
            }
        } catch (Exception ex) {
            LOGGER.error("update master workers error.", ex);
        }
    }

    private Set<Long> grepHasDesignateJobIds(Long appGroupId, List<Long> jobIds) throws Exception{
        String url = "http://{0}/app/grepHasDesignateJobIds.json?appGroupId={1}&jobIds={2}";
        return UnirestUtil.getSetData(url,
            ConfigUtil.getWorkerConfig().getString(WorkerConstants.WORKER_DOMAIN_NAME),
            appGroupId, StringUtils.join(jobIds, ","));
    }

    private void updateWorkers(Set<String> allWorkers, List<JobInstanceInfo> instanceInfos) {
        if (CollectionUtils.isEmpty(allWorkers)) {
            return;
        }

        for (JobInstanceInfo instanceInfo : instanceInfos) {
            if (instanceInfo.getAllWorkers().size() != allWorkers.size()
                || !instanceInfo.getAllWorkers().containsAll(allWorkers)) {
                instanceInfo.setAllWorkers(Lists.newCopyOnWriteArrayList(allWorkers));
                LOGGER.info("update appGroupId={} instanceId={} workers.", instanceInfo.getAppGroupId(),
                    instanceInfo.getJobInstanceId());
            }
        }
    }

    private Set<String> getAllWorkers(Long appGroupId, Long jobId) throws Exception {
        String url = "http://{0}/app/getAllUsefulWorkerList.json?appGroupId={1}&jobId={2}";
        try {
            return UnirestUtil.getSetData(url,
                ConfigUtil.getWorkerConfig().getString(WorkerConstants.WORKER_DOMAIN_NAME),
                appGroupId, jobId);
        } catch (Exception ex) {
            LOGGER.error("getAllWorkers failed.", ex);
        }

        return null;
    }
}