package com.alibaba.schedulerx.worker.processor;

import java.io.IOException;
import java.util.List;

import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.ExceptionUtil;
import com.alibaba.schedulerx.common.util.HessianUtil;
import com.alibaba.schedulerx.protocol.Worker.WorkerMapTaskRequest;
import com.alibaba.schedulerx.protocol.Worker.WorkerMapTaskResponse;
import com.alibaba.schedulerx.protocol.utils.FutureUtils;
import com.alibaba.schedulerx.worker.SchedulerxWorker;
import com.alibaba.schedulerx.worker.container.ContainerFactory;
import com.alibaba.schedulerx.worker.domain.JobContext;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.alibaba.schedulerx.worker.logcollector.LogCollector;
import com.alibaba.schedulerx.worker.logcollector.LogCollectorFactory;

import akka.actor.ActorSelection;
import com.google.common.collect.Lists;
import com.google.protobuf.ByteString;
import org.apache.commons.collections.CollectionUtils;

/**
 *
 * @author xiaomeng.hxm
 */
public abstract class MapJobProcessor extends JavaProcessor {
    private LogCollector logCollector = LogCollectorFactory.get();
    private static final Logger LOGGER = LogFactory.getLogger(MapJobProcessor.class);

    /**
     * distribute tasks to all workers.
     * 
     * @param taskList, every element in taskList shoudn't beyond 64KB
     * @param taskName
     * @return
     */
    public ProcessResult map(List<? extends Object> taskList, String taskName) {
        ProcessResult result = new ProcessResult(false);
        JobContext context = ContainerFactory.getContainerPool().getContext();
        ActorSelection masterAkkaSelection = SchedulerxWorker.actorSystem.actorSelection(context.getInstanceMasterActorPath());
        if (masterAkkaSelection == null) {
            String errMsg = "get taskMaster akka path error, path=" + context.getInstanceMasterActorPath();
            LOGGER.error(errMsg);
            result.setResult(errMsg);
            return result;
        }

        if (CollectionUtils.isEmpty(taskList)) {
            result.setResult("task list is empty");
            return result;
        }

        int batchSize = ConfigUtil.getWorkerConfig().getInt(WorkerConstants.WORKER_MAP_PAGE_SIZE,
            WorkerConstants.WORKER_MAP_PAGE_SIZE_DEFAULT);
        int size = taskList.size();
        LOGGER.info("map task list, jobInstanceId={}, taskName={}, size={}, batchSize={}", context.getJobInstanceId(), taskName, size, batchSize);
        int quotient = size / batchSize;
        int remainder = size % batchSize;
        /*
            map taskList in #batchNumber batch, every batch has no more than 3000 tasks;
         */
        int batchNumber = remainder > 0 ? quotient + 1 : quotient;
        List<WorkerMapTaskRequest.Builder> builders = Lists.newArrayList();
        for (int i=0; i<batchNumber; i++) {
            builders.add(WorkerMapTaskRequest.newBuilder());
        }

        int position = 0;
        int maxTaskBodySize = ConfigUtil.getWorkerConfig().getInt(WorkerConstants.TASK_BODY_SIZE_MAX,
                WorkerConstants.TASK_BODY_SIZE_MAX_DEFAULT);
        try {
            for (Object task : taskList) {
                int batchIdx = position++ / batchSize;
                byte[] taskBody = HessianUtil.toBytes(task);
                if (taskBody.length > maxTaskBodySize) {
                    throw new IOException("taskBody size more than " + maxTaskBodySize + "B!");
                }
                builders.get(batchIdx).addTaskBody(ByteString.copyFrom(taskBody));
            }
            for (WorkerMapTaskRequest.Builder builder : builders) {
                builder.setJobId(context.getJobId());
                builder.setJobInstanceId(context.getJobInstanceId());
                builder.setTaskId(context.getTaskId());
                builder.setTaskName(taskName);
                WorkerMapTaskResponse response = (WorkerMapTaskResponse) FutureUtils.awaitResult(masterAkkaSelection, builder.build(), 30);
                if (!response.getSuccess()) {
                    LOGGER.error(response.getMessage());
                    logCollector.collect(context.getUniqueId(), response.getMessage());
                    result.setResult(response.getMessage());
                    return result;
                }
                if (response.hasOverload() && response.getOverload()) {
                    // overload, sleep a while
                    LOGGER.warn("Task Master is busy, sleeping a while {}s...", 10);
                    Thread.sleep(10 * 1000);
                }
            }
            result.setStatus(true);
        } catch (Throwable e) {
            LOGGER.error("", e);
            logCollector.collect(context.getUniqueId(), ExceptionUtil.getTrace(e));
            result.setResult(ExceptionUtil.getMessage(e));
        }

        return result;
    }

    protected boolean isRootTask(JobContext context) {
        return context.getTaskName().equals(WorkerConstants.MAP_TASK_ROOT_NAME);
    }
}
