package com.alibaba.schedulerx.worker.logcollector;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import com.alibaba.schedulerx.common.constants.CommonConstants;
import com.alibaba.schedulerx.common.domain.StreamType;
import com.alibaba.schedulerx.common.util.Base64Util;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.ExceptionUtil;
import com.alibaba.schedulerx.worker.SchedulerxWorker;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;

import com.aliyun.openservices.aliyun.log.producer.Callback;
import com.aliyun.openservices.aliyun.log.producer.LogProducer;
import com.aliyun.openservices.aliyun.log.producer.Producer;
import com.aliyun.openservices.aliyun.log.producer.ProducerConfig;
import com.aliyun.openservices.aliyun.log.producer.ProjectConfig;
import com.aliyun.openservices.aliyun.log.producer.Result;
import com.aliyun.openservices.aliyun.log.producer.errors.LogSizeTooLargeException;
import com.aliyun.openservices.aliyun.log.producer.errors.TimeoutException;
import com.aliyun.openservices.log.common.LogItem;
import org.apache.commons.configuration.Configuration;

/**
 *
 * @author xiaomeng.hxm
 */
public class SlsLogCollector extends LogCollector {
    private static final Logger LOGGER = LogFactory.getLogger(SlsLogCollector.class);
    private final String project;
    private final String logStore;
    private final String endpoint;
    private final String accessKeyId;
    private final String accessKeySecret;
    private final ExecutorService es;
    private final Producer producer;

    public SlsLogCollector() {
        Configuration conf = ConfigUtil.getWorkerConfig();
        project = conf.getString(CommonConstants.SLS_PROJECT);
        logStore = conf.getString(CommonConstants.SLS_LOG_STORE);
        endpoint = conf.getString(CommonConstants.SLS_ENDPOINT);
        if (conf.getBoolean(CommonConstants.SLS_AKSK_ENCODED, false)) {
            String encodedAccessKeyId = conf.getString(CommonConstants.SLS_AK);
            accessKeyId = Base64Util.decode(encodedAccessKeyId);
            String encodedAccessKeySecret = conf.getString(CommonConstants.SLS_SK);
            accessKeySecret = Base64Util.decode(encodedAccessKeySecret);
        } else {
            accessKeyId = conf.getString(CommonConstants.SLS_AK);
            accessKeySecret = conf.getString(CommonConstants.SLS_SK);
        }

        int sendThreads = conf.getInt(WorkerConstants.SLS_SEND_THREADS, WorkerConstants.SLS_SEND_THREADS_DEFAULT);
        es = new ThreadPoolExecutor(
            sendThreads, sendThreads, 30, TimeUnit.SECONDS,
            new LinkedBlockingQueue<Runnable>(1024 * 10), new ThreadFactory() {
            private final AtomicInteger nextId = new AtomicInteger(1);
            private final String namePrefix = "Schedulerx-SLS-Send-Thread-";
            @Override
            public Thread newThread(Runnable r) {
                return new Thread(r, namePrefix + nextId.getAndIncrement());
            }
        }, new ThreadPoolExecutor.DiscardPolicy());

        ProducerConfig producerConfig = new ProducerConfig();
        producerConfig.setMaxBlockMs(0);
        producerConfig.setTotalSizeInBytes(10 * 1024 * 1024);
        producer = new LogProducer(producerConfig);
        producer.putProjectConfig(new ProjectConfig(project, endpoint, accessKeyId, accessKeySecret));
    }

    @Override
    public void collect(final String key, final String line, final Throwable t, StreamType streamType, boolean isEnd) {
        es.submit(new Runnable() {
            @Override
            public void run() {
                try {
                    producer.send(project, logStore, key, SchedulerxWorker.WORKER_ADDR, generateLogItem(line, t), new Callback() {
                        @Override
                        public void onCompletion(Result result) {
                          if (result.isSuccessful()) {
                              //TODO 这里做日志计量统计
                              LOGGER.debug("");
                          }
                        }
                      });
                } catch (TimeoutException | LogSizeTooLargeException ignore) {
                    LOGGER.info("sls producer out of 100M memory, msg line={}.", line);
                } catch (Exception e) {
                    LOGGER.error("SlsLogCollector error", e);
                }
            }
        });
    }

    private static LogItem generateLogItem(String line, Throwable t) {
        LogItem logItem = new LogItem();
        logItem.PushBack("message", line);
        if (t != null) {
            logItem.PushBack("trace", ExceptionUtil.getTrace(t));
        }
        return logItem;
      }

}
