package com.aliyun.openservices.ons.api.impl.auth;

import java.text.SimpleDateFormat;
import java.util.TimeZone;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import com.aliyun.openservices.shade.com.alibaba.fastjson.JSONException;
import com.aliyun.openservices.shade.com.alibaba.fastjson.JSONObject;
import com.aliyun.openservices.shade.com.alibaba.rocketmq.common.utils.HttpTinyClient;
import com.aliyun.openservices.shade.com.alibaba.rocketmq.logging.InternalLogger;

import com.aliyun.openservices.ons.api.exception.ONSClientException;
import com.aliyun.openservices.ons.api.impl.authority.SessionCredentials;
import com.aliyun.openservices.ons.api.impl.util.ClientLoggerUtil;

/**
 * @author mq-develop
 * @createTime 06
 * @description
 */
public class RamRoleService {

    public static final InternalLogger LOGGER = ClientLoggerUtil.getClientLogger();

    private static final String URL_ECS_RAMROLE = "http://100.100.100.200/latest/meta-data/Ram/security-credentials/";

    //默认更新token时间30分钟
    private static final long DEFAULT_UPDATE_BUFFER_TIME = 30L * 60 * 60L * 1000L;
    //默认最小更新时间间隔2秒
    private static final long MIN_UPDATE_INTERVAL = 5000L;
    //HTTP请求超时时间
    private static final long HTTP_TIMEOUT = 3000L;

    private final String ramRoleName;
    private final SessionCredentials sessionCredentials;

    private final ScheduledExecutorService scheduledExecutorService = new ScheduledThreadPoolExecutor(1,
        new ThreadFactory() {
            @Override
            public Thread newThread(Runnable r) {
                return new Thread(r, "ONSClient-UpdateTokenByRamRoleThread");
            }
        });

    public RamRoleService(String ramRoleName, SessionCredentials sessionCredentials) {
        this.ramRoleName = ramRoleName.trim();
        this.sessionCredentials = sessionCredentials;
        updateStsCredentials(ramRoleName);
    }

    private void updateStsCredentials(String ramRoleName) {
        StsCredentials stsCredentials = fetchStsCredentialsByRamRole(ramRoleName);
        sessionCredentials.setAccessKey(stsCredentials.getAccessKeyId());
        sessionCredentials.setSecretKey(stsCredentials.getAccessKeySecret());
        sessionCredentials.setSecurityToken(stsCredentials.getSecurityToken());
        LOGGER.info("update sts credentials success. ramRoleName=" + ramRoleName);

        addUpdateTimerTask(stsCredentials);
    }

    /**
     * ECS 默认token规则：token有效时间为6个小时，失效前1小时更新
     *
     * @param stsCredentials
     */
    private void addUpdateTimerTask(final StsCredentials stsCredentials) {
        long now = System.currentTimeMillis();
        long interval = stsCredentials.getExpiration() - now;
        long delay = MIN_UPDATE_INTERVAL;
        if (interval > DEFAULT_UPDATE_BUFFER_TIME) {
            delay = stsCredentials.getExpiration() - now - DEFAULT_UPDATE_BUFFER_TIME;
        } else {
            delay = Math.max(MIN_UPDATE_INTERVAL, (stsCredentials.getExpiration() - now) / 2);
        }

        scheduledExecutorService.schedule(new Runnable() {
            @Override
            public void run() {
                try {
                    updateStsCredentials(stsCredentials.getRamRoleName());
                } catch (Exception e) {
                    LOGGER.error("add update sts credentials timer task error.", e);
                }
            }
        }, delay, TimeUnit.MILLISECONDS);
    }

    /**
     * TOKEN 过期时间，单位为秒。 过期时间最小值为900秒，最大值为MaxSessionDuration设置的时间。默认值为3600秒。
     * <p> ECS的token有效期是6个小时，且是失效前一个小时更新
     *
     * @param ramRoleName
     * @return
     */
    private StsCredentials fetchStsCredentialsByRamRole(String ramRoleName) {
        String responseStr = null;
        try {
            HttpTinyClient.HttpResult result = HttpTinyClient.httpGet(URL_ECS_RAMROLE + ramRoleName,
                null, null, "UTF-8", HTTP_TIMEOUT);
            if (200 != result.code || null == result.content || "".equals(result.content)) {
                LOGGER.error("fetch sts credentials by ramRole error. ramRole=" + ramRoleName);
                throw new ONSClientException("fetch sts credentials by ram role[ " + ramRoleName + " ] error. please set right ram role");
            }
            responseStr = result.content;
            JSONObject jsonObject = JSONObject.parseObject(responseStr);

            StsCredentials stsCredentials = new StsCredentials();
            stsCredentials.setRamRoleName(ramRoleName);
            stsCredentials.setAccessKeyId(jsonObject.getString("AccessKeyId"));
            stsCredentials.setAccessKeySecret(jsonObject.getString("AccessKeySecret"));
            stsCredentials.setSecurityToken(jsonObject.getString("SecurityToken"));

            SimpleDateFormat utcFormater = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
            utcFormater.setTimeZone(TimeZone.getTimeZone("UTC"));
            stsCredentials.setExpiration(utcFormater.parse(jsonObject.getString("Expiration")).getTime());

            return stsCredentials;
        } catch (JSONException e) {
            LOGGER.error("fetch sts credentials by ram role error. ramRole=" + ramRoleName + ", response=" + responseStr, e);
            throw new ONSClientException("fetch sts credentials by ram role[ " + ramRoleName + " ] error. please set right ram role");
        } catch (Exception e) {
            LOGGER.error("fetch sts credentials by ram role error. ramRole=" + ramRoleName, e);
            throw new ONSClientException("fetch sts credentials by ram role[ " + ramRoleName + " ] error. please set right ram role");
        }
    }

}
