package org.hl7.fhir.utilities.npm;

import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Set;

import org.hl7.fhir.utilities.CommaSeparatedStringBuilder;
import org.hl7.fhir.utilities.SimpleHTTPClient;
import org.hl7.fhir.utilities.SimpleHTTPClient.HTTPResult;
import org.hl7.fhir.utilities.TextFile;
import org.hl7.fhir.utilities.Utilities;
import org.hl7.fhir.utilities.VersionUtilities;
import org.hl7.fhir.utilities.json.model.JsonArray;
import org.hl7.fhir.utilities.json.model.JsonObject;
import org.hl7.fhir.utilities.json.model.JsonProperty;
import org.hl7.fhir.utilities.json.parser.JsonParser;

public class PackageClient {



  public static final String PRIMARY_SERVER = "http://packages.fhir.org";
  public static final String SECONDARY_SERVER = "https://packages2.fhir.org/packages";

  private String address;
  private String cacheFolder;


  public PackageClient(String address) {
    super();
    this.address = address;
    try {
      cacheFolder = Utilities.path(System.getProperty("user.home"), ".fhir", "package-client");
      Utilities.createDirectory(cacheFolder);
    } catch (IOException e) {
    }
  }

  public boolean exists(String id, String ver) throws IOException {
    List<PackageInfo> vl = getVersions(id);
    for (PackageInfo pi : vl) {
      if (ver == null || ver.equals(pi.getVersion())) {
        return true;
      }
    }
    return false;
  }

  public InputStream fetch(String id, String ver) throws IOException {
    return fetchCached(Utilities.pathURL(address, id, ver));
  }

  public InputStream fetch(PackageInfo info) throws IOException {
    return fetchCached(Utilities.pathURL(address, info.getId(), info.getVersion()));
  }

  public InputStream fetchNpm(String id, String ver) throws IOException {
    return fetchCached(Utilities.pathURL(address, id, "-", id+"-"+ver+".tgz"));
  }

  public InputStream fetchCached(String url) throws IOException, FileNotFoundException {
    return fetchUrl(url, null);
  }

  protected String fn(String url) {
    String[] p = url.split("\\/");
    return p[2]+"-"+p[p.length-2]+"-"+p[p.length-1]+".tgz";
  }

  public List<PackageInfo> getVersions(String id) throws IOException {
    String url = Utilities.pathURL(address, id);
    List<PackageInfo> res = new ArrayList<>();
    JsonObject json;
    try {
      json = fetchJson(url);
      JsonObject versions = json.getJsonObject("versions");
      boolean hasDates = true;
      if (versions != null) {
        for (JsonProperty v : versions.getProperties()) {
          JsonObject obj = versions.getJsonObject(v.getName());
          Instant d = obj.hasString("date") ? obj.asDate("date") : null;
          if (d == null) {
            hasDates = false;
          }
          res.add(new PackageInfo(obj.asString("Name", "name"),
            obj.asString("Version", "version"),
            obj.asString("FhirVersion", "fhirVersion"),
            obj.asString("Description", "description"),
            obj.asString("url"),
            obj.asString("canonical"),
            address, d));
        }
      }
      if (hasDates) {
        Collections.sort(res, new PackageInfo.PackageInfoSorter(true));
      } else {
        Collections.sort(res, new PackageInfo.PackageInfoSorter(false));
      }
    } catch (Exception e) {
      System.out.println("Error fetching "+url+": "+e.getMessage());
    }
    return res;    
  }
   
  public List<PackageInfo> search(String name, String canonical, String fhirVersion, boolean preRelease) throws IOException {
    CommaSeparatedStringBuilder params = new CommaSeparatedStringBuilder("&");
    if (!Utilities.noString(name)) {
      params.append("name="+name);
    }
    if (!Utilities.noString(canonical)) {
      params.append("pkgcanonical="+canonical);
    }
    if (!Utilities.noString(fhirVersion)) {
      params.append("fhirversion="+fhirVersion);
    }
    if (preRelease) {
      params.append("prerelease="+preRelease);
    }
    List<PackageInfo> res = new ArrayList<>();
    try {
      JsonArray json = fetchJsonArray(Utilities.pathURL(address, "catalog?")+params.toString());
      boolean hasDates = true;
      for (JsonObject obj : json.asJsonObjects()) {
        Instant d = obj.has("date") ? obj.asDate("date") : null;
        if (d == null) {
          hasDates = false;
        }
        res.add(new PackageInfo(obj.asString("Name", "name"),
          obj.asString("Version", "version"),
          obj.asString("FhirVersion", "fhirVersion"),
          obj.asString("Description", "description"),
          obj.asString("url"),
          obj.asString("canonical"),
          address, d));
      }
      if (hasDates) {
        Collections.sort(res, new PackageInfo.PackageInfoSorter(true));
      } else {
        Collections.sort(res, new PackageInfo.PackageInfoSorter(false));
      }
    } catch (IOException e1) {
    }
    return res;    
  }  

  public Date getNewPackages(Date lastCalled, List<PackageInfo> updates) {
    return null;
  }
 
  private InputStream fetchUrl(String source, String accept) throws IOException {
    SimpleHTTPClient http = new SimpleHTTPClient();
    HTTPResult res = http.get(source, accept);
    res.checkThrowException();
    return new ByteArrayInputStream(res.getContent());
  }
  
  private JsonObject fetchJson(String source) throws IOException {
    String src = TextFile.streamToString(fetchUrl(source, "application/json"));
    //System.out.println(src);
    return JsonParser.parseObject(src);
  }
  
  private JsonArray fetchJsonArray(String source) throws IOException {
    String src = TextFile.streamToString(fetchUrl(source, "application/json"));
    //System.out.println(src);
    return (JsonArray) JsonParser.parse(src);
  }

  public String url(String id, String v) {
    return Utilities.pathURL(address, id, v);
  }

  public String getLatestVersion(String id) throws IOException {
    List<PackageInfo> list = getVersions(id);
    if (list.isEmpty()) {
      throw new IOException("Package not found: "+id);
    } else {
      String v = list.get(0).getVersion();
      for (PackageInfo p : list) {
        if (VersionUtilities.isThisOrLater(v, p.getVersion())) {
          v = p.getVersion();
        }
      }
      return v;
    }
  }
  
  public String getLatestVersion(String id, String specVersion) throws IOException {
    List<PackageInfo> list = getVersions(id);
    if (list.isEmpty()) {
      throw new IOException("Package not found: "+id);
    } else {
      String v = null;
      for (PackageInfo p : list) {
        if (VersionUtilities.isMajMinOrLaterPatch(specVersion, p.getVersion())) {
          v = p.getVersion();
        }
      }
      return v;
    }
  }

  protected PackageInfo getPackageInfoFromJSON(JsonObject o, String name, String canonical, String fhirVersion) {
      String id = o.asString("npm-name");
      String pname = o.asString("name");
      String pcanonical = o.asString("canonical");
      String description = o.asString("description");
      Instant d = o.has("date") ? o.asDate("date") : null;
      boolean ok = true;
      if (ok && !Utilities.noString(name)) {
        ok = (pname != null && pname.contains(name)) || (description != null && description.contains(name)) || (id != null && id.contains(name));
      }
      if (ok && !Utilities.noString(canonical)) {
        ok = pcanonical.contains(canonical);
      }
      String version = null;
      String fVersion = null;
      String url = null;

      if (ok) {
        // if we can find something...
        for (JsonObject e : o.getJsonObjects("editions")) {
          if (fhirVersion == null || fhirVersion.equals(e.asString("fhir-version"))) {
            String v = e.asString("ig-version");
            if (version == null || VersionUtilities.isThisOrLater(version, v)) {
              version = v;
              fVersion = e.getJsonArray("fhir-version").get(0).asString();
              url = e.asString("url");

              String npmPackage = e.asString("package");
              if (npmPackage != null && id == null) {
                id = npmPackage.substring(0, npmPackage.indexOf("#"));
              }
            }
          }
        }
      }
      return new PackageInfo(id, version, fVersion, description, url, pcanonical, address, d);
  }
  
  public List<PackageInfo> listFromRegistry(String name, String canonical, String fhirVersion) throws IOException {
    List<PackageInfo> result = new ArrayList<>();
    JsonObject packages = JsonParser.parseObjectFromUrl("https://raw.githubusercontent.com/FHIR/ig-registry/master/fhir-ig-list.json?nocache=" + System.currentTimeMillis());
    for (JsonObject o : packages.getJsonObjects("guides")) {
      if (o.has("canonical")) {
      final PackageInfo packageInfo = getPackageInfoFromJSON(o, name, canonical, fhirVersion);
        if (packageInfo.getVersion() != null) {
          result.add(packageInfo);
        }
      }
    }
    return result;
  }

  public void findDependents(Set<String> list, String id) {
    CommaSeparatedStringBuilder params = new CommaSeparatedStringBuilder("&");
    params.append("dependency="+id.replace("#", "|"));
    try {
      JsonArray json = fetchJsonArray(Utilities.pathURL(address, "catalog?")+params.toString());
      for (JsonObject obj : json.asJsonObjects()) {
        list.add(obj.asString("Name", "name"));
      }
    } catch (IOException e1) {
    }
  }
  
}