/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssmincidents.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.ssmincidents.SsmIncidentsClient;
import software.amazon.awssdk.services.ssmincidents.model.GetReplicationSetRequest;
import software.amazon.awssdk.services.ssmincidents.model.GetReplicationSetResponse;
import software.amazon.awssdk.services.ssmincidents.model.SsmIncidentsRequest;
import software.amazon.awssdk.services.ssmincidents.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultSsmIncidentsWaiter implements SsmIncidentsWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final SsmIncidentsClient client;

    private final AttributeMap managedResources;

    private final Waiter<GetReplicationSetResponse> waitForReplicationSetActiveWaiter;

    private final Waiter<GetReplicationSetResponse> waitForReplicationSetDeletedWaiter;

    private DefaultSsmIncidentsWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = SsmIncidentsClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.waitForReplicationSetActiveWaiter = Waiter.builder(GetReplicationSetResponse.class)
                .acceptors(waitForReplicationSetActiveWaiterAcceptors())
                .overrideConfiguration(waitForReplicationSetActiveWaiterConfig(builder.overrideConfiguration)).build();
        this.waitForReplicationSetDeletedWaiter = Waiter.builder(GetReplicationSetResponse.class)
                .acceptors(waitForReplicationSetDeletedWaiterAcceptors())
                .overrideConfiguration(waitForReplicationSetDeletedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<GetReplicationSetResponse> waitUntilWaitForReplicationSetActive(
            GetReplicationSetRequest getReplicationSetRequest) {
        return waitForReplicationSetActiveWaiter.run(() -> client
                .getReplicationSet(applyWaitersUserAgent(getReplicationSetRequest)));
    }

    @Override
    public WaiterResponse<GetReplicationSetResponse> waitUntilWaitForReplicationSetActive(
            GetReplicationSetRequest getReplicationSetRequest, WaiterOverrideConfiguration overrideConfig) {
        return waitForReplicationSetActiveWaiter.run(
                () -> client.getReplicationSet(applyWaitersUserAgent(getReplicationSetRequest)),
                waitForReplicationSetActiveWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetReplicationSetResponse> waitUntilWaitForReplicationSetDeleted(
            GetReplicationSetRequest getReplicationSetRequest) {
        return waitForReplicationSetDeletedWaiter.run(() -> client
                .getReplicationSet(applyWaitersUserAgent(getReplicationSetRequest)));
    }

    @Override
    public WaiterResponse<GetReplicationSetResponse> waitUntilWaitForReplicationSetDeleted(
            GetReplicationSetRequest getReplicationSetRequest, WaiterOverrideConfiguration overrideConfig) {
        return waitForReplicationSetDeletedWaiter.run(
                () -> client.getReplicationSet(applyWaitersUserAgent(getReplicationSetRequest)),
                waitForReplicationSetDeletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetReplicationSetResponse>> waitForReplicationSetActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super GetReplicationSetResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("replicationSet").field("status").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("replicationSet").field("status").value(), "CREATING");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("replicationSet").field("status").value(), "UPDATING");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("replicationSet").field("status").value(), "FAILED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetReplicationSetResponse>> waitForReplicationSetDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetReplicationSetResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("replicationSet").field("status").value(), "DELETING");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("replicationSet").field("status").value(), "FAILED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration waitForReplicationSetActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration waitForReplicationSetDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static SsmIncidentsWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends SsmIncidentsRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements SsmIncidentsWaiter.Builder {
        private SsmIncidentsClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public SsmIncidentsWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public SsmIncidentsWaiter.Builder client(SsmIncidentsClient client) {
            this.client = client;
            return this;
        }

        public SsmIncidentsWaiter build() {
            return new DefaultSsmIncidentsWaiter(this);
        }
    }
}
