/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssmincidents.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Basic details used in creating a response plan. The response plan is then used to create an incident record.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IncidentTemplate implements SdkPojo, Serializable,
        ToCopyableBuilder<IncidentTemplate.Builder, IncidentTemplate> {
    private static final SdkField<String> DEDUPE_STRING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("dedupeString").getter(getter(IncidentTemplate::dedupeString)).setter(setter(Builder::dedupeString))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dedupeString").build()).build();

    private static final SdkField<Integer> IMPACT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("impact").getter(getter(IncidentTemplate::impact)).setter(setter(Builder::impact))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("impact").build()).build();

    private static final SdkField<Map<String, String>> INCIDENT_TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("incidentTags")
            .getter(getter(IncidentTemplate::incidentTags))
            .setter(setter(Builder::incidentTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("incidentTags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<NotificationTargetItem>> NOTIFICATION_TARGETS_FIELD = SdkField
            .<List<NotificationTargetItem>> builder(MarshallingType.LIST)
            .memberName("notificationTargets")
            .getter(getter(IncidentTemplate::notificationTargets))
            .setter(setter(Builder::notificationTargets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("notificationTargets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NotificationTargetItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(NotificationTargetItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SUMMARY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("summary")
            .getter(getter(IncidentTemplate::summary)).setter(setter(Builder::summary))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("summary").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("title")
            .getter(getter(IncidentTemplate::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("title").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DEDUPE_STRING_FIELD,
            IMPACT_FIELD, INCIDENT_TAGS_FIELD, NOTIFICATION_TARGETS_FIELD, SUMMARY_FIELD, TITLE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String dedupeString;

    private final Integer impact;

    private final Map<String, String> incidentTags;

    private final List<NotificationTargetItem> notificationTargets;

    private final String summary;

    private final String title;

    private IncidentTemplate(BuilderImpl builder) {
        this.dedupeString = builder.dedupeString;
        this.impact = builder.impact;
        this.incidentTags = builder.incidentTags;
        this.notificationTargets = builder.notificationTargets;
        this.summary = builder.summary;
        this.title = builder.title;
    }

    /**
     * <p>
     * Used to stop Incident Manager from creating multiple incident records for the same incident.
     * </p>
     * 
     * @return Used to stop Incident Manager from creating multiple incident records for the same incident.
     */
    public final String dedupeString() {
        return dedupeString;
    }

    /**
     * <p>
     * The impact of the incident on your customers and applications.
     * </p>
     * 
     * @return The impact of the incident on your customers and applications.
     */
    public final Integer impact() {
        return impact;
    }

    /**
     * For responses, this returns true if the service returned a value for the IncidentTags property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIncidentTags() {
        return incidentTags != null && !(incidentTags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Tags to assign to the template. When the <code>StartIncident</code> API action is called, Incident Manager
     * assigns the tags specified in the template to the incident.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIncidentTags} method.
     * </p>
     * 
     * @return Tags to assign to the template. When the <code>StartIncident</code> API action is called, Incident
     *         Manager assigns the tags specified in the template to the incident.
     */
    public final Map<String, String> incidentTags() {
        return incidentTags;
    }

    /**
     * For responses, this returns true if the service returned a value for the NotificationTargets property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNotificationTargets() {
        return notificationTargets != null && !(notificationTargets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon SNS targets that are notified when updates are made to an incident.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNotificationTargets} method.
     * </p>
     * 
     * @return The Amazon SNS targets that are notified when updates are made to an incident.
     */
    public final List<NotificationTargetItem> notificationTargets() {
        return notificationTargets;
    }

    /**
     * <p>
     * The summary of the incident. The summary is a brief synopsis of what occurred, what's currently happening, and
     * context.
     * </p>
     * 
     * @return The summary of the incident. The summary is a brief synopsis of what occurred, what's currently
     *         happening, and context.
     */
    public final String summary() {
        return summary;
    }

    /**
     * <p>
     * The title of the incident.
     * </p>
     * 
     * @return The title of the incident.
     */
    public final String title() {
        return title;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dedupeString());
        hashCode = 31 * hashCode + Objects.hashCode(impact());
        hashCode = 31 * hashCode + Objects.hashCode(hasIncidentTags() ? incidentTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasNotificationTargets() ? notificationTargets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(summary());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IncidentTemplate)) {
            return false;
        }
        IncidentTemplate other = (IncidentTemplate) obj;
        return Objects.equals(dedupeString(), other.dedupeString()) && Objects.equals(impact(), other.impact())
                && hasIncidentTags() == other.hasIncidentTags() && Objects.equals(incidentTags(), other.incidentTags())
                && hasNotificationTargets() == other.hasNotificationTargets()
                && Objects.equals(notificationTargets(), other.notificationTargets())
                && Objects.equals(summary(), other.summary()) && Objects.equals(title(), other.title());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IncidentTemplate").add("DedupeString", dedupeString()).add("Impact", impact())
                .add("IncidentTags", hasIncidentTags() ? incidentTags() : null)
                .add("NotificationTargets", hasNotificationTargets() ? notificationTargets() : null).add("Summary", summary())
                .add("Title", title()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "dedupeString":
            return Optional.ofNullable(clazz.cast(dedupeString()));
        case "impact":
            return Optional.ofNullable(clazz.cast(impact()));
        case "incidentTags":
            return Optional.ofNullable(clazz.cast(incidentTags()));
        case "notificationTargets":
            return Optional.ofNullable(clazz.cast(notificationTargets()));
        case "summary":
            return Optional.ofNullable(clazz.cast(summary()));
        case "title":
            return Optional.ofNullable(clazz.cast(title()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<IncidentTemplate, T> g) {
        return obj -> g.apply((IncidentTemplate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IncidentTemplate> {
        /**
         * <p>
         * Used to stop Incident Manager from creating multiple incident records for the same incident.
         * </p>
         * 
         * @param dedupeString
         *        Used to stop Incident Manager from creating multiple incident records for the same incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dedupeString(String dedupeString);

        /**
         * <p>
         * The impact of the incident on your customers and applications.
         * </p>
         * 
         * @param impact
         *        The impact of the incident on your customers and applications.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder impact(Integer impact);

        /**
         * <p>
         * Tags to assign to the template. When the <code>StartIncident</code> API action is called, Incident Manager
         * assigns the tags specified in the template to the incident.
         * </p>
         * 
         * @param incidentTags
         *        Tags to assign to the template. When the <code>StartIncident</code> API action is called, Incident
         *        Manager assigns the tags specified in the template to the incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder incidentTags(Map<String, String> incidentTags);

        /**
         * <p>
         * The Amazon SNS targets that are notified when updates are made to an incident.
         * </p>
         * 
         * @param notificationTargets
         *        The Amazon SNS targets that are notified when updates are made to an incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationTargets(Collection<NotificationTargetItem> notificationTargets);

        /**
         * <p>
         * The Amazon SNS targets that are notified when updates are made to an incident.
         * </p>
         * 
         * @param notificationTargets
         *        The Amazon SNS targets that are notified when updates are made to an incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationTargets(NotificationTargetItem... notificationTargets);

        /**
         * <p>
         * The Amazon SNS targets that are notified when updates are made to an incident.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssmincidents.model.NotificationTargetItem.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.ssmincidents.model.NotificationTargetItem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ssmincidents.model.NotificationTargetItem.Builder#build()} is called
         * immediately and its result is passed to {@link #notificationTargets(List<NotificationTargetItem>)}.
         * 
         * @param notificationTargets
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ssmincidents.model.NotificationTargetItem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notificationTargets(java.util.Collection<NotificationTargetItem>)
         */
        Builder notificationTargets(Consumer<NotificationTargetItem.Builder>... notificationTargets);

        /**
         * <p>
         * The summary of the incident. The summary is a brief synopsis of what occurred, what's currently happening,
         * and context.
         * </p>
         * 
         * @param summary
         *        The summary of the incident. The summary is a brief synopsis of what occurred, what's currently
         *        happening, and context.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder summary(String summary);

        /**
         * <p>
         * The title of the incident.
         * </p>
         * 
         * @param title
         *        The title of the incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);
    }

    static final class BuilderImpl implements Builder {
        private String dedupeString;

        private Integer impact;

        private Map<String, String> incidentTags = DefaultSdkAutoConstructMap.getInstance();

        private List<NotificationTargetItem> notificationTargets = DefaultSdkAutoConstructList.getInstance();

        private String summary;

        private String title;

        private BuilderImpl() {
        }

        private BuilderImpl(IncidentTemplate model) {
            dedupeString(model.dedupeString);
            impact(model.impact);
            incidentTags(model.incidentTags);
            notificationTargets(model.notificationTargets);
            summary(model.summary);
            title(model.title);
        }

        public final String getDedupeString() {
            return dedupeString;
        }

        public final void setDedupeString(String dedupeString) {
            this.dedupeString = dedupeString;
        }

        @Override
        public final Builder dedupeString(String dedupeString) {
            this.dedupeString = dedupeString;
            return this;
        }

        public final Integer getImpact() {
            return impact;
        }

        public final void setImpact(Integer impact) {
            this.impact = impact;
        }

        @Override
        public final Builder impact(Integer impact) {
            this.impact = impact;
            return this;
        }

        public final Map<String, String> getIncidentTags() {
            if (incidentTags instanceof SdkAutoConstructMap) {
                return null;
            }
            return incidentTags;
        }

        public final void setIncidentTags(Map<String, String> incidentTags) {
            this.incidentTags = TagMapCopier.copy(incidentTags);
        }

        @Override
        public final Builder incidentTags(Map<String, String> incidentTags) {
            this.incidentTags = TagMapCopier.copy(incidentTags);
            return this;
        }

        public final List<NotificationTargetItem.Builder> getNotificationTargets() {
            List<NotificationTargetItem.Builder> result = NotificationTargetSetCopier.copyToBuilder(this.notificationTargets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNotificationTargets(Collection<NotificationTargetItem.BuilderImpl> notificationTargets) {
            this.notificationTargets = NotificationTargetSetCopier.copyFromBuilder(notificationTargets);
        }

        @Override
        public final Builder notificationTargets(Collection<NotificationTargetItem> notificationTargets) {
            this.notificationTargets = NotificationTargetSetCopier.copy(notificationTargets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notificationTargets(NotificationTargetItem... notificationTargets) {
            notificationTargets(Arrays.asList(notificationTargets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notificationTargets(Consumer<NotificationTargetItem.Builder>... notificationTargets) {
            notificationTargets(Stream.of(notificationTargets)
                    .map(c -> NotificationTargetItem.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getSummary() {
            return summary;
        }

        public final void setSummary(String summary) {
            this.summary = summary;
        }

        @Override
        public final Builder summary(String summary) {
            this.summary = summary;
            return this;
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        @Override
        public IncidentTemplate build() {
            return new IncidentTemplate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
