/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateDomainRequest extends SageMakerRequest implements
        ToCopyableBuilder<CreateDomainRequest.Builder, CreateDomainRequest> {
    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DomainName").getter(getter(CreateDomainRequest::domainName)).setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainName").build()).build();

    private static final SdkField<String> AUTH_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AuthMode").getter(getter(CreateDomainRequest::authModeAsString)).setter(setter(Builder::authMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthMode").build()).build();

    private static final SdkField<UserSettings> DEFAULT_USER_SETTINGS_FIELD = SdkField
            .<UserSettings> builder(MarshallingType.SDK_POJO).memberName("DefaultUserSettings")
            .getter(getter(CreateDomainRequest::defaultUserSettings)).setter(setter(Builder::defaultUserSettings))
            .constructor(UserSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultUserSettings").build())
            .build();

    private static final SdkField<List<String>> SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SubnetIds")
            .getter(getter(CreateDomainRequest::subnetIds))
            .setter(setter(Builder::subnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VpcId")
            .getter(getter(CreateDomainRequest::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateDomainRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> APP_NETWORK_ACCESS_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AppNetworkAccessType").getter(getter(CreateDomainRequest::appNetworkAccessTypeAsString))
            .setter(setter(Builder::appNetworkAccessType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AppNetworkAccessType").build())
            .build();

    private static final SdkField<String> HOME_EFS_FILE_SYSTEM_KMS_KEY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING).memberName("HomeEfsFileSystemKmsKeyId")
            .getter(getter(CreateDomainRequest::homeEfsFileSystemKmsKeyId)).setter(setter(Builder::homeEfsFileSystemKmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HomeEfsFileSystemKmsKeyId").build())
            .build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(CreateDomainRequest::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<String> APP_SECURITY_GROUP_MANAGEMENT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AppSecurityGroupManagement")
            .getter(getter(CreateDomainRequest::appSecurityGroupManagementAsString))
            .setter(setter(Builder::appSecurityGroupManagement))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AppSecurityGroupManagement").build())
            .build();

    private static final SdkField<DomainSettings> DOMAIN_SETTINGS_FIELD = SdkField
            .<DomainSettings> builder(MarshallingType.SDK_POJO).memberName("DomainSettings")
            .getter(getter(CreateDomainRequest::domainSettings)).setter(setter(Builder::domainSettings))
            .constructor(DomainSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainSettings").build()).build();

    private static final SdkField<DefaultSpaceSettings> DEFAULT_SPACE_SETTINGS_FIELD = SdkField
            .<DefaultSpaceSettings> builder(MarshallingType.SDK_POJO).memberName("DefaultSpaceSettings")
            .getter(getter(CreateDomainRequest::defaultSpaceSettings)).setter(setter(Builder::defaultSpaceSettings))
            .constructor(DefaultSpaceSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultSpaceSettings").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOMAIN_NAME_FIELD,
            AUTH_MODE_FIELD, DEFAULT_USER_SETTINGS_FIELD, SUBNET_IDS_FIELD, VPC_ID_FIELD, TAGS_FIELD,
            APP_NETWORK_ACCESS_TYPE_FIELD, HOME_EFS_FILE_SYSTEM_KMS_KEY_ID_FIELD, KMS_KEY_ID_FIELD,
            APP_SECURITY_GROUP_MANAGEMENT_FIELD, DOMAIN_SETTINGS_FIELD, DEFAULT_SPACE_SETTINGS_FIELD));

    private final String domainName;

    private final String authMode;

    private final UserSettings defaultUserSettings;

    private final List<String> subnetIds;

    private final String vpcId;

    private final List<Tag> tags;

    private final String appNetworkAccessType;

    private final String homeEfsFileSystemKmsKeyId;

    private final String kmsKeyId;

    private final String appSecurityGroupManagement;

    private final DomainSettings domainSettings;

    private final DefaultSpaceSettings defaultSpaceSettings;

    private CreateDomainRequest(BuilderImpl builder) {
        super(builder);
        this.domainName = builder.domainName;
        this.authMode = builder.authMode;
        this.defaultUserSettings = builder.defaultUserSettings;
        this.subnetIds = builder.subnetIds;
        this.vpcId = builder.vpcId;
        this.tags = builder.tags;
        this.appNetworkAccessType = builder.appNetworkAccessType;
        this.homeEfsFileSystemKmsKeyId = builder.homeEfsFileSystemKmsKeyId;
        this.kmsKeyId = builder.kmsKeyId;
        this.appSecurityGroupManagement = builder.appSecurityGroupManagement;
        this.domainSettings = builder.domainSettings;
        this.defaultSpaceSettings = builder.defaultSpaceSettings;
    }

    /**
     * <p>
     * A name for the domain.
     * </p>
     * 
     * @return A name for the domain.
     */
    public final String domainName() {
        return domainName;
    }

    /**
     * <p>
     * The mode of authentication that members use to access the domain.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #authMode} will
     * return {@link AuthMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #authModeAsString}.
     * </p>
     * 
     * @return The mode of authentication that members use to access the domain.
     * @see AuthMode
     */
    public final AuthMode authMode() {
        return AuthMode.fromValue(authMode);
    }

    /**
     * <p>
     * The mode of authentication that members use to access the domain.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #authMode} will
     * return {@link AuthMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #authModeAsString}.
     * </p>
     * 
     * @return The mode of authentication that members use to access the domain.
     * @see AuthMode
     */
    public final String authModeAsString() {
        return authMode;
    }

    /**
     * <p>
     * The default settings to use to create a user profile when <code>UserSettings</code> isn't specified in the call
     * to the <code>CreateUserProfile</code> API.
     * </p>
     * <p>
     * <code>SecurityGroups</code> is aggregated when specified in both calls. For all other settings in
     * <code>UserSettings</code>, the values specified in <code>CreateUserProfile</code> take precedence over those
     * specified in <code>CreateDomain</code>.
     * </p>
     * 
     * @return The default settings to use to create a user profile when <code>UserSettings</code> isn't specified in
     *         the call to the <code>CreateUserProfile</code> API.</p>
     *         <p>
     *         <code>SecurityGroups</code> is aggregated when specified in both calls. For all other settings in
     *         <code>UserSettings</code>, the values specified in <code>CreateUserProfile</code> take precedence over
     *         those specified in <code>CreateDomain</code>.
     */
    public final UserSettings defaultUserSettings() {
        return defaultUserSettings;
    }

    /**
     * For responses, this returns true if the service returned a value for the SubnetIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSubnetIds() {
        return subnetIds != null && !(subnetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The VPC subnets that Studio uses for communication.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubnetIds} method.
     * </p>
     * 
     * @return The VPC subnets that Studio uses for communication.
     */
    public final List<String> subnetIds() {
        return subnetIds;
    }

    /**
     * <p>
     * The ID of the Amazon Virtual Private Cloud (VPC) that Studio uses for communication.
     * </p>
     * 
     * @return The ID of the Amazon Virtual Private Cloud (VPC) that Studio uses for communication.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Tags to associated with the Domain. Each tag consists of a key and an optional value. Tag keys must be unique per
     * resource. Tags are searchable using the <code>Search</code> API.
     * </p>
     * <p>
     * Tags that you specify for the Domain are also added to all Apps that the Domain launches.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Tags to associated with the Domain. Each tag consists of a key and an optional value. Tag keys must be
     *         unique per resource. Tags are searchable using the <code>Search</code> API.</p>
     *         <p>
     *         Tags that you specify for the Domain are also added to all Apps that the Domain launches.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Specifies the VPC used for non-EFS traffic. The default value is <code>PublicInternetOnly</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PublicInternetOnly</code> - Non-EFS traffic is through a VPC managed by Amazon SageMaker, which allows
     * direct internet access
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VpcOnly</code> - All Studio traffic is through the specified VPC and subnets
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #appNetworkAccessType} will return {@link AppNetworkAccessType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #appNetworkAccessTypeAsString}.
     * </p>
     * 
     * @return Specifies the VPC used for non-EFS traffic. The default value is <code>PublicInternetOnly</code>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PublicInternetOnly</code> - Non-EFS traffic is through a VPC managed by Amazon SageMaker, which
     *         allows direct internet access
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VpcOnly</code> - All Studio traffic is through the specified VPC and subnets
     *         </p>
     *         </li>
     * @see AppNetworkAccessType
     */
    public final AppNetworkAccessType appNetworkAccessType() {
        return AppNetworkAccessType.fromValue(appNetworkAccessType);
    }

    /**
     * <p>
     * Specifies the VPC used for non-EFS traffic. The default value is <code>PublicInternetOnly</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PublicInternetOnly</code> - Non-EFS traffic is through a VPC managed by Amazon SageMaker, which allows
     * direct internet access
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VpcOnly</code> - All Studio traffic is through the specified VPC and subnets
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #appNetworkAccessType} will return {@link AppNetworkAccessType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #appNetworkAccessTypeAsString}.
     * </p>
     * 
     * @return Specifies the VPC used for non-EFS traffic. The default value is <code>PublicInternetOnly</code>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PublicInternetOnly</code> - Non-EFS traffic is through a VPC managed by Amazon SageMaker, which
     *         allows direct internet access
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VpcOnly</code> - All Studio traffic is through the specified VPC and subnets
     *         </p>
     *         </li>
     * @see AppNetworkAccessType
     */
    public final String appNetworkAccessTypeAsString() {
        return appNetworkAccessType;
    }

    /**
     * <p>
     * Use <code>KmsKeyId</code>.
     * </p>
     * 
     * @return Use <code>KmsKeyId</code>.
     * @deprecated This property is deprecated, use KmsKeyId instead.
     */
    @Deprecated
    public final String homeEfsFileSystemKmsKeyId() {
        return homeEfsFileSystemKmsKeyId;
    }

    /**
     * <p>
     * SageMaker uses Amazon Web Services KMS to encrypt the EFS volume attached to the domain with an Amazon Web
     * Services managed key by default. For more control, specify a customer managed key.
     * </p>
     * 
     * @return SageMaker uses Amazon Web Services KMS to encrypt the EFS volume attached to the domain with an Amazon
     *         Web Services managed key by default. For more control, specify a customer managed key.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The entity that creates and manages the required security groups for inter-app communication in
     * <code>VPCOnly</code> mode. Required when <code>CreateDomain.AppNetworkAccessType</code> is <code>VPCOnly</code>
     * and <code>DomainSettings.RStudioServerProDomainSettings.DomainExecutionRoleArn</code> is provided. If setting up
     * the domain for use with RStudio, this value must be set to <code>Service</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #appSecurityGroupManagement} will return {@link AppSecurityGroupManagement#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #appSecurityGroupManagementAsString}.
     * </p>
     * 
     * @return The entity that creates and manages the required security groups for inter-app communication in
     *         <code>VPCOnly</code> mode. Required when <code>CreateDomain.AppNetworkAccessType</code> is
     *         <code>VPCOnly</code> and
     *         <code>DomainSettings.RStudioServerProDomainSettings.DomainExecutionRoleArn</code> is provided. If setting
     *         up the domain for use with RStudio, this value must be set to <code>Service</code>.
     * @see AppSecurityGroupManagement
     */
    public final AppSecurityGroupManagement appSecurityGroupManagement() {
        return AppSecurityGroupManagement.fromValue(appSecurityGroupManagement);
    }

    /**
     * <p>
     * The entity that creates and manages the required security groups for inter-app communication in
     * <code>VPCOnly</code> mode. Required when <code>CreateDomain.AppNetworkAccessType</code> is <code>VPCOnly</code>
     * and <code>DomainSettings.RStudioServerProDomainSettings.DomainExecutionRoleArn</code> is provided. If setting up
     * the domain for use with RStudio, this value must be set to <code>Service</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #appSecurityGroupManagement} will return {@link AppSecurityGroupManagement#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #appSecurityGroupManagementAsString}.
     * </p>
     * 
     * @return The entity that creates and manages the required security groups for inter-app communication in
     *         <code>VPCOnly</code> mode. Required when <code>CreateDomain.AppNetworkAccessType</code> is
     *         <code>VPCOnly</code> and
     *         <code>DomainSettings.RStudioServerProDomainSettings.DomainExecutionRoleArn</code> is provided. If setting
     *         up the domain for use with RStudio, this value must be set to <code>Service</code>.
     * @see AppSecurityGroupManagement
     */
    public final String appSecurityGroupManagementAsString() {
        return appSecurityGroupManagement;
    }

    /**
     * <p>
     * A collection of <code>Domain</code> settings.
     * </p>
     * 
     * @return A collection of <code>Domain</code> settings.
     */
    public final DomainSettings domainSettings() {
        return domainSettings;
    }

    /**
     * <p>
     * The default settings used to create a space.
     * </p>
     * 
     * @return The default settings used to create a space.
     */
    public final DefaultSpaceSettings defaultSpaceSettings() {
        return defaultSpaceSettings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(authModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(defaultUserSettings());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnetIds() ? subnetIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(appNetworkAccessTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(homeEfsFileSystemKmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(appSecurityGroupManagementAsString());
        hashCode = 31 * hashCode + Objects.hashCode(domainSettings());
        hashCode = 31 * hashCode + Objects.hashCode(defaultSpaceSettings());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateDomainRequest)) {
            return false;
        }
        CreateDomainRequest other = (CreateDomainRequest) obj;
        return Objects.equals(domainName(), other.domainName()) && Objects.equals(authModeAsString(), other.authModeAsString())
                && Objects.equals(defaultUserSettings(), other.defaultUserSettings()) && hasSubnetIds() == other.hasSubnetIds()
                && Objects.equals(subnetIds(), other.subnetIds()) && Objects.equals(vpcId(), other.vpcId())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(appNetworkAccessTypeAsString(), other.appNetworkAccessTypeAsString())
                && Objects.equals(homeEfsFileSystemKmsKeyId(), other.homeEfsFileSystemKmsKeyId())
                && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(appSecurityGroupManagementAsString(), other.appSecurityGroupManagementAsString())
                && Objects.equals(domainSettings(), other.domainSettings())
                && Objects.equals(defaultSpaceSettings(), other.defaultSpaceSettings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateDomainRequest").add("DomainName", domainName()).add("AuthMode", authModeAsString())
                .add("DefaultUserSettings", defaultUserSettings()).add("SubnetIds", hasSubnetIds() ? subnetIds() : null)
                .add("VpcId", vpcId()).add("Tags", hasTags() ? tags() : null)
                .add("AppNetworkAccessType", appNetworkAccessTypeAsString())
                .add("HomeEfsFileSystemKmsKeyId", homeEfsFileSystemKmsKeyId()).add("KmsKeyId", kmsKeyId())
                .add("AppSecurityGroupManagement", appSecurityGroupManagementAsString()).add("DomainSettings", domainSettings())
                .add("DefaultSpaceSettings", defaultSpaceSettings()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DomainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "AuthMode":
            return Optional.ofNullable(clazz.cast(authModeAsString()));
        case "DefaultUserSettings":
            return Optional.ofNullable(clazz.cast(defaultUserSettings()));
        case "SubnetIds":
            return Optional.ofNullable(clazz.cast(subnetIds()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "AppNetworkAccessType":
            return Optional.ofNullable(clazz.cast(appNetworkAccessTypeAsString()));
        case "HomeEfsFileSystemKmsKeyId":
            return Optional.ofNullable(clazz.cast(homeEfsFileSystemKmsKeyId()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "AppSecurityGroupManagement":
            return Optional.ofNullable(clazz.cast(appSecurityGroupManagementAsString()));
        case "DomainSettings":
            return Optional.ofNullable(clazz.cast(domainSettings()));
        case "DefaultSpaceSettings":
            return Optional.ofNullable(clazz.cast(defaultSpaceSettings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateDomainRequest, T> g) {
        return obj -> g.apply((CreateDomainRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SageMakerRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateDomainRequest> {
        /**
         * <p>
         * A name for the domain.
         * </p>
         * 
         * @param domainName
         *        A name for the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * The mode of authentication that members use to access the domain.
         * </p>
         * 
         * @param authMode
         *        The mode of authentication that members use to access the domain.
         * @see AuthMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthMode
         */
        Builder authMode(String authMode);

        /**
         * <p>
         * The mode of authentication that members use to access the domain.
         * </p>
         * 
         * @param authMode
         *        The mode of authentication that members use to access the domain.
         * @see AuthMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthMode
         */
        Builder authMode(AuthMode authMode);

        /**
         * <p>
         * The default settings to use to create a user profile when <code>UserSettings</code> isn't specified in the
         * call to the <code>CreateUserProfile</code> API.
         * </p>
         * <p>
         * <code>SecurityGroups</code> is aggregated when specified in both calls. For all other settings in
         * <code>UserSettings</code>, the values specified in <code>CreateUserProfile</code> take precedence over those
         * specified in <code>CreateDomain</code>.
         * </p>
         * 
         * @param defaultUserSettings
         *        The default settings to use to create a user profile when <code>UserSettings</code> isn't specified in
         *        the call to the <code>CreateUserProfile</code> API.</p>
         *        <p>
         *        <code>SecurityGroups</code> is aggregated when specified in both calls. For all other settings in
         *        <code>UserSettings</code>, the values specified in <code>CreateUserProfile</code> take precedence over
         *        those specified in <code>CreateDomain</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultUserSettings(UserSettings defaultUserSettings);

        /**
         * <p>
         * The default settings to use to create a user profile when <code>UserSettings</code> isn't specified in the
         * call to the <code>CreateUserProfile</code> API.
         * </p>
         * <p>
         * <code>SecurityGroups</code> is aggregated when specified in both calls. For all other settings in
         * <code>UserSettings</code>, the values specified in <code>CreateUserProfile</code> take precedence over those
         * specified in <code>CreateDomain</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link UserSettings.Builder} avoiding the need
         * to create one manually via {@link UserSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link UserSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #defaultUserSettings(UserSettings)}.
         * 
         * @param defaultUserSettings
         *        a consumer that will call methods on {@link UserSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #defaultUserSettings(UserSettings)
         */
        default Builder defaultUserSettings(Consumer<UserSettings.Builder> defaultUserSettings) {
            return defaultUserSettings(UserSettings.builder().applyMutation(defaultUserSettings).build());
        }

        /**
         * <p>
         * The VPC subnets that Studio uses for communication.
         * </p>
         * 
         * @param subnetIds
         *        The VPC subnets that Studio uses for communication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(Collection<String> subnetIds);

        /**
         * <p>
         * The VPC subnets that Studio uses for communication.
         * </p>
         * 
         * @param subnetIds
         *        The VPC subnets that Studio uses for communication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(String... subnetIds);

        /**
         * <p>
         * The ID of the Amazon Virtual Private Cloud (VPC) that Studio uses for communication.
         * </p>
         * 
         * @param vpcId
         *        The ID of the Amazon Virtual Private Cloud (VPC) that Studio uses for communication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * Tags to associated with the Domain. Each tag consists of a key and an optional value. Tag keys must be unique
         * per resource. Tags are searchable using the <code>Search</code> API.
         * </p>
         * <p>
         * Tags that you specify for the Domain are also added to all Apps that the Domain launches.
         * </p>
         * 
         * @param tags
         *        Tags to associated with the Domain. Each tag consists of a key and an optional value. Tag keys must be
         *        unique per resource. Tags are searchable using the <code>Search</code> API.</p>
         *        <p>
         *        Tags that you specify for the Domain are also added to all Apps that the Domain launches.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Tags to associated with the Domain. Each tag consists of a key and an optional value. Tag keys must be unique
         * per resource. Tags are searchable using the <code>Search</code> API.
         * </p>
         * <p>
         * Tags that you specify for the Domain are also added to all Apps that the Domain launches.
         * </p>
         * 
         * @param tags
         *        Tags to associated with the Domain. Each tag consists of a key and an optional value. Tag keys must be
         *        unique per resource. Tags are searchable using the <code>Search</code> API.</p>
         *        <p>
         *        Tags that you specify for the Domain are also added to all Apps that the Domain launches.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Tags to associated with the Domain. Each tag consists of a key and an optional value. Tag keys must be unique
         * per resource. Tags are searchable using the <code>Search</code> API.
         * </p>
         * <p>
         * Tags that you specify for the Domain are also added to all Apps that the Domain launches.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.sagemaker.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Specifies the VPC used for non-EFS traffic. The default value is <code>PublicInternetOnly</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PublicInternetOnly</code> - Non-EFS traffic is through a VPC managed by Amazon SageMaker, which allows
         * direct internet access
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VpcOnly</code> - All Studio traffic is through the specified VPC and subnets
         * </p>
         * </li>
         * </ul>
         * 
         * @param appNetworkAccessType
         *        Specifies the VPC used for non-EFS traffic. The default value is <code>PublicInternetOnly</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PublicInternetOnly</code> - Non-EFS traffic is through a VPC managed by Amazon SageMaker, which
         *        allows direct internet access
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VpcOnly</code> - All Studio traffic is through the specified VPC and subnets
         *        </p>
         *        </li>
         * @see AppNetworkAccessType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AppNetworkAccessType
         */
        Builder appNetworkAccessType(String appNetworkAccessType);

        /**
         * <p>
         * Specifies the VPC used for non-EFS traffic. The default value is <code>PublicInternetOnly</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PublicInternetOnly</code> - Non-EFS traffic is through a VPC managed by Amazon SageMaker, which allows
         * direct internet access
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VpcOnly</code> - All Studio traffic is through the specified VPC and subnets
         * </p>
         * </li>
         * </ul>
         * 
         * @param appNetworkAccessType
         *        Specifies the VPC used for non-EFS traffic. The default value is <code>PublicInternetOnly</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PublicInternetOnly</code> - Non-EFS traffic is through a VPC managed by Amazon SageMaker, which
         *        allows direct internet access
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VpcOnly</code> - All Studio traffic is through the specified VPC and subnets
         *        </p>
         *        </li>
         * @see AppNetworkAccessType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AppNetworkAccessType
         */
        Builder appNetworkAccessType(AppNetworkAccessType appNetworkAccessType);

        /**
         * <p>
         * Use <code>KmsKeyId</code>.
         * </p>
         * 
         * @param homeEfsFileSystemKmsKeyId
         *        Use <code>KmsKeyId</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated This property is deprecated, use KmsKeyId instead.
         */
        @Deprecated
        Builder homeEfsFileSystemKmsKeyId(String homeEfsFileSystemKmsKeyId);

        /**
         * <p>
         * SageMaker uses Amazon Web Services KMS to encrypt the EFS volume attached to the domain with an Amazon Web
         * Services managed key by default. For more control, specify a customer managed key.
         * </p>
         * 
         * @param kmsKeyId
         *        SageMaker uses Amazon Web Services KMS to encrypt the EFS volume attached to the domain with an Amazon
         *        Web Services managed key by default. For more control, specify a customer managed key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The entity that creates and manages the required security groups for inter-app communication in
         * <code>VPCOnly</code> mode. Required when <code>CreateDomain.AppNetworkAccessType</code> is
         * <code>VPCOnly</code> and <code>DomainSettings.RStudioServerProDomainSettings.DomainExecutionRoleArn</code> is
         * provided. If setting up the domain for use with RStudio, this value must be set to <code>Service</code>.
         * </p>
         * 
         * @param appSecurityGroupManagement
         *        The entity that creates and manages the required security groups for inter-app communication in
         *        <code>VPCOnly</code> mode. Required when <code>CreateDomain.AppNetworkAccessType</code> is
         *        <code>VPCOnly</code> and
         *        <code>DomainSettings.RStudioServerProDomainSettings.DomainExecutionRoleArn</code> is provided. If
         *        setting up the domain for use with RStudio, this value must be set to <code>Service</code>.
         * @see AppSecurityGroupManagement
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AppSecurityGroupManagement
         */
        Builder appSecurityGroupManagement(String appSecurityGroupManagement);

        /**
         * <p>
         * The entity that creates and manages the required security groups for inter-app communication in
         * <code>VPCOnly</code> mode. Required when <code>CreateDomain.AppNetworkAccessType</code> is
         * <code>VPCOnly</code> and <code>DomainSettings.RStudioServerProDomainSettings.DomainExecutionRoleArn</code> is
         * provided. If setting up the domain for use with RStudio, this value must be set to <code>Service</code>.
         * </p>
         * 
         * @param appSecurityGroupManagement
         *        The entity that creates and manages the required security groups for inter-app communication in
         *        <code>VPCOnly</code> mode. Required when <code>CreateDomain.AppNetworkAccessType</code> is
         *        <code>VPCOnly</code> and
         *        <code>DomainSettings.RStudioServerProDomainSettings.DomainExecutionRoleArn</code> is provided. If
         *        setting up the domain for use with RStudio, this value must be set to <code>Service</code>.
         * @see AppSecurityGroupManagement
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AppSecurityGroupManagement
         */
        Builder appSecurityGroupManagement(AppSecurityGroupManagement appSecurityGroupManagement);

        /**
         * <p>
         * A collection of <code>Domain</code> settings.
         * </p>
         * 
         * @param domainSettings
         *        A collection of <code>Domain</code> settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainSettings(DomainSettings domainSettings);

        /**
         * <p>
         * A collection of <code>Domain</code> settings.
         * </p>
         * This is a convenience method that creates an instance of the {@link DomainSettings.Builder} avoiding the need
         * to create one manually via {@link DomainSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DomainSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #domainSettings(DomainSettings)}.
         * 
         * @param domainSettings
         *        a consumer that will call methods on {@link DomainSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #domainSettings(DomainSettings)
         */
        default Builder domainSettings(Consumer<DomainSettings.Builder> domainSettings) {
            return domainSettings(DomainSettings.builder().applyMutation(domainSettings).build());
        }

        /**
         * <p>
         * The default settings used to create a space.
         * </p>
         * 
         * @param defaultSpaceSettings
         *        The default settings used to create a space.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultSpaceSettings(DefaultSpaceSettings defaultSpaceSettings);

        /**
         * <p>
         * The default settings used to create a space.
         * </p>
         * This is a convenience method that creates an instance of the {@link DefaultSpaceSettings.Builder} avoiding
         * the need to create one manually via {@link DefaultSpaceSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DefaultSpaceSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #defaultSpaceSettings(DefaultSpaceSettings)}.
         * 
         * @param defaultSpaceSettings
         *        a consumer that will call methods on {@link DefaultSpaceSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #defaultSpaceSettings(DefaultSpaceSettings)
         */
        default Builder defaultSpaceSettings(Consumer<DefaultSpaceSettings.Builder> defaultSpaceSettings) {
            return defaultSpaceSettings(DefaultSpaceSettings.builder().applyMutation(defaultSpaceSettings).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SageMakerRequest.BuilderImpl implements Builder {
        private String domainName;

        private String authMode;

        private UserSettings defaultUserSettings;

        private List<String> subnetIds = DefaultSdkAutoConstructList.getInstance();

        private String vpcId;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String appNetworkAccessType;

        private String homeEfsFileSystemKmsKeyId;

        private String kmsKeyId;

        private String appSecurityGroupManagement;

        private DomainSettings domainSettings;

        private DefaultSpaceSettings defaultSpaceSettings;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateDomainRequest model) {
            super(model);
            domainName(model.domainName);
            authMode(model.authMode);
            defaultUserSettings(model.defaultUserSettings);
            subnetIds(model.subnetIds);
            vpcId(model.vpcId);
            tags(model.tags);
            appNetworkAccessType(model.appNetworkAccessType);
            homeEfsFileSystemKmsKeyId(model.homeEfsFileSystemKmsKeyId);
            kmsKeyId(model.kmsKeyId);
            appSecurityGroupManagement(model.appSecurityGroupManagement);
            domainSettings(model.domainSettings);
            defaultSpaceSettings(model.defaultSpaceSettings);
        }

        public final String getDomainName() {
            return domainName;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        @Override
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final String getAuthMode() {
            return authMode;
        }

        public final void setAuthMode(String authMode) {
            this.authMode = authMode;
        }

        @Override
        public final Builder authMode(String authMode) {
            this.authMode = authMode;
            return this;
        }

        @Override
        public final Builder authMode(AuthMode authMode) {
            this.authMode(authMode == null ? null : authMode.toString());
            return this;
        }

        public final UserSettings.Builder getDefaultUserSettings() {
            return defaultUserSettings != null ? defaultUserSettings.toBuilder() : null;
        }

        public final void setDefaultUserSettings(UserSettings.BuilderImpl defaultUserSettings) {
            this.defaultUserSettings = defaultUserSettings != null ? defaultUserSettings.build() : null;
        }

        @Override
        public final Builder defaultUserSettings(UserSettings defaultUserSettings) {
            this.defaultUserSettings = defaultUserSettings;
            return this;
        }

        public final Collection<String> getSubnetIds() {
            if (subnetIds instanceof SdkAutoConstructList) {
                return null;
            }
            return subnetIds;
        }

        public final void setSubnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetsCopier.copy(subnetIds);
        }

        @Override
        public final Builder subnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetsCopier.copy(subnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetIds(String... subnetIds) {
            subnetIds(Arrays.asList(subnetIds));
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getAppNetworkAccessType() {
            return appNetworkAccessType;
        }

        public final void setAppNetworkAccessType(String appNetworkAccessType) {
            this.appNetworkAccessType = appNetworkAccessType;
        }

        @Override
        public final Builder appNetworkAccessType(String appNetworkAccessType) {
            this.appNetworkAccessType = appNetworkAccessType;
            return this;
        }

        @Override
        public final Builder appNetworkAccessType(AppNetworkAccessType appNetworkAccessType) {
            this.appNetworkAccessType(appNetworkAccessType == null ? null : appNetworkAccessType.toString());
            return this;
        }

        @Deprecated
        public final String getHomeEfsFileSystemKmsKeyId() {
            return homeEfsFileSystemKmsKeyId;
        }

        @Deprecated
        public final void setHomeEfsFileSystemKmsKeyId(String homeEfsFileSystemKmsKeyId) {
            this.homeEfsFileSystemKmsKeyId = homeEfsFileSystemKmsKeyId;
        }

        @Override
        @Deprecated
        public final Builder homeEfsFileSystemKmsKeyId(String homeEfsFileSystemKmsKeyId) {
            this.homeEfsFileSystemKmsKeyId = homeEfsFileSystemKmsKeyId;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final String getAppSecurityGroupManagement() {
            return appSecurityGroupManagement;
        }

        public final void setAppSecurityGroupManagement(String appSecurityGroupManagement) {
            this.appSecurityGroupManagement = appSecurityGroupManagement;
        }

        @Override
        public final Builder appSecurityGroupManagement(String appSecurityGroupManagement) {
            this.appSecurityGroupManagement = appSecurityGroupManagement;
            return this;
        }

        @Override
        public final Builder appSecurityGroupManagement(AppSecurityGroupManagement appSecurityGroupManagement) {
            this.appSecurityGroupManagement(appSecurityGroupManagement == null ? null : appSecurityGroupManagement.toString());
            return this;
        }

        public final DomainSettings.Builder getDomainSettings() {
            return domainSettings != null ? domainSettings.toBuilder() : null;
        }

        public final void setDomainSettings(DomainSettings.BuilderImpl domainSettings) {
            this.domainSettings = domainSettings != null ? domainSettings.build() : null;
        }

        @Override
        public final Builder domainSettings(DomainSettings domainSettings) {
            this.domainSettings = domainSettings;
            return this;
        }

        public final DefaultSpaceSettings.Builder getDefaultSpaceSettings() {
            return defaultSpaceSettings != null ? defaultSpaceSettings.toBuilder() : null;
        }

        public final void setDefaultSpaceSettings(DefaultSpaceSettings.BuilderImpl defaultSpaceSettings) {
            this.defaultSpaceSettings = defaultSpaceSettings != null ? defaultSpaceSettings.build() : null;
        }

        @Override
        public final Builder defaultSpaceSettings(DefaultSpaceSettings defaultSpaceSettings) {
            this.defaultSpaceSettings = defaultSpaceSettings;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateDomainRequest build() {
            return new CreateDomainRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
