/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53recoverycontrolconfig.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A gating rule verifies that a gating routing control or set of gating routing controls, evaluates as true, based on a
 * rule configuration that you specify, which allows a set of routing control state changes to complete.
 * </p>
 * <p>
 * For example, if you specify one gating routing control and you set the Type in the rule configuration to OR, that
 * indicates that you must set the gating routing control to On for the rule to evaluate as true; that is, for the
 * gating control "switch" to be "On". When you do that, then you can update the routing control states for the target
 * routing controls that you specify in the gating rule.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GatingRule implements SdkPojo, Serializable, ToCopyableBuilder<GatingRule.Builder, GatingRule> {
    private static final SdkField<String> CONTROL_PANEL_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ControlPanelArn").getter(getter(GatingRule::controlPanelArn)).setter(setter(Builder::controlPanelArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ControlPanelArn").build()).build();

    private static final SdkField<List<String>> GATING_CONTROLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("GatingControls")
            .getter(getter(GatingRule::gatingControls))
            .setter(setter(Builder::gatingControls))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GatingControls").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(GatingRule::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<RuleConfig> RULE_CONFIG_FIELD = SdkField.<RuleConfig> builder(MarshallingType.SDK_POJO)
            .memberName("RuleConfig").getter(getter(GatingRule::ruleConfig)).setter(setter(Builder::ruleConfig))
            .constructor(RuleConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleConfig").build()).build();

    private static final SdkField<String> SAFETY_RULE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SafetyRuleArn").getter(getter(GatingRule::safetyRuleArn)).setter(setter(Builder::safetyRuleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SafetyRuleArn").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(GatingRule::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<List<String>> TARGET_CONTROLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TargetControls")
            .getter(getter(GatingRule::targetControls))
            .setter(setter(Builder::targetControls))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetControls").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> WAIT_PERIOD_MS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("WaitPeriodMs").getter(getter(GatingRule::waitPeriodMs)).setter(setter(Builder::waitPeriodMs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WaitPeriodMs").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTROL_PANEL_ARN_FIELD,
            GATING_CONTROLS_FIELD, NAME_FIELD, RULE_CONFIG_FIELD, SAFETY_RULE_ARN_FIELD, STATUS_FIELD, TARGET_CONTROLS_FIELD,
            WAIT_PERIOD_MS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String controlPanelArn;

    private final List<String> gatingControls;

    private final String name;

    private final RuleConfig ruleConfig;

    private final String safetyRuleArn;

    private final String status;

    private final List<String> targetControls;

    private final Integer waitPeriodMs;

    private GatingRule(BuilderImpl builder) {
        this.controlPanelArn = builder.controlPanelArn;
        this.gatingControls = builder.gatingControls;
        this.name = builder.name;
        this.ruleConfig = builder.ruleConfig;
        this.safetyRuleArn = builder.safetyRuleArn;
        this.status = builder.status;
        this.targetControls = builder.targetControls;
        this.waitPeriodMs = builder.waitPeriodMs;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the control panel.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the control panel.
     */
    public final String controlPanelArn() {
        return controlPanelArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the GatingControls property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasGatingControls() {
        return gatingControls != null && !(gatingControls instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of gating routing control Amazon Resource Names (ARNs). For a simple "on/off" switch, specify the ARN
     * for one routing control. The gating routing controls are evaluated by the rule configuration that you specify to
     * determine if the target routing control states can be changed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGatingControls} method.
     * </p>
     * 
     * @return An array of gating routing control Amazon Resource Names (ARNs). For a simple "on/off" switch, specify
     *         the ARN for one routing control. The gating routing controls are evaluated by the rule configuration that
     *         you specify to determine if the target routing control states can be changed.
     */
    public final List<String> gatingControls() {
        return gatingControls;
    }

    /**
     * <p>
     * The name for the gating rule. You can use any non-white space character in the name.
     * </p>
     * 
     * @return The name for the gating rule. You can use any non-white space character in the name.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The criteria that you set for gating routing controls that designate how many of the routing control states must
     * be ON to allow you to update target routing control states.
     * </p>
     * 
     * @return The criteria that you set for gating routing controls that designate how many of the routing control
     *         states must be ON to allow you to update target routing control states.
     */
    public final RuleConfig ruleConfig() {
        return ruleConfig;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the gating rule.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the gating rule.
     */
    public final String safetyRuleArn() {
        return safetyRuleArn;
    }

    /**
     * <p>
     * The deployment status of a gating rule. Status can be one of the following: PENDING, DEPLOYED, PENDING_DELETION.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link Status#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The deployment status of a gating rule. Status can be one of the following: PENDING, DEPLOYED,
     *         PENDING_DELETION.
     * @see Status
     */
    public final Status status() {
        return Status.fromValue(status);
    }

    /**
     * <p>
     * The deployment status of a gating rule. Status can be one of the following: PENDING, DEPLOYED, PENDING_DELETION.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link Status#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The deployment status of a gating rule. Status can be one of the following: PENDING, DEPLOYED,
     *         PENDING_DELETION.
     * @see Status
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the TargetControls property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTargetControls() {
        return targetControls != null && !(targetControls instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of target routing control Amazon Resource Names (ARNs) for which the states can only be updated if the
     * rule configuration that you specify evaluates to true for the gating routing control. As a simple example, if you
     * have a single gating control, it acts as an overall "on/off" switch for a set of target routing controls. You can
     * use this to manually override automated failover, for example.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargetControls} method.
     * </p>
     * 
     * @return An array of target routing control Amazon Resource Names (ARNs) for which the states can only be updated
     *         if the rule configuration that you specify evaluates to true for the gating routing control. As a simple
     *         example, if you have a single gating control, it acts as an overall "on/off" switch for a set of target
     *         routing controls. You can use this to manually override automated failover, for example.
     */
    public final List<String> targetControls() {
        return targetControls;
    }

    /**
     * <p>
     * An evaluation period, in milliseconds (ms), during which any request against the target routing controls will
     * fail. This helps prevent "flapping" of state. The wait period is 5000 ms by default, but you can choose a custom
     * value.
     * </p>
     * 
     * @return An evaluation period, in milliseconds (ms), during which any request against the target routing controls
     *         will fail. This helps prevent "flapping" of state. The wait period is 5000 ms by default, but you can
     *         choose a custom value.
     */
    public final Integer waitPeriodMs() {
        return waitPeriodMs;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(controlPanelArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasGatingControls() ? gatingControls() : null);
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(ruleConfig());
        hashCode = 31 * hashCode + Objects.hashCode(safetyRuleArn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargetControls() ? targetControls() : null);
        hashCode = 31 * hashCode + Objects.hashCode(waitPeriodMs());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GatingRule)) {
            return false;
        }
        GatingRule other = (GatingRule) obj;
        return Objects.equals(controlPanelArn(), other.controlPanelArn()) && hasGatingControls() == other.hasGatingControls()
                && Objects.equals(gatingControls(), other.gatingControls()) && Objects.equals(name(), other.name())
                && Objects.equals(ruleConfig(), other.ruleConfig()) && Objects.equals(safetyRuleArn(), other.safetyRuleArn())
                && Objects.equals(statusAsString(), other.statusAsString()) && hasTargetControls() == other.hasTargetControls()
                && Objects.equals(targetControls(), other.targetControls())
                && Objects.equals(waitPeriodMs(), other.waitPeriodMs());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GatingRule").add("ControlPanelArn", controlPanelArn())
                .add("GatingControls", hasGatingControls() ? gatingControls() : null).add("Name", name())
                .add("RuleConfig", ruleConfig()).add("SafetyRuleArn", safetyRuleArn()).add("Status", statusAsString())
                .add("TargetControls", hasTargetControls() ? targetControls() : null).add("WaitPeriodMs", waitPeriodMs()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ControlPanelArn":
            return Optional.ofNullable(clazz.cast(controlPanelArn()));
        case "GatingControls":
            return Optional.ofNullable(clazz.cast(gatingControls()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "RuleConfig":
            return Optional.ofNullable(clazz.cast(ruleConfig()));
        case "SafetyRuleArn":
            return Optional.ofNullable(clazz.cast(safetyRuleArn()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "TargetControls":
            return Optional.ofNullable(clazz.cast(targetControls()));
        case "WaitPeriodMs":
            return Optional.ofNullable(clazz.cast(waitPeriodMs()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GatingRule, T> g) {
        return obj -> g.apply((GatingRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GatingRule> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the control panel.
         * </p>
         * 
         * @param controlPanelArn
         *        The Amazon Resource Name (ARN) of the control panel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder controlPanelArn(String controlPanelArn);

        /**
         * <p>
         * An array of gating routing control Amazon Resource Names (ARNs). For a simple "on/off" switch, specify the
         * ARN for one routing control. The gating routing controls are evaluated by the rule configuration that you
         * specify to determine if the target routing control states can be changed.
         * </p>
         * 
         * @param gatingControls
         *        An array of gating routing control Amazon Resource Names (ARNs). For a simple "on/off" switch, specify
         *        the ARN for one routing control. The gating routing controls are evaluated by the rule configuration
         *        that you specify to determine if the target routing control states can be changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gatingControls(Collection<String> gatingControls);

        /**
         * <p>
         * An array of gating routing control Amazon Resource Names (ARNs). For a simple "on/off" switch, specify the
         * ARN for one routing control. The gating routing controls are evaluated by the rule configuration that you
         * specify to determine if the target routing control states can be changed.
         * </p>
         * 
         * @param gatingControls
         *        An array of gating routing control Amazon Resource Names (ARNs). For a simple "on/off" switch, specify
         *        the ARN for one routing control. The gating routing controls are evaluated by the rule configuration
         *        that you specify to determine if the target routing control states can be changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gatingControls(String... gatingControls);

        /**
         * <p>
         * The name for the gating rule. You can use any non-white space character in the name.
         * </p>
         * 
         * @param name
         *        The name for the gating rule. You can use any non-white space character in the name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The criteria that you set for gating routing controls that designate how many of the routing control states
         * must be ON to allow you to update target routing control states.
         * </p>
         * 
         * @param ruleConfig
         *        The criteria that you set for gating routing controls that designate how many of the routing control
         *        states must be ON to allow you to update target routing control states.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleConfig(RuleConfig ruleConfig);

        /**
         * <p>
         * The criteria that you set for gating routing controls that designate how many of the routing control states
         * must be ON to allow you to update target routing control states.
         * </p>
         * This is a convenience method that creates an instance of the {@link RuleConfig.Builder} avoiding the need to
         * create one manually via {@link RuleConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RuleConfig.Builder#build()} is called immediately and its result
         * is passed to {@link #ruleConfig(RuleConfig)}.
         * 
         * @param ruleConfig
         *        a consumer that will call methods on {@link RuleConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ruleConfig(RuleConfig)
         */
        default Builder ruleConfig(Consumer<RuleConfig.Builder> ruleConfig) {
            return ruleConfig(RuleConfig.builder().applyMutation(ruleConfig).build());
        }

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the gating rule.
         * </p>
         * 
         * @param safetyRuleArn
         *        The Amazon Resource Name (ARN) of the gating rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder safetyRuleArn(String safetyRuleArn);

        /**
         * <p>
         * The deployment status of a gating rule. Status can be one of the following: PENDING, DEPLOYED,
         * PENDING_DELETION.
         * </p>
         * 
         * @param status
         *        The deployment status of a gating rule. Status can be one of the following: PENDING, DEPLOYED,
         *        PENDING_DELETION.
         * @see Status
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Status
         */
        Builder status(String status);

        /**
         * <p>
         * The deployment status of a gating rule. Status can be one of the following: PENDING, DEPLOYED,
         * PENDING_DELETION.
         * </p>
         * 
         * @param status
         *        The deployment status of a gating rule. Status can be one of the following: PENDING, DEPLOYED,
         *        PENDING_DELETION.
         * @see Status
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Status
         */
        Builder status(Status status);

        /**
         * <p>
         * An array of target routing control Amazon Resource Names (ARNs) for which the states can only be updated if
         * the rule configuration that you specify evaluates to true for the gating routing control. As a simple
         * example, if you have a single gating control, it acts as an overall "on/off" switch for a set of target
         * routing controls. You can use this to manually override automated failover, for example.
         * </p>
         * 
         * @param targetControls
         *        An array of target routing control Amazon Resource Names (ARNs) for which the states can only be
         *        updated if the rule configuration that you specify evaluates to true for the gating routing control.
         *        As a simple example, if you have a single gating control, it acts as an overall "on/off" switch for a
         *        set of target routing controls. You can use this to manually override automated failover, for example.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetControls(Collection<String> targetControls);

        /**
         * <p>
         * An array of target routing control Amazon Resource Names (ARNs) for which the states can only be updated if
         * the rule configuration that you specify evaluates to true for the gating routing control. As a simple
         * example, if you have a single gating control, it acts as an overall "on/off" switch for a set of target
         * routing controls. You can use this to manually override automated failover, for example.
         * </p>
         * 
         * @param targetControls
         *        An array of target routing control Amazon Resource Names (ARNs) for which the states can only be
         *        updated if the rule configuration that you specify evaluates to true for the gating routing control.
         *        As a simple example, if you have a single gating control, it acts as an overall "on/off" switch for a
         *        set of target routing controls. You can use this to manually override automated failover, for example.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetControls(String... targetControls);

        /**
         * <p>
         * An evaluation period, in milliseconds (ms), during which any request against the target routing controls will
         * fail. This helps prevent "flapping" of state. The wait period is 5000 ms by default, but you can choose a
         * custom value.
         * </p>
         * 
         * @param waitPeriodMs
         *        An evaluation period, in milliseconds (ms), during which any request against the target routing
         *        controls will fail. This helps prevent "flapping" of state. The wait period is 5000 ms by default, but
         *        you can choose a custom value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder waitPeriodMs(Integer waitPeriodMs);
    }

    static final class BuilderImpl implements Builder {
        private String controlPanelArn;

        private List<String> gatingControls = DefaultSdkAutoConstructList.getInstance();

        private String name;

        private RuleConfig ruleConfig;

        private String safetyRuleArn;

        private String status;

        private List<String> targetControls = DefaultSdkAutoConstructList.getInstance();

        private Integer waitPeriodMs;

        private BuilderImpl() {
        }

        private BuilderImpl(GatingRule model) {
            controlPanelArn(model.controlPanelArn);
            gatingControls(model.gatingControls);
            name(model.name);
            ruleConfig(model.ruleConfig);
            safetyRuleArn(model.safetyRuleArn);
            status(model.status);
            targetControls(model.targetControls);
            waitPeriodMs(model.waitPeriodMs);
        }

        public final String getControlPanelArn() {
            return controlPanelArn;
        }

        public final void setControlPanelArn(String controlPanelArn) {
            this.controlPanelArn = controlPanelArn;
        }

        @Override
        public final Builder controlPanelArn(String controlPanelArn) {
            this.controlPanelArn = controlPanelArn;
            return this;
        }

        public final Collection<String> getGatingControls() {
            if (gatingControls instanceof SdkAutoConstructList) {
                return null;
            }
            return gatingControls;
        }

        public final void setGatingControls(Collection<String> gatingControls) {
            this.gatingControls = ___listOf__stringMin1Max256PatternAZaZ09Copier.copy(gatingControls);
        }

        @Override
        public final Builder gatingControls(Collection<String> gatingControls) {
            this.gatingControls = ___listOf__stringMin1Max256PatternAZaZ09Copier.copy(gatingControls);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder gatingControls(String... gatingControls) {
            gatingControls(Arrays.asList(gatingControls));
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final RuleConfig.Builder getRuleConfig() {
            return ruleConfig != null ? ruleConfig.toBuilder() : null;
        }

        public final void setRuleConfig(RuleConfig.BuilderImpl ruleConfig) {
            this.ruleConfig = ruleConfig != null ? ruleConfig.build() : null;
        }

        @Override
        public final Builder ruleConfig(RuleConfig ruleConfig) {
            this.ruleConfig = ruleConfig;
            return this;
        }

        public final String getSafetyRuleArn() {
            return safetyRuleArn;
        }

        public final void setSafetyRuleArn(String safetyRuleArn) {
            this.safetyRuleArn = safetyRuleArn;
        }

        @Override
        public final Builder safetyRuleArn(String safetyRuleArn) {
            this.safetyRuleArn = safetyRuleArn;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(Status status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Collection<String> getTargetControls() {
            if (targetControls instanceof SdkAutoConstructList) {
                return null;
            }
            return targetControls;
        }

        public final void setTargetControls(Collection<String> targetControls) {
            this.targetControls = ___listOf__stringMin1Max256PatternAZaZ09Copier.copy(targetControls);
        }

        @Override
        public final Builder targetControls(Collection<String> targetControls) {
            this.targetControls = ___listOf__stringMin1Max256PatternAZaZ09Copier.copy(targetControls);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetControls(String... targetControls) {
            targetControls(Arrays.asList(targetControls));
            return this;
        }

        public final Integer getWaitPeriodMs() {
            return waitPeriodMs;
        }

        public final void setWaitPeriodMs(Integer waitPeriodMs) {
            this.waitPeriodMs = waitPeriodMs;
        }

        @Override
        public final Builder waitPeriodMs(Integer waitPeriodMs) {
            this.waitPeriodMs = waitPeriodMs;
            return this;
        }

        @Override
        public GatingRule build() {
            return new GatingRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
