/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rds.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p/>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribeEventsRequest extends RdsRequest implements
        ToCopyableBuilder<DescribeEventsRequest.Builder, DescribeEventsRequest> {
    private static final SdkField<String> SOURCE_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceIdentifier").getter(getter(DescribeEventsRequest::sourceIdentifier))
            .setter(setter(Builder::sourceIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceIdentifier").build()).build();

    private static final SdkField<String> SOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceType").getter(getter(DescribeEventsRequest::sourceTypeAsString))
            .setter(setter(Builder::sourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceType").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(DescribeEventsRequest::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTime").getter(getter(DescribeEventsRequest::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final SdkField<Integer> DURATION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Duration").getter(getter(DescribeEventsRequest::duration)).setter(setter(Builder::duration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Duration").build()).build();

    private static final SdkField<List<String>> EVENT_CATEGORIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EventCategories")
            .getter(getter(DescribeEventsRequest::eventCategories))
            .setter(setter(Builder::eventCategories))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventCategories").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("EventCategory")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("EventCategory").build()).build()).build()).build();

    private static final SdkField<List<Filter>> FILTERS_FIELD = SdkField
            .<List<Filter>> builder(MarshallingType.LIST)
            .memberName("Filters")
            .getter(getter(DescribeEventsRequest::filters))
            .setter(setter(Builder::filters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Filters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Filter")
                            .memberFieldInfo(
                                    SdkField.<Filter> builder(MarshallingType.SDK_POJO)
                                            .constructor(Filter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Filter").build()).build()).build()).build();

    private static final SdkField<Integer> MAX_RECORDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxRecords").getter(getter(DescribeEventsRequest::maxRecords)).setter(setter(Builder::maxRecords))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxRecords").build()).build();

    private static final SdkField<String> MARKER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Marker")
            .getter(getter(DescribeEventsRequest::marker)).setter(setter(Builder::marker))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Marker").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SOURCE_IDENTIFIER_FIELD,
            SOURCE_TYPE_FIELD, START_TIME_FIELD, END_TIME_FIELD, DURATION_FIELD, EVENT_CATEGORIES_FIELD, FILTERS_FIELD,
            MAX_RECORDS_FIELD, MARKER_FIELD));

    private final String sourceIdentifier;

    private final String sourceType;

    private final Instant startTime;

    private final Instant endTime;

    private final Integer duration;

    private final List<String> eventCategories;

    private final List<Filter> filters;

    private final Integer maxRecords;

    private final String marker;

    private DescribeEventsRequest(BuilderImpl builder) {
        super(builder);
        this.sourceIdentifier = builder.sourceIdentifier;
        this.sourceType = builder.sourceType;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.duration = builder.duration;
        this.eventCategories = builder.eventCategories;
        this.filters = builder.filters;
        this.maxRecords = builder.maxRecords;
        this.marker = builder.marker;
    }

    /**
     * <p>
     * The identifier of the event source for which events are returned. If not specified, then all sources are included
     * in the response.
     * </p>
     * <p>
     * Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If <code>SourceIdentifier</code> is supplied, <code>SourceType</code> must also be provided.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the source type is a DB instance, a <code>DBInstanceIdentifier</code> value must be supplied.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the source type is a DB cluster, a <code>DBClusterIdentifier</code> value must be supplied.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the source type is a DB parameter group, a <code>DBParameterGroupName</code> value must be supplied.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the source type is a DB security group, a <code>DBSecurityGroupName</code> value must be supplied.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the source type is a DB snapshot, a <code>DBSnapshotIdentifier</code> value must be supplied.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the source type is a DB cluster snapshot, a <code>DBClusterSnapshotIdentifier</code> value must be supplied.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the source type is an RDS Proxy, a <code>DBProxyName</code> value must be supplied.
     * </p>
     * </li>
     * <li>
     * <p>
     * Can't end with a hyphen or contain two consecutive hyphens.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The identifier of the event source for which events are returned. If not specified, then all sources are
     *         included in the response.</p>
     *         <p>
     *         Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If <code>SourceIdentifier</code> is supplied, <code>SourceType</code> must also be provided.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the source type is a DB instance, a <code>DBInstanceIdentifier</code> value must be supplied.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the source type is a DB cluster, a <code>DBClusterIdentifier</code> value must be supplied.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the source type is a DB parameter group, a <code>DBParameterGroupName</code> value must be supplied.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the source type is a DB security group, a <code>DBSecurityGroupName</code> value must be supplied.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the source type is a DB snapshot, a <code>DBSnapshotIdentifier</code> value must be supplied.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the source type is a DB cluster snapshot, a <code>DBClusterSnapshotIdentifier</code> value must be
     *         supplied.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the source type is an RDS Proxy, a <code>DBProxyName</code> value must be supplied.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Can't end with a hyphen or contain two consecutive hyphens.
     *         </p>
     *         </li>
     */
    public final String sourceIdentifier() {
        return sourceIdentifier;
    }

    /**
     * <p>
     * The event source to retrieve events for. If no value is specified, all events are returned.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sourceType} will
     * return {@link SourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sourceTypeAsString}.
     * </p>
     * 
     * @return The event source to retrieve events for. If no value is specified, all events are returned.
     * @see SourceType
     */
    public final SourceType sourceType() {
        return SourceType.fromValue(sourceType);
    }

    /**
     * <p>
     * The event source to retrieve events for. If no value is specified, all events are returned.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sourceType} will
     * return {@link SourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sourceTypeAsString}.
     * </p>
     * 
     * @return The event source to retrieve events for. If no value is specified, all events are returned.
     * @see SourceType
     */
    public final String sourceTypeAsString() {
        return sourceType;
    }

    /**
     * <p>
     * The beginning of the time interval to retrieve events for, specified in ISO 8601 format. For more information
     * about ISO 8601, go to the <a href="http://en.wikipedia.org/wiki/ISO_8601">ISO8601 Wikipedia page.</a>
     * </p>
     * <p>
     * Example: 2009-07-08T18:00Z
     * </p>
     * 
     * @return The beginning of the time interval to retrieve events for, specified in ISO 8601 format. For more
     *         information about ISO 8601, go to the <a href="http://en.wikipedia.org/wiki/ISO_8601">ISO8601 Wikipedia
     *         page.</a> </p>
     *         <p>
     *         Example: 2009-07-08T18:00Z
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The end of the time interval for which to retrieve events, specified in ISO 8601 format. For more information
     * about ISO 8601, go to the <a href="http://en.wikipedia.org/wiki/ISO_8601">ISO8601 Wikipedia page.</a>
     * </p>
     * <p>
     * Example: 2009-07-08T18:00Z
     * </p>
     * 
     * @return The end of the time interval for which to retrieve events, specified in ISO 8601 format. For more
     *         information about ISO 8601, go to the <a href="http://en.wikipedia.org/wiki/ISO_8601">ISO8601 Wikipedia
     *         page.</a> </p>
     *         <p>
     *         Example: 2009-07-08T18:00Z
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The number of minutes to retrieve events for.
     * </p>
     * <p>
     * Default: 60
     * </p>
     * 
     * @return The number of minutes to retrieve events for.</p>
     *         <p>
     *         Default: 60
     */
    public final Integer duration() {
        return duration;
    }

    /**
     * For responses, this returns true if the service returned a value for the EventCategories property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEventCategories() {
        return eventCategories != null && !(eventCategories instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of event categories that trigger notifications for a event notification subscription.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEventCategories} method.
     * </p>
     * 
     * @return A list of event categories that trigger notifications for a event notification subscription.
     */
    public final List<String> eventCategories() {
        return eventCategories;
    }

    /**
     * For responses, this returns true if the service returned a value for the Filters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasFilters() {
        return filters != null && !(filters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * This parameter isn't currently supported.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFilters} method.
     * </p>
     * 
     * @return This parameter isn't currently supported.
     */
    public final List<Filter> filters() {
        return filters;
    }

    /**
     * <p>
     * The maximum number of records to include in the response. If more records exist than the specified
     * <code>MaxRecords</code> value, a pagination token called a marker is included in the response so that you can
     * retrieve the remaining results.
     * </p>
     * <p>
     * Default: 100
     * </p>
     * <p>
     * Constraints: Minimum 20, maximum 100.
     * </p>
     * 
     * @return The maximum number of records to include in the response. If more records exist than the specified
     *         <code>MaxRecords</code> value, a pagination token called a marker is included in the response so that you
     *         can retrieve the remaining results.</p>
     *         <p>
     *         Default: 100
     *         </p>
     *         <p>
     *         Constraints: Minimum 20, maximum 100.
     */
    public final Integer maxRecords() {
        return maxRecords;
    }

    /**
     * <p>
     * An optional pagination token provided by a previous DescribeEvents request. If this parameter is specified, the
     * response includes only records beyond the marker, up to the value specified by <code>MaxRecords</code>.
     * </p>
     * 
     * @return An optional pagination token provided by a previous DescribeEvents request. If this parameter is
     *         specified, the response includes only records beyond the marker, up to the value specified by
     *         <code>MaxRecords</code>.
     */
    public final String marker() {
        return marker;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(sourceIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(sourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(duration());
        hashCode = 31 * hashCode + Objects.hashCode(hasEventCategories() ? eventCategories() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasFilters() ? filters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(maxRecords());
        hashCode = 31 * hashCode + Objects.hashCode(marker());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DescribeEventsRequest)) {
            return false;
        }
        DescribeEventsRequest other = (DescribeEventsRequest) obj;
        return Objects.equals(sourceIdentifier(), other.sourceIdentifier())
                && Objects.equals(sourceTypeAsString(), other.sourceTypeAsString())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(duration(), other.duration()) && hasEventCategories() == other.hasEventCategories()
                && Objects.equals(eventCategories(), other.eventCategories()) && hasFilters() == other.hasFilters()
                && Objects.equals(filters(), other.filters()) && Objects.equals(maxRecords(), other.maxRecords())
                && Objects.equals(marker(), other.marker());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DescribeEventsRequest").add("SourceIdentifier", sourceIdentifier())
                .add("SourceType", sourceTypeAsString()).add("StartTime", startTime()).add("EndTime", endTime())
                .add("Duration", duration()).add("EventCategories", hasEventCategories() ? eventCategories() : null)
                .add("Filters", hasFilters() ? filters() : null).add("MaxRecords", maxRecords()).add("Marker", marker()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SourceIdentifier":
            return Optional.ofNullable(clazz.cast(sourceIdentifier()));
        case "SourceType":
            return Optional.ofNullable(clazz.cast(sourceTypeAsString()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "Duration":
            return Optional.ofNullable(clazz.cast(duration()));
        case "EventCategories":
            return Optional.ofNullable(clazz.cast(eventCategories()));
        case "Filters":
            return Optional.ofNullable(clazz.cast(filters()));
        case "MaxRecords":
            return Optional.ofNullable(clazz.cast(maxRecords()));
        case "Marker":
            return Optional.ofNullable(clazz.cast(marker()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DescribeEventsRequest, T> g) {
        return obj -> g.apply((DescribeEventsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends RdsRequest.Builder, SdkPojo, CopyableBuilder<Builder, DescribeEventsRequest> {
        /**
         * <p>
         * The identifier of the event source for which events are returned. If not specified, then all sources are
         * included in the response.
         * </p>
         * <p>
         * Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If <code>SourceIdentifier</code> is supplied, <code>SourceType</code> must also be provided.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the source type is a DB instance, a <code>DBInstanceIdentifier</code> value must be supplied.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the source type is a DB cluster, a <code>DBClusterIdentifier</code> value must be supplied.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the source type is a DB parameter group, a <code>DBParameterGroupName</code> value must be supplied.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the source type is a DB security group, a <code>DBSecurityGroupName</code> value must be supplied.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the source type is a DB snapshot, a <code>DBSnapshotIdentifier</code> value must be supplied.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the source type is a DB cluster snapshot, a <code>DBClusterSnapshotIdentifier</code> value must be
         * supplied.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the source type is an RDS Proxy, a <code>DBProxyName</code> value must be supplied.
         * </p>
         * </li>
         * <li>
         * <p>
         * Can't end with a hyphen or contain two consecutive hyphens.
         * </p>
         * </li>
         * </ul>
         * 
         * @param sourceIdentifier
         *        The identifier of the event source for which events are returned. If not specified, then all sources
         *        are included in the response.</p>
         *        <p>
         *        Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If <code>SourceIdentifier</code> is supplied, <code>SourceType</code> must also be provided.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the source type is a DB instance, a <code>DBInstanceIdentifier</code> value must be supplied.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the source type is a DB cluster, a <code>DBClusterIdentifier</code> value must be supplied.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the source type is a DB parameter group, a <code>DBParameterGroupName</code> value must be
         *        supplied.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the source type is a DB security group, a <code>DBSecurityGroupName</code> value must be supplied.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the source type is a DB snapshot, a <code>DBSnapshotIdentifier</code> value must be supplied.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the source type is a DB cluster snapshot, a <code>DBClusterSnapshotIdentifier</code> value must be
         *        supplied.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the source type is an RDS Proxy, a <code>DBProxyName</code> value must be supplied.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Can't end with a hyphen or contain two consecutive hyphens.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceIdentifier(String sourceIdentifier);

        /**
         * <p>
         * The event source to retrieve events for. If no value is specified, all events are returned.
         * </p>
         * 
         * @param sourceType
         *        The event source to retrieve events for. If no value is specified, all events are returned.
         * @see SourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SourceType
         */
        Builder sourceType(String sourceType);

        /**
         * <p>
         * The event source to retrieve events for. If no value is specified, all events are returned.
         * </p>
         * 
         * @param sourceType
         *        The event source to retrieve events for. If no value is specified, all events are returned.
         * @see SourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SourceType
         */
        Builder sourceType(SourceType sourceType);

        /**
         * <p>
         * The beginning of the time interval to retrieve events for, specified in ISO 8601 format. For more information
         * about ISO 8601, go to the <a href="http://en.wikipedia.org/wiki/ISO_8601">ISO8601 Wikipedia page.</a>
         * </p>
         * <p>
         * Example: 2009-07-08T18:00Z
         * </p>
         * 
         * @param startTime
         *        The beginning of the time interval to retrieve events for, specified in ISO 8601 format. For more
         *        information about ISO 8601, go to the <a href="http://en.wikipedia.org/wiki/ISO_8601">ISO8601
         *        Wikipedia page.</a> </p>
         *        <p>
         *        Example: 2009-07-08T18:00Z
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The end of the time interval for which to retrieve events, specified in ISO 8601 format. For more information
         * about ISO 8601, go to the <a href="http://en.wikipedia.org/wiki/ISO_8601">ISO8601 Wikipedia page.</a>
         * </p>
         * <p>
         * Example: 2009-07-08T18:00Z
         * </p>
         * 
         * @param endTime
         *        The end of the time interval for which to retrieve events, specified in ISO 8601 format. For more
         *        information about ISO 8601, go to the <a href="http://en.wikipedia.org/wiki/ISO_8601">ISO8601
         *        Wikipedia page.</a> </p>
         *        <p>
         *        Example: 2009-07-08T18:00Z
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The number of minutes to retrieve events for.
         * </p>
         * <p>
         * Default: 60
         * </p>
         * 
         * @param duration
         *        The number of minutes to retrieve events for.</p>
         *        <p>
         *        Default: 60
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder duration(Integer duration);

        /**
         * <p>
         * A list of event categories that trigger notifications for a event notification subscription.
         * </p>
         * 
         * @param eventCategories
         *        A list of event categories that trigger notifications for a event notification subscription.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventCategories(Collection<String> eventCategories);

        /**
         * <p>
         * A list of event categories that trigger notifications for a event notification subscription.
         * </p>
         * 
         * @param eventCategories
         *        A list of event categories that trigger notifications for a event notification subscription.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventCategories(String... eventCategories);

        /**
         * <p>
         * This parameter isn't currently supported.
         * </p>
         * 
         * @param filters
         *        This parameter isn't currently supported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(Collection<Filter> filters);

        /**
         * <p>
         * This parameter isn't currently supported.
         * </p>
         * 
         * @param filters
         *        This parameter isn't currently supported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(Filter... filters);

        /**
         * <p>
         * This parameter isn't currently supported.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.rds.model.Filter.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.rds.model.Filter#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.rds.model.Filter.Builder#build()}
         * is called immediately and its result is passed to {@link #filters(List<Filter>)}.
         * 
         * @param filters
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.rds.model.Filter.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filters(java.util.Collection<Filter>)
         */
        Builder filters(Consumer<Filter.Builder>... filters);

        /**
         * <p>
         * The maximum number of records to include in the response. If more records exist than the specified
         * <code>MaxRecords</code> value, a pagination token called a marker is included in the response so that you can
         * retrieve the remaining results.
         * </p>
         * <p>
         * Default: 100
         * </p>
         * <p>
         * Constraints: Minimum 20, maximum 100.
         * </p>
         * 
         * @param maxRecords
         *        The maximum number of records to include in the response. If more records exist than the specified
         *        <code>MaxRecords</code> value, a pagination token called a marker is included in the response so that
         *        you can retrieve the remaining results.</p>
         *        <p>
         *        Default: 100
         *        </p>
         *        <p>
         *        Constraints: Minimum 20, maximum 100.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxRecords(Integer maxRecords);

        /**
         * <p>
         * An optional pagination token provided by a previous DescribeEvents request. If this parameter is specified,
         * the response includes only records beyond the marker, up to the value specified by <code>MaxRecords</code>.
         * </p>
         * 
         * @param marker
         *        An optional pagination token provided by a previous DescribeEvents request. If this parameter is
         *        specified, the response includes only records beyond the marker, up to the value specified by
         *        <code>MaxRecords</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder marker(String marker);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends RdsRequest.BuilderImpl implements Builder {
        private String sourceIdentifier;

        private String sourceType;

        private Instant startTime;

        private Instant endTime;

        private Integer duration;

        private List<String> eventCategories = DefaultSdkAutoConstructList.getInstance();

        private List<Filter> filters = DefaultSdkAutoConstructList.getInstance();

        private Integer maxRecords;

        private String marker;

        private BuilderImpl() {
        }

        private BuilderImpl(DescribeEventsRequest model) {
            super(model);
            sourceIdentifier(model.sourceIdentifier);
            sourceType(model.sourceType);
            startTime(model.startTime);
            endTime(model.endTime);
            duration(model.duration);
            eventCategories(model.eventCategories);
            filters(model.filters);
            maxRecords(model.maxRecords);
            marker(model.marker);
        }

        public final String getSourceIdentifier() {
            return sourceIdentifier;
        }

        public final void setSourceIdentifier(String sourceIdentifier) {
            this.sourceIdentifier = sourceIdentifier;
        }

        @Override
        public final Builder sourceIdentifier(String sourceIdentifier) {
            this.sourceIdentifier = sourceIdentifier;
            return this;
        }

        public final String getSourceType() {
            return sourceType;
        }

        public final void setSourceType(String sourceType) {
            this.sourceType = sourceType;
        }

        @Override
        public final Builder sourceType(String sourceType) {
            this.sourceType = sourceType;
            return this;
        }

        @Override
        public final Builder sourceType(SourceType sourceType) {
            this.sourceType(sourceType == null ? null : sourceType.toString());
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final Integer getDuration() {
            return duration;
        }

        public final void setDuration(Integer duration) {
            this.duration = duration;
        }

        @Override
        public final Builder duration(Integer duration) {
            this.duration = duration;
            return this;
        }

        public final Collection<String> getEventCategories() {
            if (eventCategories instanceof SdkAutoConstructList) {
                return null;
            }
            return eventCategories;
        }

        public final void setEventCategories(Collection<String> eventCategories) {
            this.eventCategories = EventCategoriesListCopier.copy(eventCategories);
        }

        @Override
        public final Builder eventCategories(Collection<String> eventCategories) {
            this.eventCategories = EventCategoriesListCopier.copy(eventCategories);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder eventCategories(String... eventCategories) {
            eventCategories(Arrays.asList(eventCategories));
            return this;
        }

        public final List<Filter.Builder> getFilters() {
            List<Filter.Builder> result = FilterListCopier.copyToBuilder(this.filters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFilters(Collection<Filter.BuilderImpl> filters) {
            this.filters = FilterListCopier.copyFromBuilder(filters);
        }

        @Override
        public final Builder filters(Collection<Filter> filters) {
            this.filters = FilterListCopier.copy(filters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(Filter... filters) {
            filters(Arrays.asList(filters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(Consumer<Filter.Builder>... filters) {
            filters(Stream.of(filters).map(c -> Filter.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Integer getMaxRecords() {
            return maxRecords;
        }

        public final void setMaxRecords(Integer maxRecords) {
            this.maxRecords = maxRecords;
        }

        @Override
        public final Builder maxRecords(Integer maxRecords) {
            this.maxRecords = maxRecords;
            return this;
        }

        public final String getMarker() {
            return marker;
        }

        public final void setMarker(String marker) {
            this.marker = marker;
        }

        @Override
        public final Builder marker(String marker) {
            this.marker = marker;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public DescribeEventsRequest build() {
            return new DescribeEventsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
