/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rds.model;

import java.beans.Transient;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p/>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CopyDbSnapshotRequest extends RdsRequest implements
        ToCopyableBuilder<CopyDbSnapshotRequest.Builder, CopyDbSnapshotRequest> {
    private static final SdkField<String> SOURCE_DB_SNAPSHOT_IDENTIFIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SourceDBSnapshotIdentifier")
            .getter(getter(CopyDbSnapshotRequest::sourceDBSnapshotIdentifier))
            .setter(setter(Builder::sourceDBSnapshotIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceDBSnapshotIdentifier").build())
            .build();

    private static final SdkField<String> TARGET_DB_SNAPSHOT_IDENTIFIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TargetDBSnapshotIdentifier")
            .getter(getter(CopyDbSnapshotRequest::targetDBSnapshotIdentifier))
            .setter(setter(Builder::targetDBSnapshotIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetDBSnapshotIdentifier").build())
            .build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(CopyDbSnapshotRequest::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CopyDbSnapshotRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Tag")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Tag").build()).build()).build()).build();

    private static final SdkField<Boolean> COPY_TAGS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CopyTags").getter(getter(CopyDbSnapshotRequest::copyTags)).setter(setter(Builder::copyTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopyTags").build()).build();

    private static final SdkField<String> PRE_SIGNED_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreSignedUrl").getter(getter(CopyDbSnapshotRequest::preSignedUrl)).setter(setter(Builder::preSignedUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreSignedUrl").build()).build();

    private static final SdkField<String> OPTION_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OptionGroupName").getter(getter(CopyDbSnapshotRequest::optionGroupName))
            .setter(setter(Builder::optionGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OptionGroupName").build()).build();

    private static final SdkField<String> TARGET_CUSTOM_AVAILABILITY_ZONE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TargetCustomAvailabilityZone")
            .getter(getter(CopyDbSnapshotRequest::targetCustomAvailabilityZone))
            .setter(setter(Builder::targetCustomAvailabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetCustomAvailabilityZone")
                    .build()).build();

    private static final SdkField<String> SOURCE_REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceRegion").getter(getter(CopyDbSnapshotRequest::sourceRegion)).setter(setter(Builder::sourceRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceRegion").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            SOURCE_DB_SNAPSHOT_IDENTIFIER_FIELD, TARGET_DB_SNAPSHOT_IDENTIFIER_FIELD, KMS_KEY_ID_FIELD, TAGS_FIELD,
            COPY_TAGS_FIELD, PRE_SIGNED_URL_FIELD, OPTION_GROUP_NAME_FIELD, TARGET_CUSTOM_AVAILABILITY_ZONE_FIELD,
            SOURCE_REGION_FIELD));

    private final String sourceDBSnapshotIdentifier;

    private final String targetDBSnapshotIdentifier;

    private final String kmsKeyId;

    private final List<Tag> tags;

    private final Boolean copyTags;

    private final String preSignedUrl;

    private final String optionGroupName;

    private final String targetCustomAvailabilityZone;

    private final String sourceRegion;

    private CopyDbSnapshotRequest(BuilderImpl builder) {
        super(builder);
        this.sourceDBSnapshotIdentifier = builder.sourceDBSnapshotIdentifier;
        this.targetDBSnapshotIdentifier = builder.targetDBSnapshotIdentifier;
        this.kmsKeyId = builder.kmsKeyId;
        this.tags = builder.tags;
        this.copyTags = builder.copyTags;
        this.preSignedUrl = builder.preSignedUrl;
        this.optionGroupName = builder.optionGroupName;
        this.targetCustomAvailabilityZone = builder.targetCustomAvailabilityZone;
        this.sourceRegion = builder.sourceRegion;
    }

    /**
     * <p>
     * The identifier for the source DB snapshot.
     * </p>
     * <p>
     * If the source snapshot is in the same Amazon Web Services Region as the copy, specify a valid DB snapshot
     * identifier. For example, you might specify <code>rds:mysql-instance1-snapshot-20130805</code>.
     * </p>
     * <p>
     * If the source snapshot is in a different Amazon Web Services Region than the copy, specify a valid DB snapshot
     * ARN. For example, you might specify
     * <code>arn:aws:rds:us-west-2:123456789012:snapshot:mysql-instance1-snapshot-20130805</code>.
     * </p>
     * <p>
     * If you are copying from a shared manual DB snapshot, this parameter must be the Amazon Resource Name (ARN) of the
     * shared DB snapshot.
     * </p>
     * <p>
     * If you are copying an encrypted snapshot this parameter must be in the ARN format for the source Amazon Web
     * Services Region, and must match the <code>SourceDBSnapshotIdentifier</code> in the <code>PreSignedUrl</code>
     * parameter.
     * </p>
     * <p>
     * Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must specify a valid system snapshot in the "available" state.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Example: <code>rds:mydb-2012-04-02-00-01</code>
     * </p>
     * <p>
     * Example: <code>arn:aws:rds:us-west-2:123456789012:snapshot:mysql-instance1-snapshot-20130805</code>
     * </p>
     * 
     * @return The identifier for the source DB snapshot.</p>
     *         <p>
     *         If the source snapshot is in the same Amazon Web Services Region as the copy, specify a valid DB snapshot
     *         identifier. For example, you might specify <code>rds:mysql-instance1-snapshot-20130805</code>.
     *         </p>
     *         <p>
     *         If the source snapshot is in a different Amazon Web Services Region than the copy, specify a valid DB
     *         snapshot ARN. For example, you might specify
     *         <code>arn:aws:rds:us-west-2:123456789012:snapshot:mysql-instance1-snapshot-20130805</code>.
     *         </p>
     *         <p>
     *         If you are copying from a shared manual DB snapshot, this parameter must be the Amazon Resource Name
     *         (ARN) of the shared DB snapshot.
     *         </p>
     *         <p>
     *         If you are copying an encrypted snapshot this parameter must be in the ARN format for the source Amazon
     *         Web Services Region, and must match the <code>SourceDBSnapshotIdentifier</code> in the
     *         <code>PreSignedUrl</code> parameter.
     *         </p>
     *         <p>
     *         Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Must specify a valid system snapshot in the "available" state.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Example: <code>rds:mydb-2012-04-02-00-01</code>
     *         </p>
     *         <p>
     *         Example: <code>arn:aws:rds:us-west-2:123456789012:snapshot:mysql-instance1-snapshot-20130805</code>
     */
    public final String sourceDBSnapshotIdentifier() {
        return sourceDBSnapshotIdentifier;
    }

    /**
     * <p>
     * The identifier for the copy of the snapshot.
     * </p>
     * <p>
     * Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Can't be null, empty, or blank
     * </p>
     * </li>
     * <li>
     * <p>
     * Must contain from 1 to 255 letters, numbers, or hyphens
     * </p>
     * </li>
     * <li>
     * <p>
     * First character must be a letter
     * </p>
     * </li>
     * <li>
     * <p>
     * Can't end with a hyphen or contain two consecutive hyphens
     * </p>
     * </li>
     * </ul>
     * <p>
     * Example: <code>my-db-snapshot</code>
     * </p>
     * 
     * @return The identifier for the copy of the snapshot.</p>
     *         <p>
     *         Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Can't be null, empty, or blank
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Must contain from 1 to 255 letters, numbers, or hyphens
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         First character must be a letter
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Can't end with a hyphen or contain two consecutive hyphens
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Example: <code>my-db-snapshot</code>
     */
    public final String targetDBSnapshotIdentifier() {
        return targetDBSnapshotIdentifier;
    }

    /**
     * <p>
     * The Amazon Web Services KMS key identifier for an encrypted DB snapshot. The Amazon Web Services KMS key
     * identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.
     * </p>
     * <p>
     * If you copy an encrypted DB snapshot from your Amazon Web Services account, you can specify a value for this
     * parameter to encrypt the copy with a new KMS key. If you don't specify a value for this parameter, then the copy
     * of the DB snapshot is encrypted with the same Amazon Web Services KMS key as the source DB snapshot.
     * </p>
     * <p>
     * If you copy an encrypted DB snapshot that is shared from another Amazon Web Services account, then you must
     * specify a value for this parameter.
     * </p>
     * <p>
     * If you specify this parameter when you copy an unencrypted snapshot, the copy is encrypted.
     * </p>
     * <p>
     * If you copy an encrypted snapshot to a different Amazon Web Services Region, then you must specify an Amazon Web
     * Services KMS key identifier for the destination Amazon Web Services Region. KMS keys are specific to the Amazon
     * Web Services Region that they are created in, and you can't use KMS keys from one Amazon Web Services Region in
     * another Amazon Web Services Region.
     * </p>
     * 
     * @return The Amazon Web Services KMS key identifier for an encrypted DB snapshot. The Amazon Web Services KMS key
     *         identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.</p>
     *         <p>
     *         If you copy an encrypted DB snapshot from your Amazon Web Services account, you can specify a value for
     *         this parameter to encrypt the copy with a new KMS key. If you don't specify a value for this parameter,
     *         then the copy of the DB snapshot is encrypted with the same Amazon Web Services KMS key as the source DB
     *         snapshot.
     *         </p>
     *         <p>
     *         If you copy an encrypted DB snapshot that is shared from another Amazon Web Services account, then you
     *         must specify a value for this parameter.
     *         </p>
     *         <p>
     *         If you specify this parameter when you copy an unencrypted snapshot, the copy is encrypted.
     *         </p>
     *         <p>
     *         If you copy an encrypted snapshot to a different Amazon Web Services Region, then you must specify an
     *         Amazon Web Services KMS key identifier for the destination Amazon Web Services Region. KMS keys are
     *         specific to the Amazon Web Services Region that they are created in, and you can't use KMS keys from one
     *         Amazon Web Services Region in another Amazon Web Services Region.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * Returns the value of the Tags property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The value of the Tags property for this object.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * A value that indicates whether to copy all tags from the source DB snapshot to the target DB snapshot. By
     * default, tags are not copied.
     * </p>
     * 
     * @return A value that indicates whether to copy all tags from the source DB snapshot to the target DB snapshot. By
     *         default, tags are not copied.
     */
    public final Boolean copyTags() {
        return copyTags;
    }

    /**
     * <p>
     * The URL that contains a Signature Version 4 signed request for the <code>CopyDBSnapshot</code> API action in the
     * source Amazon Web Services Region that contains the source DB snapshot to copy.
     * </p>
     * <p>
     * You must specify this parameter when you copy an encrypted DB snapshot from another Amazon Web Services Region by
     * using the Amazon RDS API. Don't specify <code>PreSignedUrl</code> when you are copying an encrypted DB snapshot
     * in the same Amazon Web Services Region.
     * </p>
     * <p>
     * The presigned URL must be a valid request for the <code>CopyDBSnapshot</code> API action that can be executed in
     * the source Amazon Web Services Region that contains the encrypted DB snapshot to be copied. The presigned URL
     * request must contain the following parameter values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DestinationRegion</code> - The Amazon Web Services Region that the encrypted DB snapshot is copied to. This
     * Amazon Web Services Region is the same one where the <code>CopyDBSnapshot</code> action is called that contains
     * this presigned URL.
     * </p>
     * <p>
     * For example, if you copy an encrypted DB snapshot from the us-west-2 Amazon Web Services Region to the us-east-1
     * Amazon Web Services Region, then you call the <code>CopyDBSnapshot</code> action in the us-east-1 Amazon Web
     * Services Region and provide a presigned URL that contains a call to the <code>CopyDBSnapshot</code> action in the
     * us-west-2 Amazon Web Services Region. For this example, the <code>DestinationRegion</code> in the presigned URL
     * must be set to the us-east-1 Amazon Web Services Region.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KmsKeyId</code> - The Amazon Web Services KMS key identifier for the KMS key to use to encrypt the copy of
     * the DB snapshot in the destination Amazon Web Services Region. This is the same identifier for both the
     * <code>CopyDBSnapshot</code> action that is called in the destination Amazon Web Services Region, and the action
     * contained in the presigned URL.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SourceDBSnapshotIdentifier</code> - The DB snapshot identifier for the encrypted snapshot to be copied.
     * This identifier must be in the Amazon Resource Name (ARN) format for the source Amazon Web Services Region. For
     * example, if you are copying an encrypted DB snapshot from the us-west-2 Amazon Web Services Region, then your
     * <code>SourceDBSnapshotIdentifier</code> looks like the following example:
     * <code>arn:aws:rds:us-west-2:123456789012:snapshot:mysql-instance1-snapshot-20161115</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * To learn how to generate a Signature Version 4 signed request, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/sigv4-query-string-auth.html">Authenticating Requests:
     * Using Query Parameters (Amazon Web Services Signature Version 4)</a> and <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4 Signing
     * Process</a>.
     * </p>
     * <note>
     * <p>
     * If you are using an Amazon Web Services SDK tool or the CLI, you can specify <code>SourceRegion</code> (or
     * <code>--source-region</code> for the CLI) instead of specifying <code>PreSignedUrl</code> manually. Specifying
     * <code>SourceRegion</code> autogenerates a pre-signed URL that is a valid request for the operation that can be
     * executed in the source Amazon Web Services Region.
     * </p>
     * </note>
     * 
     * @return The URL that contains a Signature Version 4 signed request for the <code>CopyDBSnapshot</code> API action
     *         in the source Amazon Web Services Region that contains the source DB snapshot to copy.</p>
     *         <p>
     *         You must specify this parameter when you copy an encrypted DB snapshot from another Amazon Web Services
     *         Region by using the Amazon RDS API. Don't specify <code>PreSignedUrl</code> when you are copying an
     *         encrypted DB snapshot in the same Amazon Web Services Region.
     *         </p>
     *         <p>
     *         The presigned URL must be a valid request for the <code>CopyDBSnapshot</code> API action that can be
     *         executed in the source Amazon Web Services Region that contains the encrypted DB snapshot to be copied.
     *         The presigned URL request must contain the following parameter values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DestinationRegion</code> - The Amazon Web Services Region that the encrypted DB snapshot is copied
     *         to. This Amazon Web Services Region is the same one where the <code>CopyDBSnapshot</code> action is
     *         called that contains this presigned URL.
     *         </p>
     *         <p>
     *         For example, if you copy an encrypted DB snapshot from the us-west-2 Amazon Web Services Region to the
     *         us-east-1 Amazon Web Services Region, then you call the <code>CopyDBSnapshot</code> action in the
     *         us-east-1 Amazon Web Services Region and provide a presigned URL that contains a call to the
     *         <code>CopyDBSnapshot</code> action in the us-west-2 Amazon Web Services Region. For this example, the
     *         <code>DestinationRegion</code> in the presigned URL must be set to the us-east-1 Amazon Web Services
     *         Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KmsKeyId</code> - The Amazon Web Services KMS key identifier for the KMS key to use to encrypt the
     *         copy of the DB snapshot in the destination Amazon Web Services Region. This is the same identifier for
     *         both the <code>CopyDBSnapshot</code> action that is called in the destination Amazon Web Services Region,
     *         and the action contained in the presigned URL.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SourceDBSnapshotIdentifier</code> - The DB snapshot identifier for the encrypted snapshot to be
     *         copied. This identifier must be in the Amazon Resource Name (ARN) format for the source Amazon Web
     *         Services Region. For example, if you are copying an encrypted DB snapshot from the us-west-2 Amazon Web
     *         Services Region, then your <code>SourceDBSnapshotIdentifier</code> looks like the following example:
     *         <code>arn:aws:rds:us-west-2:123456789012:snapshot:mysql-instance1-snapshot-20161115</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To learn how to generate a Signature Version 4 signed request, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/API/sigv4-query-string-auth.html">Authenticating
     *         Requests: Using Query Parameters (Amazon Web Services Signature Version 4)</a> and <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4 Signing
     *         Process</a>.
     *         </p>
     *         <note>
     *         <p>
     *         If you are using an Amazon Web Services SDK tool or the CLI, you can specify <code>SourceRegion</code>
     *         (or <code>--source-region</code> for the CLI) instead of specifying <code>PreSignedUrl</code> manually.
     *         Specifying <code>SourceRegion</code> autogenerates a pre-signed URL that is a valid request for the
     *         operation that can be executed in the source Amazon Web Services Region.
     *         </p>
     */
    public final String preSignedUrl() {
        return preSignedUrl;
    }

    /**
     * <p>
     * The name of an option group to associate with the copy of the snapshot.
     * </p>
     * <p>
     * Specify this option if you are copying a snapshot from one Amazon Web Services Region to another, and your DB
     * instance uses a nondefault option group. If your source DB instance uses Transparent Data Encryption for Oracle
     * or Microsoft SQL Server, you must specify this option when copying across Amazon Web Services Regions. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_CopySnapshot.html#USER_CopySnapshot.Options"
     * >Option group considerations</a> in the <i>Amazon RDS User Guide</i>.
     * </p>
     * 
     * @return The name of an option group to associate with the copy of the snapshot.</p>
     *         <p>
     *         Specify this option if you are copying a snapshot from one Amazon Web Services Region to another, and
     *         your DB instance uses a nondefault option group. If your source DB instance uses Transparent Data
     *         Encryption for Oracle or Microsoft SQL Server, you must specify this option when copying across Amazon
     *         Web Services Regions. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_CopySnapshot.html#USER_CopySnapshot.Options"
     *         >Option group considerations</a> in the <i>Amazon RDS User Guide</i>.
     */
    public final String optionGroupName() {
        return optionGroupName;
    }

    /**
     * <p>
     * The external custom Availability Zone (CAZ) identifier for the target CAZ.
     * </p>
     * <p>
     * Example: <code>rds-caz-aiqhTgQv</code>.
     * </p>
     * 
     * @return The external custom Availability Zone (CAZ) identifier for the target CAZ.</p>
     *         <p>
     *         Example: <code>rds-caz-aiqhTgQv</code>.
     */
    public final String targetCustomAvailabilityZone() {
        return targetCustomAvailabilityZone;
    }

    /**
     * If PreSignedUrl is not specified, this is the region where the source snapshot is located. A PreSignedUrl will be
     * generated automatically by the SDK.
     * 
     * @return If PreSignedUrl is not specified, this is the region where the source snapshot is located. A PreSignedUrl
     *         will be generated automatically by the SDK.
     */
    public final String sourceRegion() {
        return sourceRegion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(sourceDBSnapshotIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(targetDBSnapshotIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(copyTags());
        hashCode = 31 * hashCode + Objects.hashCode(preSignedUrl());
        hashCode = 31 * hashCode + Objects.hashCode(optionGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(targetCustomAvailabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(sourceRegion());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CopyDbSnapshotRequest)) {
            return false;
        }
        CopyDbSnapshotRequest other = (CopyDbSnapshotRequest) obj;
        return Objects.equals(sourceDBSnapshotIdentifier(), other.sourceDBSnapshotIdentifier())
                && Objects.equals(targetDBSnapshotIdentifier(), other.targetDBSnapshotIdentifier())
                && Objects.equals(kmsKeyId(), other.kmsKeyId()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(copyTags(), other.copyTags())
                && Objects.equals(preSignedUrl(), other.preSignedUrl())
                && Objects.equals(optionGroupName(), other.optionGroupName())
                && Objects.equals(targetCustomAvailabilityZone(), other.targetCustomAvailabilityZone())
                && Objects.equals(sourceRegion(), other.sourceRegion());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CopyDbSnapshotRequest").add("SourceDBSnapshotIdentifier", sourceDBSnapshotIdentifier())
                .add("TargetDBSnapshotIdentifier", targetDBSnapshotIdentifier()).add("KmsKeyId", kmsKeyId())
                .add("Tags", hasTags() ? tags() : null).add("CopyTags", copyTags()).add("PreSignedUrl", preSignedUrl())
                .add("OptionGroupName", optionGroupName()).add("TargetCustomAvailabilityZone", targetCustomAvailabilityZone())
                .add("SourceRegion", sourceRegion()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SourceDBSnapshotIdentifier":
            return Optional.ofNullable(clazz.cast(sourceDBSnapshotIdentifier()));
        case "TargetDBSnapshotIdentifier":
            return Optional.ofNullable(clazz.cast(targetDBSnapshotIdentifier()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "CopyTags":
            return Optional.ofNullable(clazz.cast(copyTags()));
        case "PreSignedUrl":
            return Optional.ofNullable(clazz.cast(preSignedUrl()));
        case "OptionGroupName":
            return Optional.ofNullable(clazz.cast(optionGroupName()));
        case "TargetCustomAvailabilityZone":
            return Optional.ofNullable(clazz.cast(targetCustomAvailabilityZone()));
        case "SourceRegion":
            return Optional.ofNullable(clazz.cast(sourceRegion()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CopyDbSnapshotRequest, T> g) {
        return obj -> g.apply((CopyDbSnapshotRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends RdsRequest.Builder, SdkPojo, CopyableBuilder<Builder, CopyDbSnapshotRequest> {
        /**
         * <p>
         * The identifier for the source DB snapshot.
         * </p>
         * <p>
         * If the source snapshot is in the same Amazon Web Services Region as the copy, specify a valid DB snapshot
         * identifier. For example, you might specify <code>rds:mysql-instance1-snapshot-20130805</code>.
         * </p>
         * <p>
         * If the source snapshot is in a different Amazon Web Services Region than the copy, specify a valid DB
         * snapshot ARN. For example, you might specify
         * <code>arn:aws:rds:us-west-2:123456789012:snapshot:mysql-instance1-snapshot-20130805</code>.
         * </p>
         * <p>
         * If you are copying from a shared manual DB snapshot, this parameter must be the Amazon Resource Name (ARN) of
         * the shared DB snapshot.
         * </p>
         * <p>
         * If you are copying an encrypted snapshot this parameter must be in the ARN format for the source Amazon Web
         * Services Region, and must match the <code>SourceDBSnapshotIdentifier</code> in the <code>PreSignedUrl</code>
         * parameter.
         * </p>
         * <p>
         * Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Must specify a valid system snapshot in the "available" state.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Example: <code>rds:mydb-2012-04-02-00-01</code>
         * </p>
         * <p>
         * Example: <code>arn:aws:rds:us-west-2:123456789012:snapshot:mysql-instance1-snapshot-20130805</code>
         * </p>
         * 
         * @param sourceDBSnapshotIdentifier
         *        The identifier for the source DB snapshot.</p>
         *        <p>
         *        If the source snapshot is in the same Amazon Web Services Region as the copy, specify a valid DB
         *        snapshot identifier. For example, you might specify <code>rds:mysql-instance1-snapshot-20130805</code>
         *        .
         *        </p>
         *        <p>
         *        If the source snapshot is in a different Amazon Web Services Region than the copy, specify a valid DB
         *        snapshot ARN. For example, you might specify
         *        <code>arn:aws:rds:us-west-2:123456789012:snapshot:mysql-instance1-snapshot-20130805</code>.
         *        </p>
         *        <p>
         *        If you are copying from a shared manual DB snapshot, this parameter must be the Amazon Resource Name
         *        (ARN) of the shared DB snapshot.
         *        </p>
         *        <p>
         *        If you are copying an encrypted snapshot this parameter must be in the ARN format for the source
         *        Amazon Web Services Region, and must match the <code>SourceDBSnapshotIdentifier</code> in the
         *        <code>PreSignedUrl</code> parameter.
         *        </p>
         *        <p>
         *        Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Must specify a valid system snapshot in the "available" state.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Example: <code>rds:mydb-2012-04-02-00-01</code>
         *        </p>
         *        <p>
         *        Example: <code>arn:aws:rds:us-west-2:123456789012:snapshot:mysql-instance1-snapshot-20130805</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceDBSnapshotIdentifier(String sourceDBSnapshotIdentifier);

        /**
         * <p>
         * The identifier for the copy of the snapshot.
         * </p>
         * <p>
         * Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Can't be null, empty, or blank
         * </p>
         * </li>
         * <li>
         * <p>
         * Must contain from 1 to 255 letters, numbers, or hyphens
         * </p>
         * </li>
         * <li>
         * <p>
         * First character must be a letter
         * </p>
         * </li>
         * <li>
         * <p>
         * Can't end with a hyphen or contain two consecutive hyphens
         * </p>
         * </li>
         * </ul>
         * <p>
         * Example: <code>my-db-snapshot</code>
         * </p>
         * 
         * @param targetDBSnapshotIdentifier
         *        The identifier for the copy of the snapshot.</p>
         *        <p>
         *        Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Can't be null, empty, or blank
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must contain from 1 to 255 letters, numbers, or hyphens
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        First character must be a letter
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Can't end with a hyphen or contain two consecutive hyphens
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Example: <code>my-db-snapshot</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetDBSnapshotIdentifier(String targetDBSnapshotIdentifier);

        /**
         * <p>
         * The Amazon Web Services KMS key identifier for an encrypted DB snapshot. The Amazon Web Services KMS key
         * identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.
         * </p>
         * <p>
         * If you copy an encrypted DB snapshot from your Amazon Web Services account, you can specify a value for this
         * parameter to encrypt the copy with a new KMS key. If you don't specify a value for this parameter, then the
         * copy of the DB snapshot is encrypted with the same Amazon Web Services KMS key as the source DB snapshot.
         * </p>
         * <p>
         * If you copy an encrypted DB snapshot that is shared from another Amazon Web Services account, then you must
         * specify a value for this parameter.
         * </p>
         * <p>
         * If you specify this parameter when you copy an unencrypted snapshot, the copy is encrypted.
         * </p>
         * <p>
         * If you copy an encrypted snapshot to a different Amazon Web Services Region, then you must specify an Amazon
         * Web Services KMS key identifier for the destination Amazon Web Services Region. KMS keys are specific to the
         * Amazon Web Services Region that they are created in, and you can't use KMS keys from one Amazon Web Services
         * Region in another Amazon Web Services Region.
         * </p>
         * 
         * @param kmsKeyId
         *        The Amazon Web Services KMS key identifier for an encrypted DB snapshot. The Amazon Web Services KMS
         *        key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.</p>
         *        <p>
         *        If you copy an encrypted DB snapshot from your Amazon Web Services account, you can specify a value
         *        for this parameter to encrypt the copy with a new KMS key. If you don't specify a value for this
         *        parameter, then the copy of the DB snapshot is encrypted with the same Amazon Web Services KMS key as
         *        the source DB snapshot.
         *        </p>
         *        <p>
         *        If you copy an encrypted DB snapshot that is shared from another Amazon Web Services account, then you
         *        must specify a value for this parameter.
         *        </p>
         *        <p>
         *        If you specify this parameter when you copy an unencrypted snapshot, the copy is encrypted.
         *        </p>
         *        <p>
         *        If you copy an encrypted snapshot to a different Amazon Web Services Region, then you must specify an
         *        Amazon Web Services KMS key identifier for the destination Amazon Web Services Region. KMS keys are
         *        specific to the Amazon Web Services Region that they are created in, and you can't use KMS keys from
         *        one Amazon Web Services Region in another Amazon Web Services Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * Sets the value of the Tags property for this object.
         *
         * @param tags
         *        The new value for the Tags property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * Sets the value of the Tags property for this object.
         *
         * @param tags
         *        The new value for the Tags property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * Sets the value of the Tags property for this object.
         *
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * A value that indicates whether to copy all tags from the source DB snapshot to the target DB snapshot. By
         * default, tags are not copied.
         * </p>
         * 
         * @param copyTags
         *        A value that indicates whether to copy all tags from the source DB snapshot to the target DB snapshot.
         *        By default, tags are not copied.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder copyTags(Boolean copyTags);

        /**
         * <p>
         * The URL that contains a Signature Version 4 signed request for the <code>CopyDBSnapshot</code> API action in
         * the source Amazon Web Services Region that contains the source DB snapshot to copy.
         * </p>
         * <p>
         * You must specify this parameter when you copy an encrypted DB snapshot from another Amazon Web Services
         * Region by using the Amazon RDS API. Don't specify <code>PreSignedUrl</code> when you are copying an encrypted
         * DB snapshot in the same Amazon Web Services Region.
         * </p>
         * <p>
         * The presigned URL must be a valid request for the <code>CopyDBSnapshot</code> API action that can be executed
         * in the source Amazon Web Services Region that contains the encrypted DB snapshot to be copied. The presigned
         * URL request must contain the following parameter values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DestinationRegion</code> - The Amazon Web Services Region that the encrypted DB snapshot is copied to.
         * This Amazon Web Services Region is the same one where the <code>CopyDBSnapshot</code> action is called that
         * contains this presigned URL.
         * </p>
         * <p>
         * For example, if you copy an encrypted DB snapshot from the us-west-2 Amazon Web Services Region to the
         * us-east-1 Amazon Web Services Region, then you call the <code>CopyDBSnapshot</code> action in the us-east-1
         * Amazon Web Services Region and provide a presigned URL that contains a call to the
         * <code>CopyDBSnapshot</code> action in the us-west-2 Amazon Web Services Region. For this example, the
         * <code>DestinationRegion</code> in the presigned URL must be set to the us-east-1 Amazon Web Services Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KmsKeyId</code> - The Amazon Web Services KMS key identifier for the KMS key to use to encrypt the copy
         * of the DB snapshot in the destination Amazon Web Services Region. This is the same identifier for both the
         * <code>CopyDBSnapshot</code> action that is called in the destination Amazon Web Services Region, and the
         * action contained in the presigned URL.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SourceDBSnapshotIdentifier</code> - The DB snapshot identifier for the encrypted snapshot to be copied.
         * This identifier must be in the Amazon Resource Name (ARN) format for the source Amazon Web Services Region.
         * For example, if you are copying an encrypted DB snapshot from the us-west-2 Amazon Web Services Region, then
         * your <code>SourceDBSnapshotIdentifier</code> looks like the following example:
         * <code>arn:aws:rds:us-west-2:123456789012:snapshot:mysql-instance1-snapshot-20161115</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * To learn how to generate a Signature Version 4 signed request, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/API/sigv4-query-string-auth.html">Authenticating Requests:
         * Using Query Parameters (Amazon Web Services Signature Version 4)</a> and <a
         * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4 Signing
         * Process</a>.
         * </p>
         * <note>
         * <p>
         * If you are using an Amazon Web Services SDK tool or the CLI, you can specify <code>SourceRegion</code> (or
         * <code>--source-region</code> for the CLI) instead of specifying <code>PreSignedUrl</code> manually.
         * Specifying <code>SourceRegion</code> autogenerates a pre-signed URL that is a valid request for the operation
         * that can be executed in the source Amazon Web Services Region.
         * </p>
         * </note>
         * 
         * @param preSignedUrl
         *        The URL that contains a Signature Version 4 signed request for the <code>CopyDBSnapshot</code> API
         *        action in the source Amazon Web Services Region that contains the source DB snapshot to copy.</p>
         *        <p>
         *        You must specify this parameter when you copy an encrypted DB snapshot from another Amazon Web
         *        Services Region by using the Amazon RDS API. Don't specify <code>PreSignedUrl</code> when you are
         *        copying an encrypted DB snapshot in the same Amazon Web Services Region.
         *        </p>
         *        <p>
         *        The presigned URL must be a valid request for the <code>CopyDBSnapshot</code> API action that can be
         *        executed in the source Amazon Web Services Region that contains the encrypted DB snapshot to be
         *        copied. The presigned URL request must contain the following parameter values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DestinationRegion</code> - The Amazon Web Services Region that the encrypted DB snapshot is
         *        copied to. This Amazon Web Services Region is the same one where the <code>CopyDBSnapshot</code>
         *        action is called that contains this presigned URL.
         *        </p>
         *        <p>
         *        For example, if you copy an encrypted DB snapshot from the us-west-2 Amazon Web Services Region to the
         *        us-east-1 Amazon Web Services Region, then you call the <code>CopyDBSnapshot</code> action in the
         *        us-east-1 Amazon Web Services Region and provide a presigned URL that contains a call to the
         *        <code>CopyDBSnapshot</code> action in the us-west-2 Amazon Web Services Region. For this example, the
         *        <code>DestinationRegion</code> in the presigned URL must be set to the us-east-1 Amazon Web Services
         *        Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KmsKeyId</code> - The Amazon Web Services KMS key identifier for the KMS key to use to encrypt
         *        the copy of the DB snapshot in the destination Amazon Web Services Region. This is the same identifier
         *        for both the <code>CopyDBSnapshot</code> action that is called in the destination Amazon Web Services
         *        Region, and the action contained in the presigned URL.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SourceDBSnapshotIdentifier</code> - The DB snapshot identifier for the encrypted snapshot to be
         *        copied. This identifier must be in the Amazon Resource Name (ARN) format for the source Amazon Web
         *        Services Region. For example, if you are copying an encrypted DB snapshot from the us-west-2 Amazon
         *        Web Services Region, then your <code>SourceDBSnapshotIdentifier</code> looks like the following
         *        example: <code>arn:aws:rds:us-west-2:123456789012:snapshot:mysql-instance1-snapshot-20161115</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        To learn how to generate a Signature Version 4 signed request, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/API/sigv4-query-string-auth.html">Authenticating
         *        Requests: Using Query Parameters (Amazon Web Services Signature Version 4)</a> and <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
         *        Signing Process</a>.
         *        </p>
         *        <note>
         *        <p>
         *        If you are using an Amazon Web Services SDK tool or the CLI, you can specify <code>SourceRegion</code>
         *        (or <code>--source-region</code> for the CLI) instead of specifying <code>PreSignedUrl</code>
         *        manually. Specifying <code>SourceRegion</code> autogenerates a pre-signed URL that is a valid request
         *        for the operation that can be executed in the source Amazon Web Services Region.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preSignedUrl(String preSignedUrl);

        /**
         * <p>
         * The name of an option group to associate with the copy of the snapshot.
         * </p>
         * <p>
         * Specify this option if you are copying a snapshot from one Amazon Web Services Region to another, and your DB
         * instance uses a nondefault option group. If your source DB instance uses Transparent Data Encryption for
         * Oracle or Microsoft SQL Server, you must specify this option when copying across Amazon Web Services Regions.
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_CopySnapshot.html#USER_CopySnapshot.Options"
         * >Option group considerations</a> in the <i>Amazon RDS User Guide</i>.
         * </p>
         * 
         * @param optionGroupName
         *        The name of an option group to associate with the copy of the snapshot.</p>
         *        <p>
         *        Specify this option if you are copying a snapshot from one Amazon Web Services Region to another, and
         *        your DB instance uses a nondefault option group. If your source DB instance uses Transparent Data
         *        Encryption for Oracle or Microsoft SQL Server, you must specify this option when copying across Amazon
         *        Web Services Regions. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_CopySnapshot.html#USER_CopySnapshot.Options"
         *        >Option group considerations</a> in the <i>Amazon RDS User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder optionGroupName(String optionGroupName);

        /**
         * <p>
         * The external custom Availability Zone (CAZ) identifier for the target CAZ.
         * </p>
         * <p>
         * Example: <code>rds-caz-aiqhTgQv</code>.
         * </p>
         * 
         * @param targetCustomAvailabilityZone
         *        The external custom Availability Zone (CAZ) identifier for the target CAZ.</p>
         *        <p>
         *        Example: <code>rds-caz-aiqhTgQv</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetCustomAvailabilityZone(String targetCustomAvailabilityZone);

        /**
         * If PreSignedUrl is not specified, this is the region where the source snapshot is located. A PreSignedUrl
         * will be generated automatically by the SDK.
         * 
         * @param sourceRegion
         *        If PreSignedUrl is not specified, this is the region where the source snapshot is located. A
         *        PreSignedUrl will be generated automatically by the SDK.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceRegion(String sourceRegion);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends RdsRequest.BuilderImpl implements Builder {
        private String sourceDBSnapshotIdentifier;

        private String targetDBSnapshotIdentifier;

        private String kmsKeyId;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private Boolean copyTags;

        private String preSignedUrl;

        private String optionGroupName;

        private String targetCustomAvailabilityZone;

        private String sourceRegion;

        private BuilderImpl() {
        }

        private BuilderImpl(CopyDbSnapshotRequest model) {
            super(model);
            sourceDBSnapshotIdentifier(model.sourceDBSnapshotIdentifier);
            targetDBSnapshotIdentifier(model.targetDBSnapshotIdentifier);
            kmsKeyId(model.kmsKeyId);
            tags(model.tags);
            copyTags(model.copyTags);
            preSignedUrl(model.preSignedUrl);
            optionGroupName(model.optionGroupName);
            targetCustomAvailabilityZone(model.targetCustomAvailabilityZone);
            sourceRegion(model.sourceRegion);
        }

        public final String getSourceDBSnapshotIdentifier() {
            return sourceDBSnapshotIdentifier;
        }

        public final void setSourceDBSnapshotIdentifier(String sourceDBSnapshotIdentifier) {
            this.sourceDBSnapshotIdentifier = sourceDBSnapshotIdentifier;
        }

        @Override
        @Transient
        public final Builder sourceDBSnapshotIdentifier(String sourceDBSnapshotIdentifier) {
            this.sourceDBSnapshotIdentifier = sourceDBSnapshotIdentifier;
            return this;
        }

        public final String getTargetDBSnapshotIdentifier() {
            return targetDBSnapshotIdentifier;
        }

        public final void setTargetDBSnapshotIdentifier(String targetDBSnapshotIdentifier) {
            this.targetDBSnapshotIdentifier = targetDBSnapshotIdentifier;
        }

        @Override
        @Transient
        public final Builder targetDBSnapshotIdentifier(String targetDBSnapshotIdentifier) {
            this.targetDBSnapshotIdentifier = targetDBSnapshotIdentifier;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        @Transient
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        @Transient
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Boolean getCopyTags() {
            return copyTags;
        }

        public final void setCopyTags(Boolean copyTags) {
            this.copyTags = copyTags;
        }

        @Override
        @Transient
        public final Builder copyTags(Boolean copyTags) {
            this.copyTags = copyTags;
            return this;
        }

        public final String getPreSignedUrl() {
            return preSignedUrl;
        }

        public final void setPreSignedUrl(String preSignedUrl) {
            this.preSignedUrl = preSignedUrl;
        }

        @Override
        @Transient
        public final Builder preSignedUrl(String preSignedUrl) {
            this.preSignedUrl = preSignedUrl;
            return this;
        }

        public final String getOptionGroupName() {
            return optionGroupName;
        }

        public final void setOptionGroupName(String optionGroupName) {
            this.optionGroupName = optionGroupName;
        }

        @Override
        @Transient
        public final Builder optionGroupName(String optionGroupName) {
            this.optionGroupName = optionGroupName;
            return this;
        }

        public final String getTargetCustomAvailabilityZone() {
            return targetCustomAvailabilityZone;
        }

        public final void setTargetCustomAvailabilityZone(String targetCustomAvailabilityZone) {
            this.targetCustomAvailabilityZone = targetCustomAvailabilityZone;
        }

        @Override
        @Transient
        public final Builder targetCustomAvailabilityZone(String targetCustomAvailabilityZone) {
            this.targetCustomAvailabilityZone = targetCustomAvailabilityZone;
            return this;
        }

        public final String getSourceRegion() {
            return sourceRegion;
        }

        public final void setSourceRegion(String sourceRegion) {
            this.sourceRegion = sourceRegion;
        }

        @Override
        @Transient
        public final Builder sourceRegion(String sourceRegion) {
            this.sourceRegion = sourceRegion;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CopyDbSnapshotRequest build() {
            return new CopyDbSnapshotRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
