/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rds.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the settings that control the size and behavior of the connection pool associated with a
 * <code>DBProxyTargetGroup</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ConnectionPoolConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<ConnectionPoolConfiguration.Builder, ConnectionPoolConfiguration> {
    private static final SdkField<Integer> MAX_CONNECTIONS_PERCENT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxConnectionsPercent").getter(getter(ConnectionPoolConfiguration::maxConnectionsPercent))
            .setter(setter(Builder::maxConnectionsPercent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxConnectionsPercent").build())
            .build();

    private static final SdkField<Integer> MAX_IDLE_CONNECTIONS_PERCENT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("MaxIdleConnectionsPercent")
            .getter(getter(ConnectionPoolConfiguration::maxIdleConnectionsPercent))
            .setter(setter(Builder::maxIdleConnectionsPercent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxIdleConnectionsPercent").build())
            .build();

    private static final SdkField<Integer> CONNECTION_BORROW_TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ConnectionBorrowTimeout").getter(getter(ConnectionPoolConfiguration::connectionBorrowTimeout))
            .setter(setter(Builder::connectionBorrowTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionBorrowTimeout").build())
            .build();

    private static final SdkField<List<String>> SESSION_PINNING_FILTERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SessionPinningFilters")
            .getter(getter(ConnectionPoolConfiguration::sessionPinningFilters))
            .setter(setter(Builder::sessionPinningFilters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SessionPinningFilters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> INIT_QUERY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InitQuery").getter(getter(ConnectionPoolConfiguration::initQuery)).setter(setter(Builder::initQuery))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InitQuery").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections
            .unmodifiableList(Arrays.asList(MAX_CONNECTIONS_PERCENT_FIELD, MAX_IDLE_CONNECTIONS_PERCENT_FIELD,
                    CONNECTION_BORROW_TIMEOUT_FIELD, SESSION_PINNING_FILTERS_FIELD, INIT_QUERY_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer maxConnectionsPercent;

    private final Integer maxIdleConnectionsPercent;

    private final Integer connectionBorrowTimeout;

    private final List<String> sessionPinningFilters;

    private final String initQuery;

    private ConnectionPoolConfiguration(BuilderImpl builder) {
        this.maxConnectionsPercent = builder.maxConnectionsPercent;
        this.maxIdleConnectionsPercent = builder.maxIdleConnectionsPercent;
        this.connectionBorrowTimeout = builder.connectionBorrowTimeout;
        this.sessionPinningFilters = builder.sessionPinningFilters;
        this.initQuery = builder.initQuery;
    }

    /**
     * <p>
     * The maximum size of the connection pool for each target in a target group. The value is expressed as a percentage
     * of the <code>max_connections</code> setting for the RDS DB instance or Aurora DB cluster used by the target
     * group.
     * </p>
     * <p>
     * Default: 100
     * </p>
     * <p>
     * Constraints: between 1 and 100
     * </p>
     * 
     * @return The maximum size of the connection pool for each target in a target group. The value is expressed as a
     *         percentage of the <code>max_connections</code> setting for the RDS DB instance or Aurora DB cluster used
     *         by the target group.</p>
     *         <p>
     *         Default: 100
     *         </p>
     *         <p>
     *         Constraints: between 1 and 100
     */
    public final Integer maxConnectionsPercent() {
        return maxConnectionsPercent;
    }

    /**
     * <p>
     * Controls how actively the proxy closes idle database connections in the connection pool. The value is expressed
     * as a percentage of the <code>max_connections</code> setting for the RDS DB instance or Aurora DB cluster used by
     * the target group. With a high value, the proxy leaves a high percentage of idle database connections open. A low
     * value causes the proxy to close more idle connections and return them to the database.
     * </p>
     * <p>
     * Default: 50
     * </p>
     * <p>
     * Constraints: between 0 and <code>MaxConnectionsPercent</code>
     * </p>
     * 
     * @return Controls how actively the proxy closes idle database connections in the connection pool. The value is
     *         expressed as a percentage of the <code>max_connections</code> setting for the RDS DB instance or Aurora
     *         DB cluster used by the target group. With a high value, the proxy leaves a high percentage of idle
     *         database connections open. A low value causes the proxy to close more idle connections and return them to
     *         the database.</p>
     *         <p>
     *         Default: 50
     *         </p>
     *         <p>
     *         Constraints: between 0 and <code>MaxConnectionsPercent</code>
     */
    public final Integer maxIdleConnectionsPercent() {
        return maxIdleConnectionsPercent;
    }

    /**
     * <p>
     * The number of seconds for a proxy to wait for a connection to become available in the connection pool. Only
     * applies when the proxy has opened its maximum number of connections and all connections are busy with client
     * sessions.
     * </p>
     * <p>
     * Default: 120
     * </p>
     * <p>
     * Constraints: between 1 and 3600, or 0 representing unlimited
     * </p>
     * 
     * @return The number of seconds for a proxy to wait for a connection to become available in the connection pool.
     *         Only applies when the proxy has opened its maximum number of connections and all connections are busy
     *         with client sessions.</p>
     *         <p>
     *         Default: 120
     *         </p>
     *         <p>
     *         Constraints: between 1 and 3600, or 0 representing unlimited
     */
    public final Integer connectionBorrowTimeout() {
        return connectionBorrowTimeout;
    }

    /**
     * For responses, this returns true if the service returned a value for the SessionPinningFilters property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSessionPinningFilters() {
        return sessionPinningFilters != null && !(sessionPinningFilters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Each item in the list represents a class of SQL operations that normally cause all later statements in a session
     * using a proxy to be pinned to the same underlying database connection. Including an item in the list exempts that
     * class of SQL operations from the pinning behavior.
     * </p>
     * <p>
     * Default: no session pinning filters
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSessionPinningFilters} method.
     * </p>
     * 
     * @return Each item in the list represents a class of SQL operations that normally cause all later statements in a
     *         session using a proxy to be pinned to the same underlying database connection. Including an item in the
     *         list exempts that class of SQL operations from the pinning behavior.</p>
     *         <p>
     *         Default: no session pinning filters
     */
    public final List<String> sessionPinningFilters() {
        return sessionPinningFilters;
    }

    /**
     * <p>
     * One or more SQL statements for the proxy to run when opening each new database connection. Typically used with
     * <code>SET</code> statements to make sure that each connection has identical settings such as time zone and
     * character set. For multiple statements, use semicolons as the separator. You can also include multiple variables
     * in a single <code>SET</code> statement, such as <code>SET x=1, y=2</code>.
     * </p>
     * <p>
     * Default: no initialization query
     * </p>
     * 
     * @return One or more SQL statements for the proxy to run when opening each new database connection. Typically used
     *         with <code>SET</code> statements to make sure that each connection has identical settings such as time
     *         zone and character set. For multiple statements, use semicolons as the separator. You can also include
     *         multiple variables in a single <code>SET</code> statement, such as <code>SET x=1, y=2</code>.</p>
     *         <p>
     *         Default: no initialization query
     */
    public final String initQuery() {
        return initQuery;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(maxConnectionsPercent());
        hashCode = 31 * hashCode + Objects.hashCode(maxIdleConnectionsPercent());
        hashCode = 31 * hashCode + Objects.hashCode(connectionBorrowTimeout());
        hashCode = 31 * hashCode + Objects.hashCode(hasSessionPinningFilters() ? sessionPinningFilters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(initQuery());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConnectionPoolConfiguration)) {
            return false;
        }
        ConnectionPoolConfiguration other = (ConnectionPoolConfiguration) obj;
        return Objects.equals(maxConnectionsPercent(), other.maxConnectionsPercent())
                && Objects.equals(maxIdleConnectionsPercent(), other.maxIdleConnectionsPercent())
                && Objects.equals(connectionBorrowTimeout(), other.connectionBorrowTimeout())
                && hasSessionPinningFilters() == other.hasSessionPinningFilters()
                && Objects.equals(sessionPinningFilters(), other.sessionPinningFilters())
                && Objects.equals(initQuery(), other.initQuery());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ConnectionPoolConfiguration").add("MaxConnectionsPercent", maxConnectionsPercent())
                .add("MaxIdleConnectionsPercent", maxIdleConnectionsPercent())
                .add("ConnectionBorrowTimeout", connectionBorrowTimeout())
                .add("SessionPinningFilters", hasSessionPinningFilters() ? sessionPinningFilters() : null)
                .add("InitQuery", initQuery()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MaxConnectionsPercent":
            return Optional.ofNullable(clazz.cast(maxConnectionsPercent()));
        case "MaxIdleConnectionsPercent":
            return Optional.ofNullable(clazz.cast(maxIdleConnectionsPercent()));
        case "ConnectionBorrowTimeout":
            return Optional.ofNullable(clazz.cast(connectionBorrowTimeout()));
        case "SessionPinningFilters":
            return Optional.ofNullable(clazz.cast(sessionPinningFilters()));
        case "InitQuery":
            return Optional.ofNullable(clazz.cast(initQuery()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ConnectionPoolConfiguration, T> g) {
        return obj -> g.apply((ConnectionPoolConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ConnectionPoolConfiguration> {
        /**
         * <p>
         * The maximum size of the connection pool for each target in a target group. The value is expressed as a
         * percentage of the <code>max_connections</code> setting for the RDS DB instance or Aurora DB cluster used by
         * the target group.
         * </p>
         * <p>
         * Default: 100
         * </p>
         * <p>
         * Constraints: between 1 and 100
         * </p>
         * 
         * @param maxConnectionsPercent
         *        The maximum size of the connection pool for each target in a target group. The value is expressed as a
         *        percentage of the <code>max_connections</code> setting for the RDS DB instance or Aurora DB cluster
         *        used by the target group.</p>
         *        <p>
         *        Default: 100
         *        </p>
         *        <p>
         *        Constraints: between 1 and 100
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConnectionsPercent(Integer maxConnectionsPercent);

        /**
         * <p>
         * Controls how actively the proxy closes idle database connections in the connection pool. The value is
         * expressed as a percentage of the <code>max_connections</code> setting for the RDS DB instance or Aurora DB
         * cluster used by the target group. With a high value, the proxy leaves a high percentage of idle database
         * connections open. A low value causes the proxy to close more idle connections and return them to the
         * database.
         * </p>
         * <p>
         * Default: 50
         * </p>
         * <p>
         * Constraints: between 0 and <code>MaxConnectionsPercent</code>
         * </p>
         * 
         * @param maxIdleConnectionsPercent
         *        Controls how actively the proxy closes idle database connections in the connection pool. The value is
         *        expressed as a percentage of the <code>max_connections</code> setting for the RDS DB instance or
         *        Aurora DB cluster used by the target group. With a high value, the proxy leaves a high percentage of
         *        idle database connections open. A low value causes the proxy to close more idle connections and return
         *        them to the database.</p>
         *        <p>
         *        Default: 50
         *        </p>
         *        <p>
         *        Constraints: between 0 and <code>MaxConnectionsPercent</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxIdleConnectionsPercent(Integer maxIdleConnectionsPercent);

        /**
         * <p>
         * The number of seconds for a proxy to wait for a connection to become available in the connection pool. Only
         * applies when the proxy has opened its maximum number of connections and all connections are busy with client
         * sessions.
         * </p>
         * <p>
         * Default: 120
         * </p>
         * <p>
         * Constraints: between 1 and 3600, or 0 representing unlimited
         * </p>
         * 
         * @param connectionBorrowTimeout
         *        The number of seconds for a proxy to wait for a connection to become available in the connection pool.
         *        Only applies when the proxy has opened its maximum number of connections and all connections are busy
         *        with client sessions.</p>
         *        <p>
         *        Default: 120
         *        </p>
         *        <p>
         *        Constraints: between 1 and 3600, or 0 representing unlimited
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionBorrowTimeout(Integer connectionBorrowTimeout);

        /**
         * <p>
         * Each item in the list represents a class of SQL operations that normally cause all later statements in a
         * session using a proxy to be pinned to the same underlying database connection. Including an item in the list
         * exempts that class of SQL operations from the pinning behavior.
         * </p>
         * <p>
         * Default: no session pinning filters
         * </p>
         * 
         * @param sessionPinningFilters
         *        Each item in the list represents a class of SQL operations that normally cause all later statements in
         *        a session using a proxy to be pinned to the same underlying database connection. Including an item in
         *        the list exempts that class of SQL operations from the pinning behavior.</p>
         *        <p>
         *        Default: no session pinning filters
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sessionPinningFilters(Collection<String> sessionPinningFilters);

        /**
         * <p>
         * Each item in the list represents a class of SQL operations that normally cause all later statements in a
         * session using a proxy to be pinned to the same underlying database connection. Including an item in the list
         * exempts that class of SQL operations from the pinning behavior.
         * </p>
         * <p>
         * Default: no session pinning filters
         * </p>
         * 
         * @param sessionPinningFilters
         *        Each item in the list represents a class of SQL operations that normally cause all later statements in
         *        a session using a proxy to be pinned to the same underlying database connection. Including an item in
         *        the list exempts that class of SQL operations from the pinning behavior.</p>
         *        <p>
         *        Default: no session pinning filters
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sessionPinningFilters(String... sessionPinningFilters);

        /**
         * <p>
         * One or more SQL statements for the proxy to run when opening each new database connection. Typically used
         * with <code>SET</code> statements to make sure that each connection has identical settings such as time zone
         * and character set. For multiple statements, use semicolons as the separator. You can also include multiple
         * variables in a single <code>SET</code> statement, such as <code>SET x=1, y=2</code>.
         * </p>
         * <p>
         * Default: no initialization query
         * </p>
         * 
         * @param initQuery
         *        One or more SQL statements for the proxy to run when opening each new database connection. Typically
         *        used with <code>SET</code> statements to make sure that each connection has identical settings such as
         *        time zone and character set. For multiple statements, use semicolons as the separator. You can also
         *        include multiple variables in a single <code>SET</code> statement, such as <code>SET x=1, y=2</code>
         *        .</p>
         *        <p>
         *        Default: no initialization query
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initQuery(String initQuery);
    }

    static final class BuilderImpl implements Builder {
        private Integer maxConnectionsPercent;

        private Integer maxIdleConnectionsPercent;

        private Integer connectionBorrowTimeout;

        private List<String> sessionPinningFilters = DefaultSdkAutoConstructList.getInstance();

        private String initQuery;

        private BuilderImpl() {
        }

        private BuilderImpl(ConnectionPoolConfiguration model) {
            maxConnectionsPercent(model.maxConnectionsPercent);
            maxIdleConnectionsPercent(model.maxIdleConnectionsPercent);
            connectionBorrowTimeout(model.connectionBorrowTimeout);
            sessionPinningFilters(model.sessionPinningFilters);
            initQuery(model.initQuery);
        }

        public final Integer getMaxConnectionsPercent() {
            return maxConnectionsPercent;
        }

        public final void setMaxConnectionsPercent(Integer maxConnectionsPercent) {
            this.maxConnectionsPercent = maxConnectionsPercent;
        }

        @Override
        @Transient
        public final Builder maxConnectionsPercent(Integer maxConnectionsPercent) {
            this.maxConnectionsPercent = maxConnectionsPercent;
            return this;
        }

        public final Integer getMaxIdleConnectionsPercent() {
            return maxIdleConnectionsPercent;
        }

        public final void setMaxIdleConnectionsPercent(Integer maxIdleConnectionsPercent) {
            this.maxIdleConnectionsPercent = maxIdleConnectionsPercent;
        }

        @Override
        @Transient
        public final Builder maxIdleConnectionsPercent(Integer maxIdleConnectionsPercent) {
            this.maxIdleConnectionsPercent = maxIdleConnectionsPercent;
            return this;
        }

        public final Integer getConnectionBorrowTimeout() {
            return connectionBorrowTimeout;
        }

        public final void setConnectionBorrowTimeout(Integer connectionBorrowTimeout) {
            this.connectionBorrowTimeout = connectionBorrowTimeout;
        }

        @Override
        @Transient
        public final Builder connectionBorrowTimeout(Integer connectionBorrowTimeout) {
            this.connectionBorrowTimeout = connectionBorrowTimeout;
            return this;
        }

        public final Collection<String> getSessionPinningFilters() {
            if (sessionPinningFilters instanceof SdkAutoConstructList) {
                return null;
            }
            return sessionPinningFilters;
        }

        public final void setSessionPinningFilters(Collection<String> sessionPinningFilters) {
            this.sessionPinningFilters = StringListCopier.copy(sessionPinningFilters);
        }

        @Override
        @Transient
        public final Builder sessionPinningFilters(Collection<String> sessionPinningFilters) {
            this.sessionPinningFilters = StringListCopier.copy(sessionPinningFilters);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder sessionPinningFilters(String... sessionPinningFilters) {
            sessionPinningFilters(Arrays.asList(sessionPinningFilters));
            return this;
        }

        public final String getInitQuery() {
            return initQuery;
        }

        public final void setInitQuery(String initQuery) {
            this.initQuery = initQuery;
        }

        @Override
        @Transient
        public final Builder initQuery(String initQuery) {
            this.initQuery = initQuery;
            return this;
        }

        @Override
        public ConnectionPoolConfiguration build() {
            return new ConnectionPoolConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
