/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.migrationhubrefactorspaces;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.migrationhubrefactorspaces.internal.MigrationHubRefactorSpacesServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.AccessDeniedException;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.ConflictException;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.CreateApplicationRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.CreateApplicationResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.CreateEnvironmentRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.CreateEnvironmentResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.CreateRouteRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.CreateRouteResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.CreateServiceRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.CreateServiceResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.DeleteApplicationRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.DeleteApplicationResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.DeleteEnvironmentRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.DeleteEnvironmentResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.DeleteResourcePolicyRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.DeleteResourcePolicyResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.DeleteRouteRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.DeleteRouteResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.DeleteServiceRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.DeleteServiceResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.GetApplicationRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.GetApplicationResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.GetResourcePolicyResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.GetRouteRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.GetRouteResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.GetServiceRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.GetServiceResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.InternalServerException;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.InvalidResourcePolicyException;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.ListApplicationsRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.ListApplicationsResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.ListEnvironmentVpcsRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.ListEnvironmentVpcsResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.ListEnvironmentsRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.ListEnvironmentsResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.ListRoutesRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.ListRoutesResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.ListServicesRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.ListServicesResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.MigrationHubRefactorSpacesException;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.PutResourcePolicyRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.PutResourcePolicyResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.ResourceNotFoundException;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.TagResourceRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.TagResourceResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.ThrottlingException;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.UntagResourceRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.UntagResourceResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.UpdateRouteRequest;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.UpdateRouteResponse;
import software.amazon.awssdk.services.migrationhubrefactorspaces.model.ValidationException;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.CreateApplicationRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.CreateEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.CreateRouteRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.CreateServiceRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.DeleteApplicationRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.DeleteEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.DeleteResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.DeleteRouteRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.DeleteServiceRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.GetApplicationRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.GetEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.GetResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.GetRouteRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.GetServiceRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.ListApplicationsRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.ListEnvironmentVpcsRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.ListEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.ListRoutesRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.ListServicesRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.PutResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.migrationhubrefactorspaces.transform.UpdateRouteRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MigrationHubRefactorSpacesAsyncClient}.
 *
 * @see MigrationHubRefactorSpacesAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMigrationHubRefactorSpacesAsyncClient implements MigrationHubRefactorSpacesAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMigrationHubRefactorSpacesAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMigrationHubRefactorSpacesAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an Amazon Web Services Migration Hub Refactor Spaces application. The account that owns the environment
     * also owns the applications created inside the environment, regardless of the account that creates the
     * application. Refactor Spaces provisions an Amazon API Gateway, API Gateway VPC link, and Network Load Balancer
     * for the application proxy inside your account.
     * </p>
     * <p>
     * In environments created with a <a href=
     * "https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateEnvironment.html#migrationhubrefactorspaces-CreateEnvironment-request-NetworkFabricType"
     * >CreateEnvironment:NetworkFabricType</a> of <code>NONE</code> you need to configure <a href=
     * "https://docs.aws.amazon.com/whitepapers/latest/aws-vpc-connectivity-options/amazon-vpc-to-amazon-vpc-connectivity-options.html"
     * > VPC to VPC connectivity</a> between your service VPC and the application proxy VPC to route traffic through the
     * application proxy to a service with a private URL endpoint. For more information, see <a href=
     * "https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/getting-started-create-application.html"
     * > Create an application</a> in the <i>Refactor Spaces User Guide</i>.
     * </p>
     *
     * @param createApplicationRequest
     * @return A Java Future containing the result of the CreateApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.CreateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/CreateApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateApplicationResponse> createApplication(CreateApplicationRequest createApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateApplicationRequest, CreateApplicationResponse>()
                            .withOperationName("CreateApplication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createApplicationRequest));
            CompletableFuture<CreateApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon Web Services Migration Hub Refactor Spaces environment. The caller owns the environment
     * resource, and all Refactor Spaces applications, services, and routes created within the environment. They are
     * referred to as the <i>environment owner</i>. The environment owner has cross-account visibility and control of
     * Refactor Spaces resources that are added to the environment by other accounts that the environment is shared
     * with.
     * </p>
     * <p>
     * When creating an environment with a <a href=
     * "https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateEnvironment.html#migrationhubrefactorspaces-CreateEnvironment-request-NetworkFabricType"
     * >CreateEnvironment:NetworkFabricType</a> of <code>TRANSIT_GATEWAY</code>, Refactor Spaces provisions a transit
     * gateway to enable services in VPCs to communicate directly across accounts. If <a href=
     * "https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateEnvironment.html#migrationhubrefactorspaces-CreateEnvironment-request-NetworkFabricType"
     * >CreateEnvironment:NetworkFabricType</a> is <code>NONE</code>, Refactor Spaces does not create a transit gateway
     * and you must use your network infrastructure to route traffic to services with private URL endpoints.
     * </p>
     *
     * @param createEnvironmentRequest
     * @return A Java Future containing the result of the CreateEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.CreateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/CreateEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEnvironmentResponse> createEnvironment(CreateEnvironmentRequest createEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEnvironmentRequest, CreateEnvironmentResponse>()
                            .withOperationName("CreateEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createEnvironmentRequest));
            CompletableFuture<CreateEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon Web Services Migration Hub Refactor Spaces route. The account owner of the service resource is
     * always the environment owner, regardless of which account creates the route. Routes target a service in the
     * application. If an application does not have any routes, then the first route must be created as a
     * <code>DEFAULT</code> <code>RouteType</code>.
     * </p>
     * <p>
     * When created, the default route defaults to an active state so state is not a required input. However, like all
     * other state values the state of the default route can be updated after creation, but only when all other routes
     * are also inactive. Conversely, no route can be active without the default route also being active.
     * </p>
     * <p>
     * When you create a route, Refactor Spaces configures the Amazon API Gateway to send traffic to the target service
     * as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>URL Endpoints</b>
     * </p>
     * <p>
     * If the service has a URL endpoint, and the endpoint resolves to a private IP address, Refactor Spaces routes
     * traffic using the API Gateway VPC link. If a service endpoint resolves to a public IP address, Refactor Spaces
     * routes traffic over the public internet. Services can have HTTP or HTTPS URL endpoints. For HTTPS URLs,
     * publicly-signed certificates are supported. Private Certificate Authorities (CAs) are permitted only if the CA's
     * domain is also publicly resolvable.
     * </p>
     * <p>
     * Refactor Spaces automatically resolves the public Domain Name System (DNS) names that are set in
     * <code>CreateService:UrlEndpoint </code>when you create a service. The DNS names resolve when the DNS time-to-live
     * (TTL) expires, or every 60 seconds for TTLs less than 60 seconds. This periodic DNS resolution ensures that the
     * route configuration remains up-to-date.
     * </p>
     * <p/>
     * <p>
     * <b>One-time health check</b>
     * </p>
     * <p>
     * A one-time health check is performed on the service when either the route is updated from inactive to active, or
     * when it is created with an active state. If the health check fails, the route transitions the route state to
     * <code>FAILED</code>, an error code of <code>SERVICE_ENDPOINT_HEALTH_CHECK_FAILURE</code> is provided, and no
     * traffic is sent to the service.
     * </p>
     * <p>
     * For private URLs, a target group is created on the Network Load Balancer and the load balancer target group runs
     * default target health checks. By default, the health check is run against the service endpoint URL. Optionally,
     * the health check can be performed against a different protocol, port, and/or path using the <a href=
     * "https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateService.html#migrationhubrefactorspaces-CreateService-request-UrlEndpoint"
     * >CreateService:UrlEndpoint</a> parameter. All other health check settings for the load balancer use the default
     * values described in the <a
     * href="https://docs.aws.amazon.com/elasticloadbalancing/latest/application/target-group-health-checks.html">Health
     * checks for your target groups</a> in the <i>Elastic Load Balancing guide</i>. The health check is considered
     * successful if at least one target within the target group transitions to a healthy state.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <b>Lambda function endpoints</b>
     * </p>
     * <p>
     * If the service has an Lambda function endpoint, then Refactor Spaces configures the Lambda function's resource
     * policy to allow the application's API Gateway to invoke the function.
     * </p>
     * <p>
     * The Lambda function state is checked. If the function is not active, the function configuration is updated so
     * that Lambda resources are provisioned. If the Lambda state is <code>Failed</code>, then the route creation fails.
     * For more information, see the <a href=
     * "https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunctionConfiguration.html#SSS-GetFunctionConfiguration-response-State"
     * >GetFunctionConfiguration's State response parameter</a> in the <i>Lambda Developer Guide</i>.
     * </p>
     * <p>
     * A check is performed to determine that a Lambda function with the specified ARN exists. If it does not exist, the
     * health check fails. For public URLs, a connection is opened to the public endpoint. If the URL is not reachable,
     * the health check fails.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Environments without a network bridge</b>
     * </p>
     * <p>
     * When you create environments without a network bridge (<a href=
     * "https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateEnvironment.html#migrationhubrefactorspaces-CreateEnvironment-request-NetworkFabricType"
     * >CreateEnvironment:NetworkFabricType</a> is <code>NONE)</code> and you use your own networking infrastructure,
     * you need to configure <a href=
     * "https://docs.aws.amazon.com/whitepapers/latest/aws-vpc-connectivity-options/amazon-vpc-to-amazon-vpc-connectivity-options.html"
     * >VPC to VPC connectivity</a> between your network and the application proxy VPC. Route creation from the
     * application proxy to service endpoints will fail if your network is not configured to connect to the application
     * proxy VPC. For more information, see <a href=
     * "https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/getting-started-create-role.html">
     * Create a route</a> in the <i>Refactor Spaces User Guide</i>.
     * </p>
     * <p/>
     *
     * @param createRouteRequest
     * @return A Java Future containing the result of the CreateRoute operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li> <li>
     *         InternalServerException An unexpected error occurred while processing the request.</li> <li>
     *         ConflictException Updating or deleting a resource can cause an inconsistent state.</li> <li>
     *         ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li> <li>
     *         ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li> <li>
     *         AccessDeniedException The user does not have sufficient access to perform this action.</li> <li>
     *         SdkException Base class for all exceptions that can be thrown by the SDK (both service and client). Can
     *         be used for catch all scenarios.</li> <li>SdkClientException If any client side error occurs such as an
     *         IO related failure, failure to get credentials, etc.</li> <li>MigrationHubRefactorSpacesException Base
     *         class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.CreateRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/CreateRoute"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRouteResponse> createRoute(CreateRouteRequest createRouteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRouteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRoute");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRouteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateRouteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRouteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRouteRequest, CreateRouteResponse>()
                            .withOperationName("CreateRoute").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateRouteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createRouteRequest));
            CompletableFuture<CreateRouteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon Web Services Migration Hub Refactor Spaces service. The account owner of the service is always
     * the environment owner, regardless of which account in the environment creates the service. Services have either a
     * URL endpoint in a virtual private cloud (VPC), or a Lambda function endpoint.
     * </p>
     * <important>
     * <p>
     * If an Amazon Web Services resource is launched in a service VPC, and you want it to be accessible to all of an
     * environment’s services with VPCs and routes, apply the <code>RefactorSpacesSecurityGroup</code> to the resource.
     * Alternatively, to add more cross-account constraints, apply your own security group.
     * </p>
     * </important>
     *
     * @param createServiceRequest
     * @return A Java Future containing the result of the CreateService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.CreateService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/CreateService"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateServiceResponse> createService(CreateServiceRequest createServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createServiceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateServiceRequest, CreateServiceResponse>()
                            .withOperationName("CreateService").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createServiceRequest));
            CompletableFuture<CreateServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon Web Services Migration Hub Refactor Spaces application. Before you can delete an application,
     * you must first delete any services or routes within the application.
     * </p>
     *
     * @param deleteApplicationRequest
     * @return A Java Future containing the result of the DeleteApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.DeleteApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/DeleteApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteApplicationResponse> deleteApplication(DeleteApplicationRequest deleteApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteApplicationRequest, DeleteApplicationResponse>()
                            .withOperationName("DeleteApplication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteApplicationRequest));
            CompletableFuture<DeleteApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon Web Services Migration Hub Refactor Spaces environment. Before you can delete an environment,
     * you must first delete any applications and services within the environment.
     * </p>
     *
     * @param deleteEnvironmentRequest
     * @return A Java Future containing the result of the DeleteEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.DeleteEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/DeleteEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEnvironmentResponse> deleteEnvironment(DeleteEnvironmentRequest deleteEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEnvironmentRequest, DeleteEnvironmentResponse>()
                            .withOperationName("DeleteEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteEnvironmentRequest));
            CompletableFuture<DeleteEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the resource policy set for the environment.
     * </p>
     *
     * @param deleteResourcePolicyRequest
     * @return A Java Future containing the result of the DeleteResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.DeleteResourcePolicy
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/DeleteResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResourcePolicyResponse> deleteResourcePolicy(
            DeleteResourcePolicyRequest deleteResourcePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResourcePolicyRequest, DeleteResourcePolicyResponse>()
                            .withOperationName("DeleteResourcePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteResourcePolicyRequest));
            CompletableFuture<DeleteResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon Web Services Migration Hub Refactor Spaces route.
     * </p>
     *
     * @param deleteRouteRequest
     * @return A Java Future containing the result of the DeleteRoute operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.DeleteRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/DeleteRoute"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRouteResponse> deleteRoute(DeleteRouteRequest deleteRouteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRouteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRoute");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRouteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteRouteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRouteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRouteRequest, DeleteRouteResponse>()
                            .withOperationName("DeleteRoute").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRouteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRouteRequest));
            CompletableFuture<DeleteRouteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon Web Services Migration Hub Refactor Spaces service.
     * </p>
     *
     * @param deleteServiceRequest
     * @return A Java Future containing the result of the DeleteService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.DeleteService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/DeleteService"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteServiceResponse> deleteService(DeleteServiceRequest deleteServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteServiceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteServiceRequest, DeleteServiceResponse>()
                            .withOperationName("DeleteService").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteServiceRequest));
            CompletableFuture<DeleteServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets an Amazon Web Services Migration Hub Refactor Spaces application.
     * </p>
     *
     * @param getApplicationRequest
     * @return A Java Future containing the result of the GetApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.GetApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/GetApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetApplicationResponse> getApplication(GetApplicationRequest getApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getApplicationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetApplicationRequest, GetApplicationResponse>()
                            .withOperationName("GetApplication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getApplicationRequest));
            CompletableFuture<GetApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets an Amazon Web Services Migration Hub Refactor Spaces environment.
     * </p>
     *
     * @param getEnvironmentRequest
     * @return A Java Future containing the result of the GetEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.GetEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/GetEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEnvironmentResponse> getEnvironment(GetEnvironmentRequest getEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEnvironmentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEnvironmentRequest, GetEnvironmentResponse>()
                            .withOperationName("GetEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEnvironmentRequest));
            CompletableFuture<GetEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the resource-based permission policy that is set for the given environment.
     * </p>
     *
     * @param getResourcePolicyRequest
     * @return A Java Future containing the result of the GetResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.GetResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/GetResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourcePolicyResponse> getResourcePolicy(GetResourcePolicyRequest getResourcePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourcePolicyRequest, GetResourcePolicyResponse>()
                            .withOperationName("GetResourcePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourcePolicyRequest));
            CompletableFuture<GetResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets an Amazon Web Services Migration Hub Refactor Spaces route.
     * </p>
     *
     * @param getRouteRequest
     * @return A Java Future containing the result of the GetRoute operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.GetRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/GetRoute"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRouteResponse> getRoute(GetRouteRequest getRouteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRouteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRoute");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRouteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetRouteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRouteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRouteRequest, GetRouteResponse>().withOperationName("GetRoute")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRouteRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getRouteRequest));
            CompletableFuture<GetRouteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets an Amazon Web Services Migration Hub Refactor Spaces service.
     * </p>
     *
     * @param getServiceRequest
     * @return A Java Future containing the result of the GetService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.GetService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/GetService"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceResponse> getService(GetServiceRequest getServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getServiceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceRequest, GetServiceResponse>().withOperationName("GetService")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getServiceRequest));
            CompletableFuture<GetServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the Amazon Web Services Migration Hub Refactor Spaces applications within an environment.
     * </p>
     *
     * @param listApplicationsRequest
     * @return A Java Future containing the result of the ListApplications operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.ListApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/ListApplications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListApplicationsResponse> listApplications(ListApplicationsRequest listApplicationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listApplicationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplications");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListApplicationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListApplicationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApplicationsRequest, ListApplicationsResponse>()
                            .withOperationName("ListApplications").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListApplicationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listApplicationsRequest));
            CompletableFuture<ListApplicationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all Amazon Web Services Migration Hub Refactor Spaces service virtual private clouds (VPCs) that are part
     * of the environment.
     * </p>
     *
     * @param listEnvironmentVpcsRequest
     * @return A Java Future containing the result of the ListEnvironmentVpcs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.ListEnvironmentVpcs
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/ListEnvironmentVpcs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEnvironmentVpcsResponse> listEnvironmentVpcs(
            ListEnvironmentVpcsRequest listEnvironmentVpcsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEnvironmentVpcsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEnvironmentVpcsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEnvironmentVpcs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEnvironmentVpcsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEnvironmentVpcsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEnvironmentVpcsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEnvironmentVpcsRequest, ListEnvironmentVpcsResponse>()
                            .withOperationName("ListEnvironmentVpcs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEnvironmentVpcsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEnvironmentVpcsRequest));
            CompletableFuture<ListEnvironmentVpcsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists Amazon Web Services Migration Hub Refactor Spaces environments owned by a caller account or shared with the
     * caller account.
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return A Java Future containing the result of the ListEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/ListEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEnvironmentsResponse> listEnvironments(ListEnvironmentsRequest listEnvironmentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEnvironmentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEnvironmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEnvironments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEnvironmentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEnvironmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEnvironmentsRequest, ListEnvironmentsResponse>()
                            .withOperationName("ListEnvironments").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEnvironmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEnvironmentsRequest));
            CompletableFuture<ListEnvironmentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the Amazon Web Services Migration Hub Refactor Spaces routes within an application.
     * </p>
     *
     * @param listRoutesRequest
     * @return A Java Future containing the result of the ListRoutes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.ListRoutes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/ListRoutes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRoutesResponse> listRoutes(ListRoutesRequest listRoutesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRoutesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRoutesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRoutes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRoutesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListRoutesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRoutesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRoutesRequest, ListRoutesResponse>().withOperationName("ListRoutes")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRoutesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRoutesRequest));
            CompletableFuture<ListRoutesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the Amazon Web Services Migration Hub Refactor Spaces services within an application.
     * </p>
     *
     * @param listServicesRequest
     * @return A Java Future containing the result of the ListServices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.ListServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/ListServices"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServicesResponse> listServices(ListServicesRequest listServicesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServicesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListServicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServicesRequest, ListServicesResponse>()
                            .withOperationName("ListServices").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListServicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listServicesRequest));
            CompletableFuture<ListServicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags of a resource. The caller account must be the same as the resource’s <code>OwnerAccountId</code>.
     * Listing tags in other accounts is not supported.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.ListTagsForResource
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attaches a resource-based permission policy to the Amazon Web Services Migration Hub Refactor Spaces environment.
     * The policy must contain the same actions and condition statements as the
     * <code>arn:aws:ram::aws:permission/AWSRAMDefaultPermissionRefactorSpacesEnvironment</code> permission in Resource
     * Access Manager. The policy must not contain new lines or blank lines.
     * </p>
     *
     * @param putResourcePolicyRequest
     * @return A Java Future containing the result of the PutResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>InvalidResourcePolicyException The resource policy is not valid.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.PutResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/PutResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutResourcePolicyResponse> putResourcePolicy(PutResourcePolicyRequest putResourcePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutResourcePolicyRequest, PutResourcePolicyResponse>()
                            .withOperationName("PutResourcePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putResourcePolicyRequest));
            CompletableFuture<PutResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the tags of a given resource. Tags are metadata which can be used to manage a resource. To tag a
     * resource, the caller account must be the same as the resource’s <code>OwnerAccountId</code>. Tagging resources in
     * other accounts is not supported.
     * </p>
     * <note>
     * <p>
     * Amazon Web Services Migration Hub Refactor Spaces does not propagate tags to orchestrated resources, such as an
     * environment’s transit gateway.
     * </p>
     * </note>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds to or modifies the tags of the given resource. Tags are metadata which can be used to manage a resource. To
     * untag a resource, the caller account must be the same as the resource’s <code>OwnerAccountId</code>. Untagging
     * resources across accounts is not supported.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an Amazon Web Services Migration Hub Refactor Spaces route.
     * </p>
     *
     * @param updateRouteRequest
     * @return A Java Future containing the result of the UpdateRoute operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>InternalServerException An unexpected error occurred while processing the request.</li>
     *         <li>ValidationException The input does not satisfy the constraints specified by an Amazon Web Service.</li>
     *         <li>ThrottlingException Request was denied because the request was throttled.</li>
     *         <li>AccessDeniedException The user does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubRefactorSpacesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubRefactorSpacesAsyncClient.UpdateRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migration-hub-refactor-spaces-2021-10-26/UpdateRoute"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRouteResponse> updateRoute(UpdateRouteRequest updateRouteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRouteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub Refactor Spaces");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRoute");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRouteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateRouteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRouteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRouteRequest, UpdateRouteResponse>()
                            .withOperationName("UpdateRoute").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateRouteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateRouteRequest));
            CompletableFuture<UpdateRouteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final MigrationHubRefactorSpacesServiceClientConfiguration serviceClientConfiguration() {
        return new MigrationHubRefactorSpacesServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MigrationHubRefactorSpacesException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourcePolicyException")
                                .exceptionBuilderSupplier(InvalidResourcePolicyException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        MigrationHubRefactorSpacesServiceClientConfigurationBuilder serviceConfigBuilder = new MigrationHubRefactorSpacesServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
