/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendraranking;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.kendraranking.model.AccessDeniedException;
import software.amazon.awssdk.services.kendraranking.model.ConflictException;
import software.amazon.awssdk.services.kendraranking.model.CreateRescoreExecutionPlanRequest;
import software.amazon.awssdk.services.kendraranking.model.CreateRescoreExecutionPlanResponse;
import software.amazon.awssdk.services.kendraranking.model.DeleteRescoreExecutionPlanRequest;
import software.amazon.awssdk.services.kendraranking.model.DeleteRescoreExecutionPlanResponse;
import software.amazon.awssdk.services.kendraranking.model.DescribeRescoreExecutionPlanRequest;
import software.amazon.awssdk.services.kendraranking.model.DescribeRescoreExecutionPlanResponse;
import software.amazon.awssdk.services.kendraranking.model.InternalServerException;
import software.amazon.awssdk.services.kendraranking.model.KendraRankingException;
import software.amazon.awssdk.services.kendraranking.model.KendraRankingRequest;
import software.amazon.awssdk.services.kendraranking.model.ListRescoreExecutionPlansRequest;
import software.amazon.awssdk.services.kendraranking.model.ListRescoreExecutionPlansResponse;
import software.amazon.awssdk.services.kendraranking.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.kendraranking.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.kendraranking.model.RescoreRequest;
import software.amazon.awssdk.services.kendraranking.model.RescoreResponse;
import software.amazon.awssdk.services.kendraranking.model.ResourceNotFoundException;
import software.amazon.awssdk.services.kendraranking.model.ResourceUnavailableException;
import software.amazon.awssdk.services.kendraranking.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.kendraranking.model.TagResourceRequest;
import software.amazon.awssdk.services.kendraranking.model.TagResourceResponse;
import software.amazon.awssdk.services.kendraranking.model.ThrottlingException;
import software.amazon.awssdk.services.kendraranking.model.UntagResourceRequest;
import software.amazon.awssdk.services.kendraranking.model.UntagResourceResponse;
import software.amazon.awssdk.services.kendraranking.model.UpdateRescoreExecutionPlanRequest;
import software.amazon.awssdk.services.kendraranking.model.UpdateRescoreExecutionPlanResponse;
import software.amazon.awssdk.services.kendraranking.model.ValidationException;
import software.amazon.awssdk.services.kendraranking.paginators.ListRescoreExecutionPlansIterable;
import software.amazon.awssdk.services.kendraranking.transform.CreateRescoreExecutionPlanRequestMarshaller;
import software.amazon.awssdk.services.kendraranking.transform.DeleteRescoreExecutionPlanRequestMarshaller;
import software.amazon.awssdk.services.kendraranking.transform.DescribeRescoreExecutionPlanRequestMarshaller;
import software.amazon.awssdk.services.kendraranking.transform.ListRescoreExecutionPlansRequestMarshaller;
import software.amazon.awssdk.services.kendraranking.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.kendraranking.transform.RescoreRequestMarshaller;
import software.amazon.awssdk.services.kendraranking.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.kendraranking.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.kendraranking.transform.UpdateRescoreExecutionPlanRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link KendraRankingClient}.
 *
 * @see KendraRankingClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKendraRankingClient implements KendraRankingClient {
    private static final Logger log = Logger.loggerFor(DefaultKendraRankingClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final KendraRankingServiceClientConfiguration serviceClientConfiguration;

    protected DefaultKendraRankingClient(KendraRankingServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a rescore execution plan. A rescore execution plan is an Amazon Kendra Intelligent Ranking resource used
     * for provisioning the <code>Rescore</code> API. You set the number of capacity units that you require for Amazon
     * Kendra Intelligent Ranking to rescore or re-rank a search service's results.
     * </p>
     * <p>
     * For an example of using the <code>CreateRescoreExecutionPlan</code> API, including using the Python and Java
     * SDKs, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/search-service-rerank.html">Semantically ranking
     * a search service's results</a>.
     * </p>
     *
     * @param createRescoreExecutionPlanRequest
     * @return Result of the CreateRescoreExecutionPlan operation returned by the service.
     * @throws AccessDeniedException
     *         You don’t have sufficient access to perform this action. Please ensure you have the required permission
     *         policies and user accounts and try again.
     * @throws ConflictException
     *         A conflict occurred with the request. Please fix any inconsistencies with your resources and try again.
     * @throws ServiceQuotaExceededException
     *         You have exceeded the set limits for your Amazon Kendra Intelligent Ranking service. Please see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/quotas.html">Quotas</a> for more information, or
     *         contact <a href="http://aws.amazon.com/contact-us/">Support</a> to inquire about an increase of limits.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please reduce the number of requests and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints set by the Amazon Kendra Intelligent Ranking service. Please
     *         provide the correct input and try again.
     * @throws InternalServerException
     *         An issue occurred with the internal server used for your Amazon Kendra Intelligent Ranking service.
     *         Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraRankingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraRankingClient.CreateRescoreExecutionPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/CreateRescoreExecutionPlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateRescoreExecutionPlanResponse createRescoreExecutionPlan(
            CreateRescoreExecutionPlanRequest createRescoreExecutionPlanRequest) throws AccessDeniedException, ConflictException,
            ServiceQuotaExceededException, ThrottlingException, ValidationException, InternalServerException,
            AwsServiceException, SdkClientException, KendraRankingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRescoreExecutionPlanResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateRescoreExecutionPlanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRescoreExecutionPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRescoreExecutionPlan");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateRescoreExecutionPlanRequest, CreateRescoreExecutionPlanResponse>()
                            .withOperationName("CreateRescoreExecutionPlan").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createRescoreExecutionPlanRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateRescoreExecutionPlanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a rescore execution plan. A rescore execution plan is an Amazon Kendra Intelligent Ranking resource used
     * for provisioning the <code>Rescore</code> API.
     * </p>
     *
     * @param deleteRescoreExecutionPlanRequest
     * @return Result of the DeleteRescoreExecutionPlan operation returned by the service.
     * @throws AccessDeniedException
     *         You don’t have sufficient access to perform this action. Please ensure you have the required permission
     *         policies and user accounts and try again.
     * @throws ValidationException
     *         The input fails to satisfy the constraints set by the Amazon Kendra Intelligent Ranking service. Please
     *         provide the correct input and try again.
     * @throws ConflictException
     *         A conflict occurred with the request. Please fix any inconsistencies with your resources and try again.
     * @throws ResourceNotFoundException
     *         The resource you want to use doesn't exist. Please check you have provided the correct resource and try
     *         again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please reduce the number of requests and try again.
     * @throws InternalServerException
     *         An issue occurred with the internal server used for your Amazon Kendra Intelligent Ranking service.
     *         Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraRankingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraRankingClient.DeleteRescoreExecutionPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/DeleteRescoreExecutionPlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteRescoreExecutionPlanResponse deleteRescoreExecutionPlan(
            DeleteRescoreExecutionPlanRequest deleteRescoreExecutionPlanRequest) throws AccessDeniedException,
            ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException, InternalServerException,
            AwsServiceException, SdkClientException, KendraRankingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRescoreExecutionPlanResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRescoreExecutionPlanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRescoreExecutionPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRescoreExecutionPlan");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteRescoreExecutionPlanRequest, DeleteRescoreExecutionPlanResponse>()
                            .withOperationName("DeleteRescoreExecutionPlan").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteRescoreExecutionPlanRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteRescoreExecutionPlanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a rescore execution plan. A rescore execution plan is an Amazon Kendra Intelligent Ranking
     * resource used for provisioning the <code>Rescore</code> API.
     * </p>
     *
     * @param describeRescoreExecutionPlanRequest
     * @return Result of the DescribeRescoreExecutionPlan operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints set by the Amazon Kendra Intelligent Ranking service. Please
     *         provide the correct input and try again.
     * @throws ResourceNotFoundException
     *         The resource you want to use doesn't exist. Please check you have provided the correct resource and try
     *         again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please reduce the number of requests and try again.
     * @throws AccessDeniedException
     *         You don’t have sufficient access to perform this action. Please ensure you have the required permission
     *         policies and user accounts and try again.
     * @throws InternalServerException
     *         An issue occurred with the internal server used for your Amazon Kendra Intelligent Ranking service.
     *         Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraRankingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraRankingClient.DescribeRescoreExecutionPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/DescribeRescoreExecutionPlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeRescoreExecutionPlanResponse describeRescoreExecutionPlan(
            DescribeRescoreExecutionPlanRequest describeRescoreExecutionPlanRequest) throws ValidationException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, KendraRankingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeRescoreExecutionPlanResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeRescoreExecutionPlanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRescoreExecutionPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRescoreExecutionPlan");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeRescoreExecutionPlanRequest, DescribeRescoreExecutionPlanResponse>()
                            .withOperationName("DescribeRescoreExecutionPlan").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeRescoreExecutionPlanRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeRescoreExecutionPlanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists your rescore execution plans. A rescore execution plan is an Amazon Kendra Intelligent Ranking resource
     * used for provisioning the <code>Rescore</code> API.
     * </p>
     *
     * @param listRescoreExecutionPlansRequest
     * @return Result of the ListRescoreExecutionPlans operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints set by the Amazon Kendra Intelligent Ranking service. Please
     *         provide the correct input and try again.
     * @throws AccessDeniedException
     *         You don’t have sufficient access to perform this action. Please ensure you have the required permission
     *         policies and user accounts and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please reduce the number of requests and try again.
     * @throws InternalServerException
     *         An issue occurred with the internal server used for your Amazon Kendra Intelligent Ranking service.
     *         Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraRankingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraRankingClient.ListRescoreExecutionPlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/ListRescoreExecutionPlans"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRescoreExecutionPlansResponse listRescoreExecutionPlans(
            ListRescoreExecutionPlansRequest listRescoreExecutionPlansRequest) throws ValidationException, AccessDeniedException,
            ThrottlingException, InternalServerException, AwsServiceException, SdkClientException, KendraRankingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRescoreExecutionPlansResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRescoreExecutionPlansResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRescoreExecutionPlansRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRescoreExecutionPlans");

            return clientHandler
                    .execute(new ClientExecutionParams<ListRescoreExecutionPlansRequest, ListRescoreExecutionPlansResponse>()
                            .withOperationName("ListRescoreExecutionPlans").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listRescoreExecutionPlansRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListRescoreExecutionPlansRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists your rescore execution plans. A rescore execution plan is an Amazon Kendra Intelligent Ranking resource
     * used for provisioning the <code>Rescore</code> API.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRescoreExecutionPlans(software.amazon.awssdk.services.kendraranking.model.ListRescoreExecutionPlansRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kendraranking.paginators.ListRescoreExecutionPlansIterable responses = client.listRescoreExecutionPlansPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.kendraranking.paginators.ListRescoreExecutionPlansIterable responses = client
     *             .listRescoreExecutionPlansPaginator(request);
     *     for (software.amazon.awssdk.services.kendraranking.model.ListRescoreExecutionPlansResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kendraranking.paginators.ListRescoreExecutionPlansIterable responses = client.listRescoreExecutionPlansPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRescoreExecutionPlans(software.amazon.awssdk.services.kendraranking.model.ListRescoreExecutionPlansRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRescoreExecutionPlansRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the constraints set by the Amazon Kendra Intelligent Ranking service. Please
     *         provide the correct input and try again.
     * @throws AccessDeniedException
     *         You don’t have sufficient access to perform this action. Please ensure you have the required permission
     *         policies and user accounts and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please reduce the number of requests and try again.
     * @throws InternalServerException
     *         An issue occurred with the internal server used for your Amazon Kendra Intelligent Ranking service.
     *         Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraRankingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraRankingClient.ListRescoreExecutionPlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/ListRescoreExecutionPlans"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRescoreExecutionPlansIterable listRescoreExecutionPlansPaginator(
            ListRescoreExecutionPlansRequest listRescoreExecutionPlansRequest) throws ValidationException, AccessDeniedException,
            ThrottlingException, InternalServerException, AwsServiceException, SdkClientException, KendraRankingException {
        return new ListRescoreExecutionPlansIterable(this, applyPaginatorUserAgent(listRescoreExecutionPlansRequest));
    }

    /**
     * <p>
     * Gets a list of tags associated with a specified resource. A rescore execution plan is an example of a resource
     * that can have tags associated with it.
     * </p>
     *
     * @param listTagsForResourceRequest
     *        The request information for listing tags associated with a rescore execution plan. A rescore execution
     *        plan is an Amazon Kendra Intelligent Ranking resource used for provisioning the <code>Rescore</code> API.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints set by the Amazon Kendra Intelligent Ranking service. Please
     *         provide the correct input and try again.
     * @throws ResourceUnavailableException
     *         The resource you want to use is unavailable. Please check you have provided the correct resource
     *         information and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please reduce the number of requests and try again.
     * @throws AccessDeniedException
     *         You don’t have sufficient access to perform this action. Please ensure you have the required permission
     *         policies and user accounts and try again.
     * @throws InternalServerException
     *         An issue occurred with the internal server used for your Amazon Kendra Intelligent Ranking service.
     *         Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraRankingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraRankingClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, ResourceUnavailableException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, KendraRankingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Rescores or re-ranks search results from a search service such as OpenSearch (self managed). You use the semantic
     * search capabilities of Amazon Kendra Intelligent Ranking to improve the search service's results.
     * </p>
     *
     * @param rescoreRequest
     * @return Result of the Rescore operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints set by the Amazon Kendra Intelligent Ranking service. Please
     *         provide the correct input and try again.
     * @throws ConflictException
     *         A conflict occurred with the request. Please fix any inconsistencies with your resources and try again.
     * @throws AccessDeniedException
     *         You don’t have sufficient access to perform this action. Please ensure you have the required permission
     *         policies and user accounts and try again.
     * @throws ResourceNotFoundException
     *         The resource you want to use doesn't exist. Please check you have provided the correct resource and try
     *         again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please reduce the number of requests and try again.
     * @throws InternalServerException
     *         An issue occurred with the internal server used for your Amazon Kendra Intelligent Ranking service.
     *         Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraRankingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraRankingClient.Rescore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/Rescore" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RescoreResponse rescore(RescoreRequest rescoreRequest) throws ValidationException, ConflictException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, KendraRankingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RescoreResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RescoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rescoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Rescore");

            return clientHandler.execute(new ClientExecutionParams<RescoreRequest, RescoreResponse>()
                    .withOperationName("Rescore").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(rescoreRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new RescoreRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds a specified tag to a specified rescore execution plan. A rescore execution plan is an Amazon Kendra
     * Intelligent Ranking resource used for provisioning the <code>Rescore</code> API. If the tag already exists, the
     * existing value is replaced with the new value.
     * </p>
     *
     * @param tagResourceRequest
     *        The request information for tagging a rescore execution plan. A rescore execution plan is an Amazon Kendra
     *        Intelligent Ranking resource used for provisioning the <code>Rescore</code> API.
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints set by the Amazon Kendra Intelligent Ranking service. Please
     *         provide the correct input and try again.
     * @throws ResourceUnavailableException
     *         The resource you want to use is unavailable. Please check you have provided the correct resource
     *         information and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please reduce the number of requests and try again.
     * @throws AccessDeniedException
     *         You don’t have sufficient access to perform this action. Please ensure you have the required permission
     *         policies and user accounts and try again.
     * @throws InternalServerException
     *         An issue occurred with the internal server used for your Amazon Kendra Intelligent Ranking service.
     *         Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraRankingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraRankingClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            ResourceUnavailableException, ThrottlingException, AccessDeniedException, InternalServerException,
            AwsServiceException, SdkClientException, KendraRankingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a tag from a rescore execution plan. A rescore execution plan is an Amazon Kendra Intelligent Ranking
     * resource used for provisioning the <code>Rescore</code> operation.
     * </p>
     *
     * @param untagResourceRequest
     *        The request information to remove a tag from a rescore execution plan. A rescore execution plan is an
     *        Amazon Kendra Intelligent Ranking resource used for provisioning the <code>Rescore</code> API.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints set by the Amazon Kendra Intelligent Ranking service. Please
     *         provide the correct input and try again.
     * @throws ResourceUnavailableException
     *         The resource you want to use is unavailable. Please check you have provided the correct resource
     *         information and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please reduce the number of requests and try again.
     * @throws AccessDeniedException
     *         You don’t have sufficient access to perform this action. Please ensure you have the required permission
     *         policies and user accounts and try again.
     * @throws InternalServerException
     *         An issue occurred with the internal server used for your Amazon Kendra Intelligent Ranking service.
     *         Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraRankingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraRankingClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ResourceUnavailableException, ThrottlingException, AccessDeniedException, InternalServerException,
            AwsServiceException, SdkClientException, KendraRankingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a rescore execution plan. A rescore execution plan is an Amazon Kendra Intelligent Ranking resource used
     * for provisioning the <code>Rescore</code> API. You can update the number of capacity units you require for Amazon
     * Kendra Intelligent Ranking to rescore or re-rank a search service's results.
     * </p>
     *
     * @param updateRescoreExecutionPlanRequest
     * @return Result of the UpdateRescoreExecutionPlan operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints set by the Amazon Kendra Intelligent Ranking service. Please
     *         provide the correct input and try again.
     * @throws ResourceNotFoundException
     *         The resource you want to use doesn't exist. Please check you have provided the correct resource and try
     *         again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please reduce the number of requests and try again.
     * @throws AccessDeniedException
     *         You don’t have sufficient access to perform this action. Please ensure you have the required permission
     *         policies and user accounts and try again.
     * @throws ServiceQuotaExceededException
     *         You have exceeded the set limits for your Amazon Kendra Intelligent Ranking service. Please see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/quotas.html">Quotas</a> for more information, or
     *         contact <a href="http://aws.amazon.com/contact-us/">Support</a> to inquire about an increase of limits.
     * @throws ConflictException
     *         A conflict occurred with the request. Please fix any inconsistencies with your resources and try again.
     * @throws InternalServerException
     *         An issue occurred with the internal server used for your Amazon Kendra Intelligent Ranking service.
     *         Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraRankingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraRankingClient.UpdateRescoreExecutionPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/UpdateRescoreExecutionPlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateRescoreExecutionPlanResponse updateRescoreExecutionPlan(
            UpdateRescoreExecutionPlanRequest updateRescoreExecutionPlanRequest) throws ValidationException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, ServiceQuotaExceededException,
            ConflictException, InternalServerException, AwsServiceException, SdkClientException, KendraRankingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRescoreExecutionPlanResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateRescoreExecutionPlanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRescoreExecutionPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRescoreExecutionPlan");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateRescoreExecutionPlanRequest, UpdateRescoreExecutionPlanResponse>()
                            .withOperationName("UpdateRescoreExecutionPlan").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateRescoreExecutionPlanRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateRescoreExecutionPlanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends KendraRankingRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KendraRankingException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceUnavailableException")
                                .exceptionBuilderSupplier(ResourceUnavailableException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build());
    }

    @Override
    public final KendraRankingServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
