/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotfleetwise.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a node and its specifications in an in-vehicle communication network. All signal decoders must be
 * associated with a network node.
 * </p>
 * <p>
 * To return this information about all the network interfaces specified in a decoder manifest, use the API operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NetworkInterface implements SdkPojo, Serializable,
        ToCopyableBuilder<NetworkInterface.Builder, NetworkInterface> {
    private static final SdkField<String> INTERFACE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("interfaceId").getter(getter(NetworkInterface::interfaceId)).setter(setter(Builder::interfaceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("interfaceId").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(NetworkInterface::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<CanInterface> CAN_INTERFACE_FIELD = SdkField.<CanInterface> builder(MarshallingType.SDK_POJO)
            .memberName("canInterface").getter(getter(NetworkInterface::canInterface)).setter(setter(Builder::canInterface))
            .constructor(CanInterface::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("canInterface").build()).build();

    private static final SdkField<ObdInterface> OBD_INTERFACE_FIELD = SdkField.<ObdInterface> builder(MarshallingType.SDK_POJO)
            .memberName("obdInterface").getter(getter(NetworkInterface::obdInterface)).setter(setter(Builder::obdInterface))
            .constructor(ObdInterface::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("obdInterface").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INTERFACE_ID_FIELD,
            TYPE_FIELD, CAN_INTERFACE_FIELD, OBD_INTERFACE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String interfaceId;

    private final String type;

    private final CanInterface canInterface;

    private final ObdInterface obdInterface;

    private NetworkInterface(BuilderImpl builder) {
        this.interfaceId = builder.interfaceId;
        this.type = builder.type;
        this.canInterface = builder.canInterface;
        this.obdInterface = builder.obdInterface;
    }

    /**
     * <p>
     * The ID of the network interface.
     * </p>
     * 
     * @return The ID of the network interface.
     */
    public final String interfaceId() {
        return interfaceId;
    }

    /**
     * <p>
     * The network protocol for the vehicle. For example, <code>CAN_SIGNAL</code> specifies a protocol that defines how
     * data is communicated between electronic control units (ECUs). <code>OBD_SIGNAL</code> specifies a protocol that
     * defines how self-diagnostic data is communicated between ECUs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link NetworkInterfaceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The network protocol for the vehicle. For example, <code>CAN_SIGNAL</code> specifies a protocol that
     *         defines how data is communicated between electronic control units (ECUs). <code>OBD_SIGNAL</code>
     *         specifies a protocol that defines how self-diagnostic data is communicated between ECUs.
     * @see NetworkInterfaceType
     */
    public final NetworkInterfaceType type() {
        return NetworkInterfaceType.fromValue(type);
    }

    /**
     * <p>
     * The network protocol for the vehicle. For example, <code>CAN_SIGNAL</code> specifies a protocol that defines how
     * data is communicated between electronic control units (ECUs). <code>OBD_SIGNAL</code> specifies a protocol that
     * defines how self-diagnostic data is communicated between ECUs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link NetworkInterfaceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The network protocol for the vehicle. For example, <code>CAN_SIGNAL</code> specifies a protocol that
     *         defines how data is communicated between electronic control units (ECUs). <code>OBD_SIGNAL</code>
     *         specifies a protocol that defines how self-diagnostic data is communicated between ECUs.
     * @see NetworkInterfaceType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * Information about a network interface specified by the Controller Area Network (CAN) protocol.
     * </p>
     * 
     * @return Information about a network interface specified by the Controller Area Network (CAN) protocol.
     */
    public final CanInterface canInterface() {
        return canInterface;
    }

    /**
     * <p>
     * Information about a network interface specified by the On-board diagnostic (OBD) II protocol.
     * </p>
     * 
     * @return Information about a network interface specified by the On-board diagnostic (OBD) II protocol.
     */
    public final ObdInterface obdInterface() {
        return obdInterface;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(interfaceId());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(canInterface());
        hashCode = 31 * hashCode + Objects.hashCode(obdInterface());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NetworkInterface)) {
            return false;
        }
        NetworkInterface other = (NetworkInterface) obj;
        return Objects.equals(interfaceId(), other.interfaceId()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(canInterface(), other.canInterface()) && Objects.equals(obdInterface(), other.obdInterface());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NetworkInterface").add("InterfaceId", interfaceId()).add("Type", typeAsString())
                .add("CanInterface", canInterface()).add("ObdInterface", obdInterface()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "interfaceId":
            return Optional.ofNullable(clazz.cast(interfaceId()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "canInterface":
            return Optional.ofNullable(clazz.cast(canInterface()));
        case "obdInterface":
            return Optional.ofNullable(clazz.cast(obdInterface()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<NetworkInterface, T> g) {
        return obj -> g.apply((NetworkInterface) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NetworkInterface> {
        /**
         * <p>
         * The ID of the network interface.
         * </p>
         * 
         * @param interfaceId
         *        The ID of the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder interfaceId(String interfaceId);

        /**
         * <p>
         * The network protocol for the vehicle. For example, <code>CAN_SIGNAL</code> specifies a protocol that defines
         * how data is communicated between electronic control units (ECUs). <code>OBD_SIGNAL</code> specifies a
         * protocol that defines how self-diagnostic data is communicated between ECUs.
         * </p>
         * 
         * @param type
         *        The network protocol for the vehicle. For example, <code>CAN_SIGNAL</code> specifies a protocol that
         *        defines how data is communicated between electronic control units (ECUs). <code>OBD_SIGNAL</code>
         *        specifies a protocol that defines how self-diagnostic data is communicated between ECUs.
         * @see NetworkInterfaceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkInterfaceType
         */
        Builder type(String type);

        /**
         * <p>
         * The network protocol for the vehicle. For example, <code>CAN_SIGNAL</code> specifies a protocol that defines
         * how data is communicated between electronic control units (ECUs). <code>OBD_SIGNAL</code> specifies a
         * protocol that defines how self-diagnostic data is communicated between ECUs.
         * </p>
         * 
         * @param type
         *        The network protocol for the vehicle. For example, <code>CAN_SIGNAL</code> specifies a protocol that
         *        defines how data is communicated between electronic control units (ECUs). <code>OBD_SIGNAL</code>
         *        specifies a protocol that defines how self-diagnostic data is communicated between ECUs.
         * @see NetworkInterfaceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkInterfaceType
         */
        Builder type(NetworkInterfaceType type);

        /**
         * <p>
         * Information about a network interface specified by the Controller Area Network (CAN) protocol.
         * </p>
         * 
         * @param canInterface
         *        Information about a network interface specified by the Controller Area Network (CAN) protocol.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder canInterface(CanInterface canInterface);

        /**
         * <p>
         * Information about a network interface specified by the Controller Area Network (CAN) protocol.
         * </p>
         * This is a convenience method that creates an instance of the {@link CanInterface.Builder} avoiding the need
         * to create one manually via {@link CanInterface#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CanInterface.Builder#build()} is called immediately and its
         * result is passed to {@link #canInterface(CanInterface)}.
         * 
         * @param canInterface
         *        a consumer that will call methods on {@link CanInterface.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #canInterface(CanInterface)
         */
        default Builder canInterface(Consumer<CanInterface.Builder> canInterface) {
            return canInterface(CanInterface.builder().applyMutation(canInterface).build());
        }

        /**
         * <p>
         * Information about a network interface specified by the On-board diagnostic (OBD) II protocol.
         * </p>
         * 
         * @param obdInterface
         *        Information about a network interface specified by the On-board diagnostic (OBD) II protocol.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder obdInterface(ObdInterface obdInterface);

        /**
         * <p>
         * Information about a network interface specified by the On-board diagnostic (OBD) II protocol.
         * </p>
         * This is a convenience method that creates an instance of the {@link ObdInterface.Builder} avoiding the need
         * to create one manually via {@link ObdInterface#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ObdInterface.Builder#build()} is called immediately and its
         * result is passed to {@link #obdInterface(ObdInterface)}.
         * 
         * @param obdInterface
         *        a consumer that will call methods on {@link ObdInterface.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #obdInterface(ObdInterface)
         */
        default Builder obdInterface(Consumer<ObdInterface.Builder> obdInterface) {
            return obdInterface(ObdInterface.builder().applyMutation(obdInterface).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String interfaceId;

        private String type;

        private CanInterface canInterface;

        private ObdInterface obdInterface;

        private BuilderImpl() {
        }

        private BuilderImpl(NetworkInterface model) {
            interfaceId(model.interfaceId);
            type(model.type);
            canInterface(model.canInterface);
            obdInterface(model.obdInterface);
        }

        public final String getInterfaceId() {
            return interfaceId;
        }

        public final void setInterfaceId(String interfaceId) {
            this.interfaceId = interfaceId;
        }

        @Override
        public final Builder interfaceId(String interfaceId) {
            this.interfaceId = interfaceId;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(NetworkInterfaceType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final CanInterface.Builder getCanInterface() {
            return canInterface != null ? canInterface.toBuilder() : null;
        }

        public final void setCanInterface(CanInterface.BuilderImpl canInterface) {
            this.canInterface = canInterface != null ? canInterface.build() : null;
        }

        @Override
        public final Builder canInterface(CanInterface canInterface) {
            this.canInterface = canInterface;
            return this;
        }

        public final ObdInterface.Builder getObdInterface() {
            return obdInterface != null ? obdInterface.toBuilder() : null;
        }

        public final void setObdInterface(ObdInterface.BuilderImpl obdInterface) {
            this.obdInterface = obdInterface != null ? obdInterface.build() : null;
        }

        @Override
        public final Builder obdInterface(ObdInterface obdInterface) {
            this.obdInterface = obdInterface;
            return this;
        }

        @Override
        public NetworkInterface build() {
            return new NetworkInterface(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
