/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotevents.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines an action to write to the Amazon DynamoDB table that you created. The default action payload contains all the
 * information about the detector model instance and the event that triggered the action. You can customize the <a
 * href="https://docs.aws.amazon.com/iotevents/latest/apireference/API_Payload.html">payload</a>. A separate column of
 * the DynamoDB table receives one attribute-value pair in the payload that you specify.
 * </p>
 * <p>
 * You must use expressions for all parameters in <code>DynamoDBv2Action</code>. The expressions accept literals,
 * operators, functions, references, and substitution templates.
 * </p>
 * <p class="title">
 * <b>Examples</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * For literal values, the expressions must contain single quotes. For example, the value for the <code>tableName</code>
 * parameter can be <code>'GreenhouseTemperatureTable'</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * For references, you must specify either variables or input values. For example, the value for the
 * <code>tableName</code> parameter can be <code>$variable.ddbtableName</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * For a substitution template, you must use <code>${}</code>, and the template must be in single quotes. A substitution
 * template can also contain a combination of literals, operators, functions, references, and substitution templates.
 * </p>
 * <p>
 * In the following example, the value for the <code>contentExpression</code> parameter in <code>Payload</code> uses a
 * substitution template.
 * </p>
 * <p>
 * <code>'{\"sensorID\": \"${$input.GreenhouseInput.sensor_id}\", \"temperature\": \"${$input.GreenhouseInput.temperature * 9 / 5 + 32}\"}'</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * For a string concatenation, you must use <code>+</code>. A string concatenation can also contain a combination of
 * literals, operators, functions, references, and substitution templates.
 * </p>
 * <p>
 * In the following example, the value for the <code>tableName</code> parameter uses a string concatenation.
 * </p>
 * <p>
 * <code>'GreenhouseTemperatureTable ' + $input.GreenhouseInput.date</code>
 * </p>
 * </li>
 * </ul>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-expressions.html">Expressions</a> in the
 * <i>AWS IoT Events Developer Guide</i>.
 * </p>
 * <p>
 * The value for the <code>type</code> parameter in <code>Payload</code> must be <code>JSON</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DynamoDBv2Action implements SdkPojo, Serializable,
        ToCopyableBuilder<DynamoDBv2Action.Builder, DynamoDBv2Action> {
    private static final SdkField<String> TABLE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("tableName").getter(getter(DynamoDBv2Action::tableName)).setter(setter(Builder::tableName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tableName").build()).build();

    private static final SdkField<Payload> PAYLOAD_FIELD = SdkField.<Payload> builder(MarshallingType.SDK_POJO)
            .memberName("payload").getter(getter(DynamoDBv2Action::payload)).setter(setter(Builder::payload))
            .constructor(Payload::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("payload").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TABLE_NAME_FIELD,
            PAYLOAD_FIELD));

    private static final long serialVersionUID = 1L;

    private final String tableName;

    private final Payload payload;

    private DynamoDBv2Action(BuilderImpl builder) {
        this.tableName = builder.tableName;
        this.payload = builder.payload;
    }

    /**
     * <p>
     * The name of the DynamoDB table.
     * </p>
     * 
     * @return The name of the DynamoDB table.
     */
    public final String tableName() {
        return tableName;
    }

    /**
     * Returns the value of the Payload property for this object.
     * 
     * @return The value of the Payload property for this object.
     */
    public final Payload payload() {
        return payload;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(tableName());
        hashCode = 31 * hashCode + Objects.hashCode(payload());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DynamoDBv2Action)) {
            return false;
        }
        DynamoDBv2Action other = (DynamoDBv2Action) obj;
        return Objects.equals(tableName(), other.tableName()) && Objects.equals(payload(), other.payload());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DynamoDBv2Action").add("TableName", tableName()).add("Payload", payload()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "tableName":
            return Optional.ofNullable(clazz.cast(tableName()));
        case "payload":
            return Optional.ofNullable(clazz.cast(payload()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DynamoDBv2Action, T> g) {
        return obj -> g.apply((DynamoDBv2Action) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DynamoDBv2Action> {
        /**
         * <p>
         * The name of the DynamoDB table.
         * </p>
         * 
         * @param tableName
         *        The name of the DynamoDB table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableName(String tableName);

        /**
         * Sets the value of the Payload property for this object.
         *
         * @param payload
         *        The new value for the Payload property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder payload(Payload payload);

        /**
         * Sets the value of the Payload property for this object.
         *
         * This is a convenience method that creates an instance of the {@link Payload.Builder} avoiding the need to
         * create one manually via {@link Payload#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Payload.Builder#build()} is called immediately and its result is
         * passed to {@link #payload(Payload)}.
         * 
         * @param payload
         *        a consumer that will call methods on {@link Payload.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #payload(Payload)
         */
        default Builder payload(Consumer<Payload.Builder> payload) {
            return payload(Payload.builder().applyMutation(payload).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String tableName;

        private Payload payload;

        private BuilderImpl() {
        }

        private BuilderImpl(DynamoDBv2Action model) {
            tableName(model.tableName);
            payload(model.payload);
        }

        public final String getTableName() {
            return tableName;
        }

        public final void setTableName(String tableName) {
            this.tableName = tableName;
        }

        @Override
        public final Builder tableName(String tableName) {
            this.tableName = tableName;
            return this;
        }

        public final Payload.Builder getPayload() {
            return payload != null ? payload.toBuilder() : null;
        }

        public final void setPayload(Payload.BuilderImpl payload) {
            this.payload = payload != null ? payload.build() : null;
        }

        @Override
        public final Builder payload(Payload payload) {
            this.payload = payload;
            return this;
        }

        @Override
        public DynamoDBv2Action build() {
            return new DynamoDBv2Action(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
