/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.grafana.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure containing information about an Amazon Managed Grafana workspace in your account.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WorkspaceDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<WorkspaceDescription.Builder, WorkspaceDescription> {
    private static final SdkField<String> ACCOUNT_ACCESS_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accountAccessType").getter(getter(WorkspaceDescription::accountAccessTypeAsString))
            .setter(setter(Builder::accountAccessType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountAccessType").build()).build();

    private static final SdkField<AuthenticationSummary> AUTHENTICATION_FIELD = SdkField
            .<AuthenticationSummary> builder(MarshallingType.SDK_POJO).memberName("authentication")
            .getter(getter(WorkspaceDescription::authentication)).setter(setter(Builder::authentication))
            .constructor(AuthenticationSummary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authentication").build()).build();

    private static final SdkField<Instant> CREATED_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("created").getter(getter(WorkspaceDescription::created)).setter(setter(Builder::created))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("created").build()).build();

    private static final SdkField<List<String>> DATA_SOURCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("dataSources")
            .getter(getter(WorkspaceDescription::dataSourcesAsStrings))
            .setter(setter(Builder::dataSourcesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dataSources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(WorkspaceDescription::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<String> ENDPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("endpoint").getter(getter(WorkspaceDescription::endpoint)).setter(setter(Builder::endpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endpoint").build()).build();

    private static final SdkField<Boolean> FREE_TRIAL_CONSUMED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("freeTrialConsumed").getter(getter(WorkspaceDescription::freeTrialConsumed))
            .setter(setter(Builder::freeTrialConsumed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("freeTrialConsumed").build()).build();

    private static final SdkField<Instant> FREE_TRIAL_EXPIRATION_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("freeTrialExpiration").getter(getter(WorkspaceDescription::freeTrialExpiration))
            .setter(setter(Builder::freeTrialExpiration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("freeTrialExpiration").build())
            .build();

    private static final SdkField<String> GRAFANA_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("grafanaVersion").getter(getter(WorkspaceDescription::grafanaVersion))
            .setter(setter(Builder::grafanaVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("grafanaVersion").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(WorkspaceDescription::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<Instant> LICENSE_EXPIRATION_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("licenseExpiration").getter(getter(WorkspaceDescription::licenseExpiration))
            .setter(setter(Builder::licenseExpiration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("licenseExpiration").build()).build();

    private static final SdkField<String> LICENSE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("licenseType").getter(getter(WorkspaceDescription::licenseTypeAsString))
            .setter(setter(Builder::licenseType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("licenseType").build()).build();

    private static final SdkField<Instant> MODIFIED_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("modified").getter(getter(WorkspaceDescription::modified)).setter(setter(Builder::modified))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modified").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(WorkspaceDescription::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<NetworkAccessConfiguration> NETWORK_ACCESS_CONTROL_FIELD = SdkField
            .<NetworkAccessConfiguration> builder(MarshallingType.SDK_POJO).memberName("networkAccessControl")
            .getter(getter(WorkspaceDescription::networkAccessControl)).setter(setter(Builder::networkAccessControl))
            .constructor(NetworkAccessConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkAccessControl").build())
            .build();

    private static final SdkField<List<String>> NOTIFICATION_DESTINATIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("notificationDestinations")
            .getter(getter(WorkspaceDescription::notificationDestinationsAsStrings))
            .setter(setter(Builder::notificationDestinationsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("notificationDestinations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ORGANIZATION_ROLE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("organizationRoleName").getter(getter(WorkspaceDescription::organizationRoleName))
            .setter(setter(Builder::organizationRoleName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("organizationRoleName").build())
            .build();

    private static final SdkField<List<String>> ORGANIZATIONAL_UNITS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("organizationalUnits")
            .getter(getter(WorkspaceDescription::organizationalUnits))
            .setter(setter(Builder::organizationalUnits))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("organizationalUnits").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> PERMISSION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("permissionType").getter(getter(WorkspaceDescription::permissionTypeAsString))
            .setter(setter(Builder::permissionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("permissionType").build()).build();

    private static final SdkField<String> STACK_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("stackSetName").getter(getter(WorkspaceDescription::stackSetName)).setter(setter(Builder::stackSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stackSetName").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(WorkspaceDescription::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(WorkspaceDescription::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<VpcConfiguration> VPC_CONFIGURATION_FIELD = SdkField
            .<VpcConfiguration> builder(MarshallingType.SDK_POJO).memberName("vpcConfiguration")
            .getter(getter(WorkspaceDescription::vpcConfiguration)).setter(setter(Builder::vpcConfiguration))
            .constructor(VpcConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vpcConfiguration").build()).build();

    private static final SdkField<String> WORKSPACE_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("workspaceRoleArn").getter(getter(WorkspaceDescription::workspaceRoleArn))
            .setter(setter(Builder::workspaceRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("workspaceRoleArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ACCESS_TYPE_FIELD,
            AUTHENTICATION_FIELD, CREATED_FIELD, DATA_SOURCES_FIELD, DESCRIPTION_FIELD, ENDPOINT_FIELD,
            FREE_TRIAL_CONSUMED_FIELD, FREE_TRIAL_EXPIRATION_FIELD, GRAFANA_VERSION_FIELD, ID_FIELD, LICENSE_EXPIRATION_FIELD,
            LICENSE_TYPE_FIELD, MODIFIED_FIELD, NAME_FIELD, NETWORK_ACCESS_CONTROL_FIELD, NOTIFICATION_DESTINATIONS_FIELD,
            ORGANIZATION_ROLE_NAME_FIELD, ORGANIZATIONAL_UNITS_FIELD, PERMISSION_TYPE_FIELD, STACK_SET_NAME_FIELD, STATUS_FIELD,
            TAGS_FIELD, VPC_CONFIGURATION_FIELD, WORKSPACE_ROLE_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String accountAccessType;

    private final AuthenticationSummary authentication;

    private final Instant created;

    private final List<String> dataSources;

    private final String description;

    private final String endpoint;

    private final Boolean freeTrialConsumed;

    private final Instant freeTrialExpiration;

    private final String grafanaVersion;

    private final String id;

    private final Instant licenseExpiration;

    private final String licenseType;

    private final Instant modified;

    private final String name;

    private final NetworkAccessConfiguration networkAccessControl;

    private final List<String> notificationDestinations;

    private final String organizationRoleName;

    private final List<String> organizationalUnits;

    private final String permissionType;

    private final String stackSetName;

    private final String status;

    private final Map<String, String> tags;

    private final VpcConfiguration vpcConfiguration;

    private final String workspaceRoleArn;

    private WorkspaceDescription(BuilderImpl builder) {
        this.accountAccessType = builder.accountAccessType;
        this.authentication = builder.authentication;
        this.created = builder.created;
        this.dataSources = builder.dataSources;
        this.description = builder.description;
        this.endpoint = builder.endpoint;
        this.freeTrialConsumed = builder.freeTrialConsumed;
        this.freeTrialExpiration = builder.freeTrialExpiration;
        this.grafanaVersion = builder.grafanaVersion;
        this.id = builder.id;
        this.licenseExpiration = builder.licenseExpiration;
        this.licenseType = builder.licenseType;
        this.modified = builder.modified;
        this.name = builder.name;
        this.networkAccessControl = builder.networkAccessControl;
        this.notificationDestinations = builder.notificationDestinations;
        this.organizationRoleName = builder.organizationRoleName;
        this.organizationalUnits = builder.organizationalUnits;
        this.permissionType = builder.permissionType;
        this.stackSetName = builder.stackSetName;
        this.status = builder.status;
        this.tags = builder.tags;
        this.vpcConfiguration = builder.vpcConfiguration;
        this.workspaceRoleArn = builder.workspaceRoleArn;
    }

    /**
     * <p>
     * Specifies whether the workspace can access Amazon Web Services resources in this Amazon Web Services account
     * only, or whether it can also access Amazon Web Services resources in other accounts in the same organization. If
     * this is <code>ORGANIZATION</code>, the <code>workspaceOrganizationalUnits</code> parameter specifies which
     * organizational units the workspace can access.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accountAccessType}
     * will return {@link AccountAccessType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #accountAccessTypeAsString}.
     * </p>
     * 
     * @return Specifies whether the workspace can access Amazon Web Services resources in this Amazon Web Services
     *         account only, or whether it can also access Amazon Web Services resources in other accounts in the same
     *         organization. If this is <code>ORGANIZATION</code>, the <code>workspaceOrganizationalUnits</code>
     *         parameter specifies which organizational units the workspace can access.
     * @see AccountAccessType
     */
    public final AccountAccessType accountAccessType() {
        return AccountAccessType.fromValue(accountAccessType);
    }

    /**
     * <p>
     * Specifies whether the workspace can access Amazon Web Services resources in this Amazon Web Services account
     * only, or whether it can also access Amazon Web Services resources in other accounts in the same organization. If
     * this is <code>ORGANIZATION</code>, the <code>workspaceOrganizationalUnits</code> parameter specifies which
     * organizational units the workspace can access.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accountAccessType}
     * will return {@link AccountAccessType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #accountAccessTypeAsString}.
     * </p>
     * 
     * @return Specifies whether the workspace can access Amazon Web Services resources in this Amazon Web Services
     *         account only, or whether it can also access Amazon Web Services resources in other accounts in the same
     *         organization. If this is <code>ORGANIZATION</code>, the <code>workspaceOrganizationalUnits</code>
     *         parameter specifies which organizational units the workspace can access.
     * @see AccountAccessType
     */
    public final String accountAccessTypeAsString() {
        return accountAccessType;
    }

    /**
     * <p>
     * A structure that describes whether the workspace uses SAML, IAM Identity Center, or both methods for user
     * authentication.
     * </p>
     * 
     * @return A structure that describes whether the workspace uses SAML, IAM Identity Center, or both methods for user
     *         authentication.
     */
    public final AuthenticationSummary authentication() {
        return authentication;
    }

    /**
     * <p>
     * The date that the workspace was created.
     * </p>
     * 
     * @return The date that the workspace was created.
     */
    public final Instant created() {
        return created;
    }

    /**
     * <p>
     * Specifies the Amazon Web Services data sources that have been configured to have IAM roles and permissions
     * created to allow Amazon Managed Grafana to read data from these sources.
     * </p>
     * <p>
     * This list is only used when the workspace was created through the Amazon Web Services console, and the
     * <code>permissionType</code> is <code>SERVICE_MANAGED</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDataSources} method.
     * </p>
     * 
     * @return Specifies the Amazon Web Services data sources that have been configured to have IAM roles and
     *         permissions created to allow Amazon Managed Grafana to read data from these sources.</p>
     *         <p>
     *         This list is only used when the workspace was created through the Amazon Web Services console, and the
     *         <code>permissionType</code> is <code>SERVICE_MANAGED</code>.
     */
    public final List<DataSourceType> dataSources() {
        return DataSourceTypesListCopier.copyStringToEnum(dataSources);
    }

    /**
     * For responses, this returns true if the service returned a value for the DataSources property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDataSources() {
        return dataSources != null && !(dataSources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the Amazon Web Services data sources that have been configured to have IAM roles and permissions
     * created to allow Amazon Managed Grafana to read data from these sources.
     * </p>
     * <p>
     * This list is only used when the workspace was created through the Amazon Web Services console, and the
     * <code>permissionType</code> is <code>SERVICE_MANAGED</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDataSources} method.
     * </p>
     * 
     * @return Specifies the Amazon Web Services data sources that have been configured to have IAM roles and
     *         permissions created to allow Amazon Managed Grafana to read data from these sources.</p>
     *         <p>
     *         This list is only used when the workspace was created through the Amazon Web Services console, and the
     *         <code>permissionType</code> is <code>SERVICE_MANAGED</code>.
     */
    public final List<String> dataSourcesAsStrings() {
        return dataSources;
    }

    /**
     * <p>
     * The user-defined description of the workspace.
     * </p>
     * 
     * @return The user-defined description of the workspace.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The URL that users can use to access the Grafana console in the workspace.
     * </p>
     * 
     * @return The URL that users can use to access the Grafana console in the workspace.
     */
    public final String endpoint() {
        return endpoint;
    }

    /**
     * <p>
     * Specifies whether this workspace has already fully used its free trial for Grafana Enterprise.
     * </p>
     * 
     * @return Specifies whether this workspace has already fully used its free trial for Grafana Enterprise.
     */
    public final Boolean freeTrialConsumed() {
        return freeTrialConsumed;
    }

    /**
     * <p>
     * If this workspace is currently in the free trial period for Grafana Enterprise, this value specifies when that
     * free trial ends.
     * </p>
     * 
     * @return If this workspace is currently in the free trial period for Grafana Enterprise, this value specifies when
     *         that free trial ends.
     */
    public final Instant freeTrialExpiration() {
        return freeTrialExpiration;
    }

    /**
     * <p>
     * The version of Grafana supported in this workspace.
     * </p>
     * 
     * @return The version of Grafana supported in this workspace.
     */
    public final String grafanaVersion() {
        return grafanaVersion;
    }

    /**
     * <p>
     * The unique ID of this workspace.
     * </p>
     * 
     * @return The unique ID of this workspace.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * If this workspace has a full Grafana Enterprise license, this specifies when the license ends and will need to be
     * renewed.
     * </p>
     * 
     * @return If this workspace has a full Grafana Enterprise license, this specifies when the license ends and will
     *         need to be renewed.
     */
    public final Instant licenseExpiration() {
        return licenseExpiration;
    }

    /**
     * <p>
     * Specifies whether this workspace has a full Grafana Enterprise license or a free trial license.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #licenseType} will
     * return {@link LicenseType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #licenseTypeAsString}.
     * </p>
     * 
     * @return Specifies whether this workspace has a full Grafana Enterprise license or a free trial license.
     * @see LicenseType
     */
    public final LicenseType licenseType() {
        return LicenseType.fromValue(licenseType);
    }

    /**
     * <p>
     * Specifies whether this workspace has a full Grafana Enterprise license or a free trial license.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #licenseType} will
     * return {@link LicenseType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #licenseTypeAsString}.
     * </p>
     * 
     * @return Specifies whether this workspace has a full Grafana Enterprise license or a free trial license.
     * @see LicenseType
     */
    public final String licenseTypeAsString() {
        return licenseType;
    }

    /**
     * <p>
     * The most recent date that the workspace was modified.
     * </p>
     * 
     * @return The most recent date that the workspace was modified.
     */
    public final Instant modified() {
        return modified;
    }

    /**
     * <p>
     * The name of the workspace.
     * </p>
     * 
     * @return The name of the workspace.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The configuration settings for network access to your workspace.
     * </p>
     * 
     * @return The configuration settings for network access to your workspace.
     */
    public final NetworkAccessConfiguration networkAccessControl() {
        return networkAccessControl;
    }

    /**
     * <p>
     * The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM roles and
     * permissions for, to allow Amazon Managed Grafana to use these channels.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNotificationDestinations} method.
     * </p>
     * 
     * @return The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM
     *         roles and permissions for, to allow Amazon Managed Grafana to use these channels.
     */
    public final List<NotificationDestinationType> notificationDestinations() {
        return NotificationDestinationsListCopier.copyStringToEnum(notificationDestinations);
    }

    /**
     * For responses, this returns true if the service returned a value for the NotificationDestinations property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasNotificationDestinations() {
        return notificationDestinations != null && !(notificationDestinations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM roles and
     * permissions for, to allow Amazon Managed Grafana to use these channels.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNotificationDestinations} method.
     * </p>
     * 
     * @return The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM
     *         roles and permissions for, to allow Amazon Managed Grafana to use these channels.
     */
    public final List<String> notificationDestinationsAsStrings() {
        return notificationDestinations;
    }

    /**
     * <p>
     * The name of the IAM role that is used to access resources through Organizations.
     * </p>
     * 
     * @return The name of the IAM role that is used to access resources through Organizations.
     */
    public final String organizationRoleName() {
        return organizationRoleName;
    }

    /**
     * For responses, this returns true if the service returned a value for the OrganizationalUnits property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasOrganizationalUnits() {
        return organizationalUnits != null && !(organizationalUnits instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the organizational units that this workspace is allowed to use data sources from, if this workspace is
     * in an account that is part of an organization.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOrganizationalUnits} method.
     * </p>
     * 
     * @return Specifies the organizational units that this workspace is allowed to use data sources from, if this
     *         workspace is in an account that is part of an organization.
     */
    public final List<String> organizationalUnits() {
        return organizationalUnits;
    }

    /**
     * <p>
     * If this is <code>SERVICE_MANAGED</code>, and the workplace was created through the Amazon Managed Grafana
     * console, then Amazon Managed Grafana automatically creates the IAM roles and provisions the permissions that the
     * workspace needs to use Amazon Web Services data sources and notification channels.
     * </p>
     * <p>
     * If this is <code>CUSTOMER_MANAGED</code>, you must manage those roles and permissions yourself.
     * </p>
     * <p>
     * If you are working with a workspace in a member account of an organization and that account is not a delegated
     * administrator account, and you want the workspace to access data sources in other Amazon Web Services accounts in
     * the organization, this parameter must be set to <code>CUSTOMER_MANAGED</code>.
     * </p>
     * <p>
     * For more information about converting between customer and service managed, see <a
     * href="https://docs.aws.amazon.com/grafana/latest/userguide/AMG-datasource-and-notification.html">Managing
     * permissions for data sources and notification channels</a>. For more information about the roles and permissions
     * that must be managed for customer managed workspaces, see <a
     * href="https://docs.aws.amazon.com/grafana/latest/userguide/AMG-manage-permissions.html">Amazon Managed Grafana
     * permissions and policies for Amazon Web Services data sources and notification channels</a>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permissionType}
     * will return {@link PermissionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #permissionTypeAsString}.
     * </p>
     * 
     * @return If this is <code>SERVICE_MANAGED</code>, and the workplace was created through the Amazon Managed Grafana
     *         console, then Amazon Managed Grafana automatically creates the IAM roles and provisions the permissions
     *         that the workspace needs to use Amazon Web Services data sources and notification channels.</p>
     *         <p>
     *         If this is <code>CUSTOMER_MANAGED</code>, you must manage those roles and permissions yourself.
     *         </p>
     *         <p>
     *         If you are working with a workspace in a member account of an organization and that account is not a
     *         delegated administrator account, and you want the workspace to access data sources in other Amazon Web
     *         Services accounts in the organization, this parameter must be set to <code>CUSTOMER_MANAGED</code>.
     *         </p>
     *         <p>
     *         For more information about converting between customer and service managed, see <a
     *         href="https://docs.aws.amazon.com/grafana/latest/userguide/AMG-datasource-and-notification.html">Managing
     *         permissions for data sources and notification channels</a>. For more information about the roles and
     *         permissions that must be managed for customer managed workspaces, see <a
     *         href="https://docs.aws.amazon.com/grafana/latest/userguide/AMG-manage-permissions.html">Amazon Managed
     *         Grafana permissions and policies for Amazon Web Services data sources and notification channels</a>
     * @see PermissionType
     */
    public final PermissionType permissionType() {
        return PermissionType.fromValue(permissionType);
    }

    /**
     * <p>
     * If this is <code>SERVICE_MANAGED</code>, and the workplace was created through the Amazon Managed Grafana
     * console, then Amazon Managed Grafana automatically creates the IAM roles and provisions the permissions that the
     * workspace needs to use Amazon Web Services data sources and notification channels.
     * </p>
     * <p>
     * If this is <code>CUSTOMER_MANAGED</code>, you must manage those roles and permissions yourself.
     * </p>
     * <p>
     * If you are working with a workspace in a member account of an organization and that account is not a delegated
     * administrator account, and you want the workspace to access data sources in other Amazon Web Services accounts in
     * the organization, this parameter must be set to <code>CUSTOMER_MANAGED</code>.
     * </p>
     * <p>
     * For more information about converting between customer and service managed, see <a
     * href="https://docs.aws.amazon.com/grafana/latest/userguide/AMG-datasource-and-notification.html">Managing
     * permissions for data sources and notification channels</a>. For more information about the roles and permissions
     * that must be managed for customer managed workspaces, see <a
     * href="https://docs.aws.amazon.com/grafana/latest/userguide/AMG-manage-permissions.html">Amazon Managed Grafana
     * permissions and policies for Amazon Web Services data sources and notification channels</a>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permissionType}
     * will return {@link PermissionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #permissionTypeAsString}.
     * </p>
     * 
     * @return If this is <code>SERVICE_MANAGED</code>, and the workplace was created through the Amazon Managed Grafana
     *         console, then Amazon Managed Grafana automatically creates the IAM roles and provisions the permissions
     *         that the workspace needs to use Amazon Web Services data sources and notification channels.</p>
     *         <p>
     *         If this is <code>CUSTOMER_MANAGED</code>, you must manage those roles and permissions yourself.
     *         </p>
     *         <p>
     *         If you are working with a workspace in a member account of an organization and that account is not a
     *         delegated administrator account, and you want the workspace to access data sources in other Amazon Web
     *         Services accounts in the organization, this parameter must be set to <code>CUSTOMER_MANAGED</code>.
     *         </p>
     *         <p>
     *         For more information about converting between customer and service managed, see <a
     *         href="https://docs.aws.amazon.com/grafana/latest/userguide/AMG-datasource-and-notification.html">Managing
     *         permissions for data sources and notification channels</a>. For more information about the roles and
     *         permissions that must be managed for customer managed workspaces, see <a
     *         href="https://docs.aws.amazon.com/grafana/latest/userguide/AMG-manage-permissions.html">Amazon Managed
     *         Grafana permissions and policies for Amazon Web Services data sources and notification channels</a>
     * @see PermissionType
     */
    public final String permissionTypeAsString() {
        return permissionType;
    }

    /**
     * <p>
     * The name of the CloudFormation stack set that is used to generate IAM roles to be used for this workspace.
     * </p>
     * 
     * @return The name of the CloudFormation stack set that is used to generate IAM roles to be used for this
     *         workspace.
     */
    public final String stackSetName() {
        return stackSetName;
    }

    /**
     * <p>
     * The current status of the workspace.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link WorkspaceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the workspace.
     * @see WorkspaceStatus
     */
    public final WorkspaceStatus status() {
        return WorkspaceStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the workspace.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link WorkspaceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the workspace.
     * @see WorkspaceStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The list of tags associated with the workspace.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The list of tags associated with the workspace.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The configuration for connecting to data sources in a private VPC (Amazon Virtual Private Cloud).
     * </p>
     * 
     * @return The configuration for connecting to data sources in a private VPC (Amazon Virtual Private Cloud).
     */
    public final VpcConfiguration vpcConfiguration() {
        return vpcConfiguration;
    }

    /**
     * <p>
     * The IAM role that grants permissions to the Amazon Web Services resources that the workspace will view data from.
     * This role must already exist.
     * </p>
     * 
     * @return The IAM role that grants permissions to the Amazon Web Services resources that the workspace will view
     *         data from. This role must already exist.
     */
    public final String workspaceRoleArn() {
        return workspaceRoleArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accountAccessTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(authentication());
        hashCode = 31 * hashCode + Objects.hashCode(created());
        hashCode = 31 * hashCode + Objects.hashCode(hasDataSources() ? dataSourcesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(endpoint());
        hashCode = 31 * hashCode + Objects.hashCode(freeTrialConsumed());
        hashCode = 31 * hashCode + Objects.hashCode(freeTrialExpiration());
        hashCode = 31 * hashCode + Objects.hashCode(grafanaVersion());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(licenseExpiration());
        hashCode = 31 * hashCode + Objects.hashCode(licenseTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(modified());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(networkAccessControl());
        hashCode = 31 * hashCode + Objects.hashCode(hasNotificationDestinations() ? notificationDestinationsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(organizationRoleName());
        hashCode = 31 * hashCode + Objects.hashCode(hasOrganizationalUnits() ? organizationalUnits() : null);
        hashCode = 31 * hashCode + Objects.hashCode(permissionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stackSetName());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(vpcConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(workspaceRoleArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WorkspaceDescription)) {
            return false;
        }
        WorkspaceDescription other = (WorkspaceDescription) obj;
        return Objects.equals(accountAccessTypeAsString(), other.accountAccessTypeAsString())
                && Objects.equals(authentication(), other.authentication()) && Objects.equals(created(), other.created())
                && hasDataSources() == other.hasDataSources()
                && Objects.equals(dataSourcesAsStrings(), other.dataSourcesAsStrings())
                && Objects.equals(description(), other.description()) && Objects.equals(endpoint(), other.endpoint())
                && Objects.equals(freeTrialConsumed(), other.freeTrialConsumed())
                && Objects.equals(freeTrialExpiration(), other.freeTrialExpiration())
                && Objects.equals(grafanaVersion(), other.grafanaVersion()) && Objects.equals(id(), other.id())
                && Objects.equals(licenseExpiration(), other.licenseExpiration())
                && Objects.equals(licenseTypeAsString(), other.licenseTypeAsString())
                && Objects.equals(modified(), other.modified()) && Objects.equals(name(), other.name())
                && Objects.equals(networkAccessControl(), other.networkAccessControl())
                && hasNotificationDestinations() == other.hasNotificationDestinations()
                && Objects.equals(notificationDestinationsAsStrings(), other.notificationDestinationsAsStrings())
                && Objects.equals(organizationRoleName(), other.organizationRoleName())
                && hasOrganizationalUnits() == other.hasOrganizationalUnits()
                && Objects.equals(organizationalUnits(), other.organizationalUnits())
                && Objects.equals(permissionTypeAsString(), other.permissionTypeAsString())
                && Objects.equals(stackSetName(), other.stackSetName())
                && Objects.equals(statusAsString(), other.statusAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(vpcConfiguration(), other.vpcConfiguration())
                && Objects.equals(workspaceRoleArn(), other.workspaceRoleArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("WorkspaceDescription").add("AccountAccessType", accountAccessTypeAsString())
                .add("Authentication", authentication()).add("Created", created())
                .add("DataSources", hasDataSources() ? dataSourcesAsStrings() : null)
                .add("Description", description() == null ? null : "*** Sensitive Data Redacted ***").add("Endpoint", endpoint())
                .add("FreeTrialConsumed", freeTrialConsumed()).add("FreeTrialExpiration", freeTrialExpiration())
                .add("GrafanaVersion", grafanaVersion()).add("Id", id()).add("LicenseExpiration", licenseExpiration())
                .add("LicenseType", licenseTypeAsString()).add("Modified", modified())
                .add("Name", name() == null ? null : "*** Sensitive Data Redacted ***")
                .add("NetworkAccessControl", networkAccessControl())
                .add("NotificationDestinations", hasNotificationDestinations() ? notificationDestinationsAsStrings() : null)
                .add("OrganizationRoleName", organizationRoleName() == null ? null : "*** Sensitive Data Redacted ***")
                .add("OrganizationalUnits", organizationalUnits() == null ? null : "*** Sensitive Data Redacted ***")
                .add("PermissionType", permissionTypeAsString()).add("StackSetName", stackSetName())
                .add("Status", statusAsString()).add("Tags", hasTags() ? tags() : null)
                .add("VpcConfiguration", vpcConfiguration())
                .add("WorkspaceRoleArn", workspaceRoleArn() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "accountAccessType":
            return Optional.ofNullable(clazz.cast(accountAccessTypeAsString()));
        case "authentication":
            return Optional.ofNullable(clazz.cast(authentication()));
        case "created":
            return Optional.ofNullable(clazz.cast(created()));
        case "dataSources":
            return Optional.ofNullable(clazz.cast(dataSourcesAsStrings()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "endpoint":
            return Optional.ofNullable(clazz.cast(endpoint()));
        case "freeTrialConsumed":
            return Optional.ofNullable(clazz.cast(freeTrialConsumed()));
        case "freeTrialExpiration":
            return Optional.ofNullable(clazz.cast(freeTrialExpiration()));
        case "grafanaVersion":
            return Optional.ofNullable(clazz.cast(grafanaVersion()));
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "licenseExpiration":
            return Optional.ofNullable(clazz.cast(licenseExpiration()));
        case "licenseType":
            return Optional.ofNullable(clazz.cast(licenseTypeAsString()));
        case "modified":
            return Optional.ofNullable(clazz.cast(modified()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "networkAccessControl":
            return Optional.ofNullable(clazz.cast(networkAccessControl()));
        case "notificationDestinations":
            return Optional.ofNullable(clazz.cast(notificationDestinationsAsStrings()));
        case "organizationRoleName":
            return Optional.ofNullable(clazz.cast(organizationRoleName()));
        case "organizationalUnits":
            return Optional.ofNullable(clazz.cast(organizationalUnits()));
        case "permissionType":
            return Optional.ofNullable(clazz.cast(permissionTypeAsString()));
        case "stackSetName":
            return Optional.ofNullable(clazz.cast(stackSetName()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "vpcConfiguration":
            return Optional.ofNullable(clazz.cast(vpcConfiguration()));
        case "workspaceRoleArn":
            return Optional.ofNullable(clazz.cast(workspaceRoleArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WorkspaceDescription, T> g) {
        return obj -> g.apply((WorkspaceDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WorkspaceDescription> {
        /**
         * <p>
         * Specifies whether the workspace can access Amazon Web Services resources in this Amazon Web Services account
         * only, or whether it can also access Amazon Web Services resources in other accounts in the same organization.
         * If this is <code>ORGANIZATION</code>, the <code>workspaceOrganizationalUnits</code> parameter specifies which
         * organizational units the workspace can access.
         * </p>
         * 
         * @param accountAccessType
         *        Specifies whether the workspace can access Amazon Web Services resources in this Amazon Web Services
         *        account only, or whether it can also access Amazon Web Services resources in other accounts in the
         *        same organization. If this is <code>ORGANIZATION</code>, the <code>workspaceOrganizationalUnits</code>
         *        parameter specifies which organizational units the workspace can access.
         * @see AccountAccessType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountAccessType
         */
        Builder accountAccessType(String accountAccessType);

        /**
         * <p>
         * Specifies whether the workspace can access Amazon Web Services resources in this Amazon Web Services account
         * only, or whether it can also access Amazon Web Services resources in other accounts in the same organization.
         * If this is <code>ORGANIZATION</code>, the <code>workspaceOrganizationalUnits</code> parameter specifies which
         * organizational units the workspace can access.
         * </p>
         * 
         * @param accountAccessType
         *        Specifies whether the workspace can access Amazon Web Services resources in this Amazon Web Services
         *        account only, or whether it can also access Amazon Web Services resources in other accounts in the
         *        same organization. If this is <code>ORGANIZATION</code>, the <code>workspaceOrganizationalUnits</code>
         *        parameter specifies which organizational units the workspace can access.
         * @see AccountAccessType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountAccessType
         */
        Builder accountAccessType(AccountAccessType accountAccessType);

        /**
         * <p>
         * A structure that describes whether the workspace uses SAML, IAM Identity Center, or both methods for user
         * authentication.
         * </p>
         * 
         * @param authentication
         *        A structure that describes whether the workspace uses SAML, IAM Identity Center, or both methods for
         *        user authentication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authentication(AuthenticationSummary authentication);

        /**
         * <p>
         * A structure that describes whether the workspace uses SAML, IAM Identity Center, or both methods for user
         * authentication.
         * </p>
         * This is a convenience method that creates an instance of the {@link AuthenticationSummary.Builder} avoiding
         * the need to create one manually via {@link AuthenticationSummary#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AuthenticationSummary.Builder#build()} is called immediately and
         * its result is passed to {@link #authentication(AuthenticationSummary)}.
         * 
         * @param authentication
         *        a consumer that will call methods on {@link AuthenticationSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #authentication(AuthenticationSummary)
         */
        default Builder authentication(Consumer<AuthenticationSummary.Builder> authentication) {
            return authentication(AuthenticationSummary.builder().applyMutation(authentication).build());
        }

        /**
         * <p>
         * The date that the workspace was created.
         * </p>
         * 
         * @param created
         *        The date that the workspace was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder created(Instant created);

        /**
         * <p>
         * Specifies the Amazon Web Services data sources that have been configured to have IAM roles and permissions
         * created to allow Amazon Managed Grafana to read data from these sources.
         * </p>
         * <p>
         * This list is only used when the workspace was created through the Amazon Web Services console, and the
         * <code>permissionType</code> is <code>SERVICE_MANAGED</code>.
         * </p>
         * 
         * @param dataSources
         *        Specifies the Amazon Web Services data sources that have been configured to have IAM roles and
         *        permissions created to allow Amazon Managed Grafana to read data from these sources.</p>
         *        <p>
         *        This list is only used when the workspace was created through the Amazon Web Services console, and the
         *        <code>permissionType</code> is <code>SERVICE_MANAGED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSourcesWithStrings(Collection<String> dataSources);

        /**
         * <p>
         * Specifies the Amazon Web Services data sources that have been configured to have IAM roles and permissions
         * created to allow Amazon Managed Grafana to read data from these sources.
         * </p>
         * <p>
         * This list is only used when the workspace was created through the Amazon Web Services console, and the
         * <code>permissionType</code> is <code>SERVICE_MANAGED</code>.
         * </p>
         * 
         * @param dataSources
         *        Specifies the Amazon Web Services data sources that have been configured to have IAM roles and
         *        permissions created to allow Amazon Managed Grafana to read data from these sources.</p>
         *        <p>
         *        This list is only used when the workspace was created through the Amazon Web Services console, and the
         *        <code>permissionType</code> is <code>SERVICE_MANAGED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSourcesWithStrings(String... dataSources);

        /**
         * <p>
         * Specifies the Amazon Web Services data sources that have been configured to have IAM roles and permissions
         * created to allow Amazon Managed Grafana to read data from these sources.
         * </p>
         * <p>
         * This list is only used when the workspace was created through the Amazon Web Services console, and the
         * <code>permissionType</code> is <code>SERVICE_MANAGED</code>.
         * </p>
         * 
         * @param dataSources
         *        Specifies the Amazon Web Services data sources that have been configured to have IAM roles and
         *        permissions created to allow Amazon Managed Grafana to read data from these sources.</p>
         *        <p>
         *        This list is only used when the workspace was created through the Amazon Web Services console, and the
         *        <code>permissionType</code> is <code>SERVICE_MANAGED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSources(Collection<DataSourceType> dataSources);

        /**
         * <p>
         * Specifies the Amazon Web Services data sources that have been configured to have IAM roles and permissions
         * created to allow Amazon Managed Grafana to read data from these sources.
         * </p>
         * <p>
         * This list is only used when the workspace was created through the Amazon Web Services console, and the
         * <code>permissionType</code> is <code>SERVICE_MANAGED</code>.
         * </p>
         * 
         * @param dataSources
         *        Specifies the Amazon Web Services data sources that have been configured to have IAM roles and
         *        permissions created to allow Amazon Managed Grafana to read data from these sources.</p>
         *        <p>
         *        This list is only used when the workspace was created through the Amazon Web Services console, and the
         *        <code>permissionType</code> is <code>SERVICE_MANAGED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSources(DataSourceType... dataSources);

        /**
         * <p>
         * The user-defined description of the workspace.
         * </p>
         * 
         * @param description
         *        The user-defined description of the workspace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The URL that users can use to access the Grafana console in the workspace.
         * </p>
         * 
         * @param endpoint
         *        The URL that users can use to access the Grafana console in the workspace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoint(String endpoint);

        /**
         * <p>
         * Specifies whether this workspace has already fully used its free trial for Grafana Enterprise.
         * </p>
         * 
         * @param freeTrialConsumed
         *        Specifies whether this workspace has already fully used its free trial for Grafana Enterprise.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder freeTrialConsumed(Boolean freeTrialConsumed);

        /**
         * <p>
         * If this workspace is currently in the free trial period for Grafana Enterprise, this value specifies when
         * that free trial ends.
         * </p>
         * 
         * @param freeTrialExpiration
         *        If this workspace is currently in the free trial period for Grafana Enterprise, this value specifies
         *        when that free trial ends.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder freeTrialExpiration(Instant freeTrialExpiration);

        /**
         * <p>
         * The version of Grafana supported in this workspace.
         * </p>
         * 
         * @param grafanaVersion
         *        The version of Grafana supported in this workspace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grafanaVersion(String grafanaVersion);

        /**
         * <p>
         * The unique ID of this workspace.
         * </p>
         * 
         * @param id
         *        The unique ID of this workspace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * If this workspace has a full Grafana Enterprise license, this specifies when the license ends and will need
         * to be renewed.
         * </p>
         * 
         * @param licenseExpiration
         *        If this workspace has a full Grafana Enterprise license, this specifies when the license ends and will
         *        need to be renewed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseExpiration(Instant licenseExpiration);

        /**
         * <p>
         * Specifies whether this workspace has a full Grafana Enterprise license or a free trial license.
         * </p>
         * 
         * @param licenseType
         *        Specifies whether this workspace has a full Grafana Enterprise license or a free trial license.
         * @see LicenseType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LicenseType
         */
        Builder licenseType(String licenseType);

        /**
         * <p>
         * Specifies whether this workspace has a full Grafana Enterprise license or a free trial license.
         * </p>
         * 
         * @param licenseType
         *        Specifies whether this workspace has a full Grafana Enterprise license or a free trial license.
         * @see LicenseType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LicenseType
         */
        Builder licenseType(LicenseType licenseType);

        /**
         * <p>
         * The most recent date that the workspace was modified.
         * </p>
         * 
         * @param modified
         *        The most recent date that the workspace was modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modified(Instant modified);

        /**
         * <p>
         * The name of the workspace.
         * </p>
         * 
         * @param name
         *        The name of the workspace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The configuration settings for network access to your workspace.
         * </p>
         * 
         * @param networkAccessControl
         *        The configuration settings for network access to your workspace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkAccessControl(NetworkAccessConfiguration networkAccessControl);

        /**
         * <p>
         * The configuration settings for network access to your workspace.
         * </p>
         * This is a convenience method that creates an instance of the {@link NetworkAccessConfiguration.Builder}
         * avoiding the need to create one manually via {@link NetworkAccessConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NetworkAccessConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #networkAccessControl(NetworkAccessConfiguration)}.
         * 
         * @param networkAccessControl
         *        a consumer that will call methods on {@link NetworkAccessConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkAccessControl(NetworkAccessConfiguration)
         */
        default Builder networkAccessControl(Consumer<NetworkAccessConfiguration.Builder> networkAccessControl) {
            return networkAccessControl(NetworkAccessConfiguration.builder().applyMutation(networkAccessControl).build());
        }

        /**
         * <p>
         * The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM roles
         * and permissions for, to allow Amazon Managed Grafana to use these channels.
         * </p>
         * 
         * @param notificationDestinations
         *        The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM
         *        roles and permissions for, to allow Amazon Managed Grafana to use these channels.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationDestinationsWithStrings(Collection<String> notificationDestinations);

        /**
         * <p>
         * The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM roles
         * and permissions for, to allow Amazon Managed Grafana to use these channels.
         * </p>
         * 
         * @param notificationDestinations
         *        The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM
         *        roles and permissions for, to allow Amazon Managed Grafana to use these channels.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationDestinationsWithStrings(String... notificationDestinations);

        /**
         * <p>
         * The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM roles
         * and permissions for, to allow Amazon Managed Grafana to use these channels.
         * </p>
         * 
         * @param notificationDestinations
         *        The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM
         *        roles and permissions for, to allow Amazon Managed Grafana to use these channels.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationDestinations(Collection<NotificationDestinationType> notificationDestinations);

        /**
         * <p>
         * The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM roles
         * and permissions for, to allow Amazon Managed Grafana to use these channels.
         * </p>
         * 
         * @param notificationDestinations
         *        The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM
         *        roles and permissions for, to allow Amazon Managed Grafana to use these channels.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationDestinations(NotificationDestinationType... notificationDestinations);

        /**
         * <p>
         * The name of the IAM role that is used to access resources through Organizations.
         * </p>
         * 
         * @param organizationRoleName
         *        The name of the IAM role that is used to access resources through Organizations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationRoleName(String organizationRoleName);

        /**
         * <p>
         * Specifies the organizational units that this workspace is allowed to use data sources from, if this workspace
         * is in an account that is part of an organization.
         * </p>
         * 
         * @param organizationalUnits
         *        Specifies the organizational units that this workspace is allowed to use data sources from, if this
         *        workspace is in an account that is part of an organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationalUnits(Collection<String> organizationalUnits);

        /**
         * <p>
         * Specifies the organizational units that this workspace is allowed to use data sources from, if this workspace
         * is in an account that is part of an organization.
         * </p>
         * 
         * @param organizationalUnits
         *        Specifies the organizational units that this workspace is allowed to use data sources from, if this
         *        workspace is in an account that is part of an organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationalUnits(String... organizationalUnits);

        /**
         * <p>
         * If this is <code>SERVICE_MANAGED</code>, and the workplace was created through the Amazon Managed Grafana
         * console, then Amazon Managed Grafana automatically creates the IAM roles and provisions the permissions that
         * the workspace needs to use Amazon Web Services data sources and notification channels.
         * </p>
         * <p>
         * If this is <code>CUSTOMER_MANAGED</code>, you must manage those roles and permissions yourself.
         * </p>
         * <p>
         * If you are working with a workspace in a member account of an organization and that account is not a
         * delegated administrator account, and you want the workspace to access data sources in other Amazon Web
         * Services accounts in the organization, this parameter must be set to <code>CUSTOMER_MANAGED</code>.
         * </p>
         * <p>
         * For more information about converting between customer and service managed, see <a
         * href="https://docs.aws.amazon.com/grafana/latest/userguide/AMG-datasource-and-notification.html">Managing
         * permissions for data sources and notification channels</a>. For more information about the roles and
         * permissions that must be managed for customer managed workspaces, see <a
         * href="https://docs.aws.amazon.com/grafana/latest/userguide/AMG-manage-permissions.html">Amazon Managed
         * Grafana permissions and policies for Amazon Web Services data sources and notification channels</a>
         * </p>
         * 
         * @param permissionType
         *        If this is <code>SERVICE_MANAGED</code>, and the workplace was created through the Amazon Managed
         *        Grafana console, then Amazon Managed Grafana automatically creates the IAM roles and provisions the
         *        permissions that the workspace needs to use Amazon Web Services data sources and notification
         *        channels.</p>
         *        <p>
         *        If this is <code>CUSTOMER_MANAGED</code>, you must manage those roles and permissions yourself.
         *        </p>
         *        <p>
         *        If you are working with a workspace in a member account of an organization and that account is not a
         *        delegated administrator account, and you want the workspace to access data sources in other Amazon Web
         *        Services accounts in the organization, this parameter must be set to <code>CUSTOMER_MANAGED</code>.
         *        </p>
         *        <p>
         *        For more information about converting between customer and service managed, see <a
         *        href="https://docs.aws.amazon.com/grafana/latest/userguide/AMG-datasource-and-notification.html"
         *        >Managing permissions for data sources and notification channels</a>. For more information about the
         *        roles and permissions that must be managed for customer managed workspaces, see <a
         *        href="https://docs.aws.amazon.com/grafana/latest/userguide/AMG-manage-permissions.html">Amazon Managed
         *        Grafana permissions and policies for Amazon Web Services data sources and notification channels</a>
         * @see PermissionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PermissionType
         */
        Builder permissionType(String permissionType);

        /**
         * <p>
         * If this is <code>SERVICE_MANAGED</code>, and the workplace was created through the Amazon Managed Grafana
         * console, then Amazon Managed Grafana automatically creates the IAM roles and provisions the permissions that
         * the workspace needs to use Amazon Web Services data sources and notification channels.
         * </p>
         * <p>
         * If this is <code>CUSTOMER_MANAGED</code>, you must manage those roles and permissions yourself.
         * </p>
         * <p>
         * If you are working with a workspace in a member account of an organization and that account is not a
         * delegated administrator account, and you want the workspace to access data sources in other Amazon Web
         * Services accounts in the organization, this parameter must be set to <code>CUSTOMER_MANAGED</code>.
         * </p>
         * <p>
         * For more information about converting between customer and service managed, see <a
         * href="https://docs.aws.amazon.com/grafana/latest/userguide/AMG-datasource-and-notification.html">Managing
         * permissions for data sources and notification channels</a>. For more information about the roles and
         * permissions that must be managed for customer managed workspaces, see <a
         * href="https://docs.aws.amazon.com/grafana/latest/userguide/AMG-manage-permissions.html">Amazon Managed
         * Grafana permissions and policies for Amazon Web Services data sources and notification channels</a>
         * </p>
         * 
         * @param permissionType
         *        If this is <code>SERVICE_MANAGED</code>, and the workplace was created through the Amazon Managed
         *        Grafana console, then Amazon Managed Grafana automatically creates the IAM roles and provisions the
         *        permissions that the workspace needs to use Amazon Web Services data sources and notification
         *        channels.</p>
         *        <p>
         *        If this is <code>CUSTOMER_MANAGED</code>, you must manage those roles and permissions yourself.
         *        </p>
         *        <p>
         *        If you are working with a workspace in a member account of an organization and that account is not a
         *        delegated administrator account, and you want the workspace to access data sources in other Amazon Web
         *        Services accounts in the organization, this parameter must be set to <code>CUSTOMER_MANAGED</code>.
         *        </p>
         *        <p>
         *        For more information about converting between customer and service managed, see <a
         *        href="https://docs.aws.amazon.com/grafana/latest/userguide/AMG-datasource-and-notification.html"
         *        >Managing permissions for data sources and notification channels</a>. For more information about the
         *        roles and permissions that must be managed for customer managed workspaces, see <a
         *        href="https://docs.aws.amazon.com/grafana/latest/userguide/AMG-manage-permissions.html">Amazon Managed
         *        Grafana permissions and policies for Amazon Web Services data sources and notification channels</a>
         * @see PermissionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PermissionType
         */
        Builder permissionType(PermissionType permissionType);

        /**
         * <p>
         * The name of the CloudFormation stack set that is used to generate IAM roles to be used for this workspace.
         * </p>
         * 
         * @param stackSetName
         *        The name of the CloudFormation stack set that is used to generate IAM roles to be used for this
         *        workspace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetName(String stackSetName);

        /**
         * <p>
         * The current status of the workspace.
         * </p>
         * 
         * @param status
         *        The current status of the workspace.
         * @see WorkspaceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkspaceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the workspace.
         * </p>
         * 
         * @param status
         *        The current status of the workspace.
         * @see WorkspaceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkspaceStatus
         */
        Builder status(WorkspaceStatus status);

        /**
         * <p>
         * The list of tags associated with the workspace.
         * </p>
         * 
         * @param tags
         *        The list of tags associated with the workspace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The configuration for connecting to data sources in a private VPC (Amazon Virtual Private Cloud).
         * </p>
         * 
         * @param vpcConfiguration
         *        The configuration for connecting to data sources in a private VPC (Amazon Virtual Private Cloud).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConfiguration(VpcConfiguration vpcConfiguration);

        /**
         * <p>
         * The configuration for connecting to data sources in a private VPC (Amazon Virtual Private Cloud).
         * </p>
         * This is a convenience method that creates an instance of the {@link VpcConfiguration.Builder} avoiding the
         * need to create one manually via {@link VpcConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VpcConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #vpcConfiguration(VpcConfiguration)}.
         * 
         * @param vpcConfiguration
         *        a consumer that will call methods on {@link VpcConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcConfiguration(VpcConfiguration)
         */
        default Builder vpcConfiguration(Consumer<VpcConfiguration.Builder> vpcConfiguration) {
            return vpcConfiguration(VpcConfiguration.builder().applyMutation(vpcConfiguration).build());
        }

        /**
         * <p>
         * The IAM role that grants permissions to the Amazon Web Services resources that the workspace will view data
         * from. This role must already exist.
         * </p>
         * 
         * @param workspaceRoleArn
         *        The IAM role that grants permissions to the Amazon Web Services resources that the workspace will view
         *        data from. This role must already exist.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workspaceRoleArn(String workspaceRoleArn);
    }

    static final class BuilderImpl implements Builder {
        private String accountAccessType;

        private AuthenticationSummary authentication;

        private Instant created;

        private List<String> dataSources = DefaultSdkAutoConstructList.getInstance();

        private String description;

        private String endpoint;

        private Boolean freeTrialConsumed;

        private Instant freeTrialExpiration;

        private String grafanaVersion;

        private String id;

        private Instant licenseExpiration;

        private String licenseType;

        private Instant modified;

        private String name;

        private NetworkAccessConfiguration networkAccessControl;

        private List<String> notificationDestinations = DefaultSdkAutoConstructList.getInstance();

        private String organizationRoleName;

        private List<String> organizationalUnits = DefaultSdkAutoConstructList.getInstance();

        private String permissionType;

        private String stackSetName;

        private String status;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private VpcConfiguration vpcConfiguration;

        private String workspaceRoleArn;

        private BuilderImpl() {
        }

        private BuilderImpl(WorkspaceDescription model) {
            accountAccessType(model.accountAccessType);
            authentication(model.authentication);
            created(model.created);
            dataSourcesWithStrings(model.dataSources);
            description(model.description);
            endpoint(model.endpoint);
            freeTrialConsumed(model.freeTrialConsumed);
            freeTrialExpiration(model.freeTrialExpiration);
            grafanaVersion(model.grafanaVersion);
            id(model.id);
            licenseExpiration(model.licenseExpiration);
            licenseType(model.licenseType);
            modified(model.modified);
            name(model.name);
            networkAccessControl(model.networkAccessControl);
            notificationDestinationsWithStrings(model.notificationDestinations);
            organizationRoleName(model.organizationRoleName);
            organizationalUnits(model.organizationalUnits);
            permissionType(model.permissionType);
            stackSetName(model.stackSetName);
            status(model.status);
            tags(model.tags);
            vpcConfiguration(model.vpcConfiguration);
            workspaceRoleArn(model.workspaceRoleArn);
        }

        public final String getAccountAccessType() {
            return accountAccessType;
        }

        public final void setAccountAccessType(String accountAccessType) {
            this.accountAccessType = accountAccessType;
        }

        @Override
        public final Builder accountAccessType(String accountAccessType) {
            this.accountAccessType = accountAccessType;
            return this;
        }

        @Override
        public final Builder accountAccessType(AccountAccessType accountAccessType) {
            this.accountAccessType(accountAccessType == null ? null : accountAccessType.toString());
            return this;
        }

        public final AuthenticationSummary.Builder getAuthentication() {
            return authentication != null ? authentication.toBuilder() : null;
        }

        public final void setAuthentication(AuthenticationSummary.BuilderImpl authentication) {
            this.authentication = authentication != null ? authentication.build() : null;
        }

        @Override
        public final Builder authentication(AuthenticationSummary authentication) {
            this.authentication = authentication;
            return this;
        }

        public final Instant getCreated() {
            return created;
        }

        public final void setCreated(Instant created) {
            this.created = created;
        }

        @Override
        public final Builder created(Instant created) {
            this.created = created;
            return this;
        }

        public final Collection<String> getDataSources() {
            if (dataSources instanceof SdkAutoConstructList) {
                return null;
            }
            return dataSources;
        }

        public final void setDataSources(Collection<String> dataSources) {
            this.dataSources = DataSourceTypesListCopier.copy(dataSources);
        }

        @Override
        public final Builder dataSourcesWithStrings(Collection<String> dataSources) {
            this.dataSources = DataSourceTypesListCopier.copy(dataSources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dataSourcesWithStrings(String... dataSources) {
            dataSourcesWithStrings(Arrays.asList(dataSources));
            return this;
        }

        @Override
        public final Builder dataSources(Collection<DataSourceType> dataSources) {
            this.dataSources = DataSourceTypesListCopier.copyEnumToString(dataSources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dataSources(DataSourceType... dataSources) {
            dataSources(Arrays.asList(dataSources));
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getEndpoint() {
            return endpoint;
        }

        public final void setEndpoint(String endpoint) {
            this.endpoint = endpoint;
        }

        @Override
        public final Builder endpoint(String endpoint) {
            this.endpoint = endpoint;
            return this;
        }

        public final Boolean getFreeTrialConsumed() {
            return freeTrialConsumed;
        }

        public final void setFreeTrialConsumed(Boolean freeTrialConsumed) {
            this.freeTrialConsumed = freeTrialConsumed;
        }

        @Override
        public final Builder freeTrialConsumed(Boolean freeTrialConsumed) {
            this.freeTrialConsumed = freeTrialConsumed;
            return this;
        }

        public final Instant getFreeTrialExpiration() {
            return freeTrialExpiration;
        }

        public final void setFreeTrialExpiration(Instant freeTrialExpiration) {
            this.freeTrialExpiration = freeTrialExpiration;
        }

        @Override
        public final Builder freeTrialExpiration(Instant freeTrialExpiration) {
            this.freeTrialExpiration = freeTrialExpiration;
            return this;
        }

        public final String getGrafanaVersion() {
            return grafanaVersion;
        }

        public final void setGrafanaVersion(String grafanaVersion) {
            this.grafanaVersion = grafanaVersion;
        }

        @Override
        public final Builder grafanaVersion(String grafanaVersion) {
            this.grafanaVersion = grafanaVersion;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final Instant getLicenseExpiration() {
            return licenseExpiration;
        }

        public final void setLicenseExpiration(Instant licenseExpiration) {
            this.licenseExpiration = licenseExpiration;
        }

        @Override
        public final Builder licenseExpiration(Instant licenseExpiration) {
            this.licenseExpiration = licenseExpiration;
            return this;
        }

        public final String getLicenseType() {
            return licenseType;
        }

        public final void setLicenseType(String licenseType) {
            this.licenseType = licenseType;
        }

        @Override
        public final Builder licenseType(String licenseType) {
            this.licenseType = licenseType;
            return this;
        }

        @Override
        public final Builder licenseType(LicenseType licenseType) {
            this.licenseType(licenseType == null ? null : licenseType.toString());
            return this;
        }

        public final Instant getModified() {
            return modified;
        }

        public final void setModified(Instant modified) {
            this.modified = modified;
        }

        @Override
        public final Builder modified(Instant modified) {
            this.modified = modified;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final NetworkAccessConfiguration.Builder getNetworkAccessControl() {
            return networkAccessControl != null ? networkAccessControl.toBuilder() : null;
        }

        public final void setNetworkAccessControl(NetworkAccessConfiguration.BuilderImpl networkAccessControl) {
            this.networkAccessControl = networkAccessControl != null ? networkAccessControl.build() : null;
        }

        @Override
        public final Builder networkAccessControl(NetworkAccessConfiguration networkAccessControl) {
            this.networkAccessControl = networkAccessControl;
            return this;
        }

        public final Collection<String> getNotificationDestinations() {
            if (notificationDestinations instanceof SdkAutoConstructList) {
                return null;
            }
            return notificationDestinations;
        }

        public final void setNotificationDestinations(Collection<String> notificationDestinations) {
            this.notificationDestinations = NotificationDestinationsListCopier.copy(notificationDestinations);
        }

        @Override
        public final Builder notificationDestinationsWithStrings(Collection<String> notificationDestinations) {
            this.notificationDestinations = NotificationDestinationsListCopier.copy(notificationDestinations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notificationDestinationsWithStrings(String... notificationDestinations) {
            notificationDestinationsWithStrings(Arrays.asList(notificationDestinations));
            return this;
        }

        @Override
        public final Builder notificationDestinations(Collection<NotificationDestinationType> notificationDestinations) {
            this.notificationDestinations = NotificationDestinationsListCopier.copyEnumToString(notificationDestinations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notificationDestinations(NotificationDestinationType... notificationDestinations) {
            notificationDestinations(Arrays.asList(notificationDestinations));
            return this;
        }

        public final String getOrganizationRoleName() {
            return organizationRoleName;
        }

        public final void setOrganizationRoleName(String organizationRoleName) {
            this.organizationRoleName = organizationRoleName;
        }

        @Override
        public final Builder organizationRoleName(String organizationRoleName) {
            this.organizationRoleName = organizationRoleName;
            return this;
        }

        public final Collection<String> getOrganizationalUnits() {
            if (organizationalUnits instanceof SdkAutoConstructList) {
                return null;
            }
            return organizationalUnits;
        }

        public final void setOrganizationalUnits(Collection<String> organizationalUnits) {
            this.organizationalUnits = OrganizationalUnitListCopier.copy(organizationalUnits);
        }

        @Override
        public final Builder organizationalUnits(Collection<String> organizationalUnits) {
            this.organizationalUnits = OrganizationalUnitListCopier.copy(organizationalUnits);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder organizationalUnits(String... organizationalUnits) {
            organizationalUnits(Arrays.asList(organizationalUnits));
            return this;
        }

        public final String getPermissionType() {
            return permissionType;
        }

        public final void setPermissionType(String permissionType) {
            this.permissionType = permissionType;
        }

        @Override
        public final Builder permissionType(String permissionType) {
            this.permissionType = permissionType;
            return this;
        }

        @Override
        public final Builder permissionType(PermissionType permissionType) {
            this.permissionType(permissionType == null ? null : permissionType.toString());
            return this;
        }

        public final String getStackSetName() {
            return stackSetName;
        }

        public final void setStackSetName(String stackSetName) {
            this.stackSetName = stackSetName;
        }

        @Override
        public final Builder stackSetName(String stackSetName) {
            this.stackSetName = stackSetName;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(WorkspaceStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final VpcConfiguration.Builder getVpcConfiguration() {
            return vpcConfiguration != null ? vpcConfiguration.toBuilder() : null;
        }

        public final void setVpcConfiguration(VpcConfiguration.BuilderImpl vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration != null ? vpcConfiguration.build() : null;
        }

        @Override
        public final Builder vpcConfiguration(VpcConfiguration vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration;
            return this;
        }

        public final String getWorkspaceRoleArn() {
            return workspaceRoleArn;
        }

        public final void setWorkspaceRoleArn(String workspaceRoleArn) {
            this.workspaceRoleArn = workspaceRoleArn;
        }

        @Override
        public final Builder workspaceRoleArn(String workspaceRoleArn) {
            this.workspaceRoleArn = workspaceRoleArn;
            return this;
        }

        @Override
        public WorkspaceDescription build() {
            return new WorkspaceDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
