/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.evidently;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.evidently.model.AccessDeniedException;
import software.amazon.awssdk.services.evidently.model.BatchEvaluateFeatureRequest;
import software.amazon.awssdk.services.evidently.model.BatchEvaluateFeatureResponse;
import software.amazon.awssdk.services.evidently.model.ConflictException;
import software.amazon.awssdk.services.evidently.model.CreateExperimentRequest;
import software.amazon.awssdk.services.evidently.model.CreateExperimentResponse;
import software.amazon.awssdk.services.evidently.model.CreateFeatureRequest;
import software.amazon.awssdk.services.evidently.model.CreateFeatureResponse;
import software.amazon.awssdk.services.evidently.model.CreateLaunchRequest;
import software.amazon.awssdk.services.evidently.model.CreateLaunchResponse;
import software.amazon.awssdk.services.evidently.model.CreateProjectRequest;
import software.amazon.awssdk.services.evidently.model.CreateProjectResponse;
import software.amazon.awssdk.services.evidently.model.CreateSegmentRequest;
import software.amazon.awssdk.services.evidently.model.CreateSegmentResponse;
import software.amazon.awssdk.services.evidently.model.DeleteExperimentRequest;
import software.amazon.awssdk.services.evidently.model.DeleteExperimentResponse;
import software.amazon.awssdk.services.evidently.model.DeleteFeatureRequest;
import software.amazon.awssdk.services.evidently.model.DeleteFeatureResponse;
import software.amazon.awssdk.services.evidently.model.DeleteLaunchRequest;
import software.amazon.awssdk.services.evidently.model.DeleteLaunchResponse;
import software.amazon.awssdk.services.evidently.model.DeleteProjectRequest;
import software.amazon.awssdk.services.evidently.model.DeleteProjectResponse;
import software.amazon.awssdk.services.evidently.model.DeleteSegmentRequest;
import software.amazon.awssdk.services.evidently.model.DeleteSegmentResponse;
import software.amazon.awssdk.services.evidently.model.EvaluateFeatureRequest;
import software.amazon.awssdk.services.evidently.model.EvaluateFeatureResponse;
import software.amazon.awssdk.services.evidently.model.EvidentlyException;
import software.amazon.awssdk.services.evidently.model.GetExperimentRequest;
import software.amazon.awssdk.services.evidently.model.GetExperimentResponse;
import software.amazon.awssdk.services.evidently.model.GetExperimentResultsRequest;
import software.amazon.awssdk.services.evidently.model.GetExperimentResultsResponse;
import software.amazon.awssdk.services.evidently.model.GetFeatureRequest;
import software.amazon.awssdk.services.evidently.model.GetFeatureResponse;
import software.amazon.awssdk.services.evidently.model.GetLaunchRequest;
import software.amazon.awssdk.services.evidently.model.GetLaunchResponse;
import software.amazon.awssdk.services.evidently.model.GetProjectRequest;
import software.amazon.awssdk.services.evidently.model.GetProjectResponse;
import software.amazon.awssdk.services.evidently.model.GetSegmentRequest;
import software.amazon.awssdk.services.evidently.model.GetSegmentResponse;
import software.amazon.awssdk.services.evidently.model.InternalServerException;
import software.amazon.awssdk.services.evidently.model.ListExperimentsRequest;
import software.amazon.awssdk.services.evidently.model.ListExperimentsResponse;
import software.amazon.awssdk.services.evidently.model.ListFeaturesRequest;
import software.amazon.awssdk.services.evidently.model.ListFeaturesResponse;
import software.amazon.awssdk.services.evidently.model.ListLaunchesRequest;
import software.amazon.awssdk.services.evidently.model.ListLaunchesResponse;
import software.amazon.awssdk.services.evidently.model.ListProjectsRequest;
import software.amazon.awssdk.services.evidently.model.ListProjectsResponse;
import software.amazon.awssdk.services.evidently.model.ListSegmentReferencesRequest;
import software.amazon.awssdk.services.evidently.model.ListSegmentReferencesResponse;
import software.amazon.awssdk.services.evidently.model.ListSegmentsRequest;
import software.amazon.awssdk.services.evidently.model.ListSegmentsResponse;
import software.amazon.awssdk.services.evidently.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.evidently.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.evidently.model.PutProjectEventsRequest;
import software.amazon.awssdk.services.evidently.model.PutProjectEventsResponse;
import software.amazon.awssdk.services.evidently.model.ResourceNotFoundException;
import software.amazon.awssdk.services.evidently.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.evidently.model.ServiceUnavailableException;
import software.amazon.awssdk.services.evidently.model.StartExperimentRequest;
import software.amazon.awssdk.services.evidently.model.StartExperimentResponse;
import software.amazon.awssdk.services.evidently.model.StartLaunchRequest;
import software.amazon.awssdk.services.evidently.model.StartLaunchResponse;
import software.amazon.awssdk.services.evidently.model.StopExperimentRequest;
import software.amazon.awssdk.services.evidently.model.StopExperimentResponse;
import software.amazon.awssdk.services.evidently.model.StopLaunchRequest;
import software.amazon.awssdk.services.evidently.model.StopLaunchResponse;
import software.amazon.awssdk.services.evidently.model.TagResourceRequest;
import software.amazon.awssdk.services.evidently.model.TagResourceResponse;
import software.amazon.awssdk.services.evidently.model.TestSegmentPatternRequest;
import software.amazon.awssdk.services.evidently.model.TestSegmentPatternResponse;
import software.amazon.awssdk.services.evidently.model.ThrottlingException;
import software.amazon.awssdk.services.evidently.model.UntagResourceRequest;
import software.amazon.awssdk.services.evidently.model.UntagResourceResponse;
import software.amazon.awssdk.services.evidently.model.UpdateExperimentRequest;
import software.amazon.awssdk.services.evidently.model.UpdateExperimentResponse;
import software.amazon.awssdk.services.evidently.model.UpdateFeatureRequest;
import software.amazon.awssdk.services.evidently.model.UpdateFeatureResponse;
import software.amazon.awssdk.services.evidently.model.UpdateLaunchRequest;
import software.amazon.awssdk.services.evidently.model.UpdateLaunchResponse;
import software.amazon.awssdk.services.evidently.model.UpdateProjectDataDeliveryRequest;
import software.amazon.awssdk.services.evidently.model.UpdateProjectDataDeliveryResponse;
import software.amazon.awssdk.services.evidently.model.UpdateProjectRequest;
import software.amazon.awssdk.services.evidently.model.UpdateProjectResponse;
import software.amazon.awssdk.services.evidently.model.ValidationException;
import software.amazon.awssdk.services.evidently.transform.BatchEvaluateFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.CreateExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.CreateFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.CreateLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.CreateProjectRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.CreateSegmentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.DeleteExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.DeleteFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.DeleteLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.DeleteProjectRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.DeleteSegmentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.EvaluateFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.GetExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.GetExperimentResultsRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.GetFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.GetLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.GetProjectRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.GetSegmentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListExperimentsRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListFeaturesRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListLaunchesRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListProjectsRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListSegmentReferencesRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListSegmentsRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.PutProjectEventsRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.StartExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.StartLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.StopExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.StopLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.TestSegmentPatternRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UpdateExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UpdateFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UpdateLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UpdateProjectDataDeliveryRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UpdateProjectRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link EvidentlyClient}.
 *
 * @see EvidentlyClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEvidentlyClient implements EvidentlyClient {
    private static final Logger log = Logger.loggerFor(DefaultEvidentlyClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final EvidentlyServiceClientConfiguration serviceClientConfiguration;

    protected DefaultEvidentlyClient(EvidentlyServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * This operation assigns feature variation to user sessions. For each user session, you pass in an
     * <code>entityID</code> that represents the user. Evidently then checks the evaluation rules and assigns the
     * variation.
     * </p>
     * <p>
     * The first rules that are evaluated are the override rules. If the user's <code>entityID</code> matches an
     * override rule, the user is served the variation specified by that rule.
     * </p>
     * <p>
     * Next, if there is a launch of the feature, the user might be assigned to a variation in the launch. The chance of
     * this depends on the percentage of users that are allocated to that launch. If the user is enrolled in the launch,
     * the variation they are served depends on the allocation of the various feature variations used for the launch.
     * </p>
     * <p>
     * If the user is not assigned to a launch, and there is an ongoing experiment for this feature, the user might be
     * assigned to a variation in the experiment. The chance of this depends on the percentage of users that are
     * allocated to that experiment. If the user is enrolled in the experiment, the variation they are served depends on
     * the allocation of the various feature variations used for the experiment.
     * </p>
     * <p>
     * If the user is not assigned to a launch or experiment, they are served the default variation.
     * </p>
     *
     * @param batchEvaluateFeatureRequest
     * @return Result of the BatchEvaluateFeature operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.BatchEvaluateFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/BatchEvaluateFeature"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchEvaluateFeatureResponse batchEvaluateFeature(BatchEvaluateFeatureRequest batchEvaluateFeatureRequest)
            throws ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchEvaluateFeatureResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchEvaluateFeatureResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchEvaluateFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchEvaluateFeature");
            String hostPrefix = "dataplane.";
            String resolvedHostExpression = "dataplane.";

            return clientHandler.execute(new ClientExecutionParams<BatchEvaluateFeatureRequest, BatchEvaluateFeatureResponse>()
                    .withOperationName("BatchEvaluateFeature").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(batchEvaluateFeatureRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchEvaluateFeatureRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Evidently <i>experiment</i>. Before you create an experiment, you must create the feature to use for
     * the experiment.
     * </p>
     * <p>
     * An experiment helps you make feature design decisions based on evidence and data. An experiment can test as many
     * as five variations at once. Evidently collects experiment data and analyzes it by statistical methods, and
     * provides clear recommendations about which variations perform better.
     * </p>
     * <p>
     * You can optionally specify a <code>segment</code> to have the experiment consider only certain audience types in
     * the experiment, such as using only user sessions from a certain location or who use a certain internet browser.
     * </p>
     * <p>
     * Don't use this operation to update an existing experiment. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_UpdateExperiment.html"
     * >UpdateExperiment</a>.
     * </p>
     *
     * @param createExperimentRequest
     * @return Result of the CreateExperiment operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.CreateExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/CreateExperiment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateExperimentResponse createExperiment(CreateExperimentRequest createExperimentRequest) throws ValidationException,
            ConflictException, ServiceQuotaExceededException, ResourceNotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateExperimentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateExperimentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateExperiment");

            return clientHandler.execute(new ClientExecutionParams<CreateExperimentRequest, CreateExperimentResponse>()
                    .withOperationName("CreateExperiment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createExperimentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateExperimentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Evidently <i>feature</i> that you want to launch or test. You can define up to five variations of a
     * feature, and use these variations in your launches and experiments. A feature must be created in a project. For
     * information about creating a project, see <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_CreateProject.html"
     * >CreateProject</a>.
     * </p>
     * <p>
     * Don't use this operation to update an existing feature. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_UpdateFeature.html"
     * >UpdateFeature</a>.
     * </p>
     *
     * @param createFeatureRequest
     * @return Result of the CreateFeature operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.CreateFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/CreateFeature" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateFeatureResponse createFeature(CreateFeatureRequest createFeatureRequest) throws ValidationException,
            ConflictException, ServiceQuotaExceededException, ResourceNotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateFeatureResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateFeatureResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFeature");

            return clientHandler.execute(new ClientExecutionParams<CreateFeatureRequest, CreateFeatureResponse>()
                    .withOperationName("CreateFeature").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createFeatureRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateFeatureRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a <i>launch</i> of a given feature. Before you create a launch, you must create the feature to use for
     * the launch.
     * </p>
     * <p>
     * You can use a launch to safely validate new features by serving them to a specified percentage of your users
     * while you roll out the feature. You can monitor the performance of the new feature to help you decide when to
     * ramp up traffic to more users. This helps you reduce risk and identify unintended consequences before you fully
     * launch the feature.
     * </p>
     * <p>
     * Don't use this operation to update an existing launch. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_UpdateLaunch.html"
     * >UpdateLaunch</a>.
     * </p>
     *
     * @param createLaunchRequest
     * @return Result of the CreateLaunch operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.CreateLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/CreateLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateLaunchResponse createLaunch(CreateLaunchRequest createLaunchRequest) throws ValidationException,
            ConflictException, ServiceQuotaExceededException, ResourceNotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateLaunchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLaunch");

            return clientHandler.execute(new ClientExecutionParams<CreateLaunchRequest, CreateLaunchResponse>()
                    .withOperationName("CreateLaunch").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createLaunchRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateLaunchRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a project, which is the logical object in Evidently that can contain features, launches, and experiments.
     * Use projects to group similar features together.
     * </p>
     * <p>
     * To update an existing project, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_UpdateProject.html"
     * >UpdateProject</a>.
     * </p>
     *
     * @param createProjectRequest
     * @return Result of the CreateProject operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.CreateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/CreateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateProjectResponse createProject(CreateProjectRequest createProjectRequest) throws ValidationException,
            ConflictException, ServiceQuotaExceededException, AccessDeniedException, AwsServiceException, SdkClientException,
            EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProject");

            return clientHandler.execute(new ClientExecutionParams<CreateProjectRequest, CreateProjectResponse>()
                    .withOperationName("CreateProject").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createProjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use this operation to define a <i>segment</i> of your audience. A segment is a portion of your audience that
     * share one or more characteristics. Examples could be Chrome browser users, users in Europe, or Firefox browser
     * users in Europe who also fit other criteria that your application collects, such as age.
     * </p>
     * <p>
     * Using a segment in an experiment limits that experiment to evaluate only the users who match the segment
     * criteria. Using one or more segments in a launch allows you to define different traffic splits for the different
     * audience segments.
     * </p>
     * <p>
     * For more information about segment pattern syntax, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Evidently-segments.html#CloudWatch-Evidently-segments-syntax.html"
     * > Segment rule pattern syntax</a>.
     * </p>
     * <p>
     * The pattern that you define for a segment is matched against the value of <code>evaluationContext</code>, which
     * is passed into Evidently in the <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_EvaluateFeature.html"
     * >EvaluateFeature</a> operation, when Evidently assigns a feature variation to a user.
     * </p>
     *
     * @param createSegmentRequest
     * @return Result of the CreateSegment operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.CreateSegment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/CreateSegment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateSegmentResponse createSegment(CreateSegmentRequest createSegmentRequest) throws ValidationException,
            ConflictException, ServiceQuotaExceededException, AccessDeniedException, AwsServiceException, SdkClientException,
            EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSegmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateSegmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSegmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSegment");

            return clientHandler.execute(new ClientExecutionParams<CreateSegmentRequest, CreateSegmentResponse>()
                    .withOperationName("CreateSegment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createSegmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSegmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Evidently experiment. The feature used for the experiment is not deleted.
     * </p>
     * <p>
     * To stop an experiment without deleting it, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_StopExperiment.html"
     * >StopExperiment</a>.
     * </p>
     *
     * @param deleteExperimentRequest
     * @return Result of the DeleteExperiment operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ServiceUnavailableException
     *         The service was unavailable. Retry the request.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.DeleteExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/DeleteExperiment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteExperimentResponse deleteExperiment(DeleteExperimentRequest deleteExperimentRequest) throws ValidationException,
            InternalServerException, ConflictException, ServiceUnavailableException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteExperimentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteExperimentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteExperiment");

            return clientHandler.execute(new ClientExecutionParams<DeleteExperimentRequest, DeleteExperimentResponse>()
                    .withOperationName("DeleteExperiment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteExperimentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteExperimentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Evidently feature.
     * </p>
     *
     * @param deleteFeatureRequest
     * @return Result of the DeleteFeature operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.DeleteFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/DeleteFeature" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteFeatureResponse deleteFeature(DeleteFeatureRequest deleteFeatureRequest) throws ThrottlingException,
            ValidationException, ConflictException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFeatureResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteFeatureResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFeature");

            return clientHandler.execute(new ClientExecutionParams<DeleteFeatureRequest, DeleteFeatureResponse>()
                    .withOperationName("DeleteFeature").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteFeatureRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteFeatureRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Evidently launch. The feature used for the launch is not deleted.
     * </p>
     * <p>
     * To stop a launch without deleting it, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_StopLaunch.html">StopLaunch</a>.
     * </p>
     *
     * @param deleteLaunchRequest
     * @return Result of the DeleteLaunch operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.DeleteLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/DeleteLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteLaunchResponse deleteLaunch(DeleteLaunchRequest deleteLaunchRequest) throws ThrottlingException,
            ValidationException, ConflictException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteLaunchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLaunch");

            return clientHandler.execute(new ClientExecutionParams<DeleteLaunchRequest, DeleteLaunchResponse>()
                    .withOperationName("DeleteLaunch").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteLaunchRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteLaunchRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Evidently project. Before you can delete a project, you must delete all the features that the project
     * contains. To delete a feature, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_DeleteFeature.html"
     * >DeleteFeature</a>.
     * </p>
     *
     * @param deleteProjectRequest
     * @return Result of the DeleteProject operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.DeleteProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/DeleteProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteProjectResponse deleteProject(DeleteProjectRequest deleteProjectRequest) throws ThrottlingException,
            ValidationException, ConflictException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProject");

            return clientHandler.execute(new ClientExecutionParams<DeleteProjectRequest, DeleteProjectResponse>()
                    .withOperationName("DeleteProject").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteProjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a segment. You can't delete a segment that is being used in a launch or experiment, even if that launch
     * or experiment is not currently running.
     * </p>
     *
     * @param deleteSegmentRequest
     * @return Result of the DeleteSegment operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.DeleteSegment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/DeleteSegment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteSegmentResponse deleteSegment(DeleteSegmentRequest deleteSegmentRequest) throws ThrottlingException,
            ValidationException, ConflictException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSegmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteSegmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSegmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSegment");

            return clientHandler.execute(new ClientExecutionParams<DeleteSegmentRequest, DeleteSegmentResponse>()
                    .withOperationName("DeleteSegment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteSegmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSegmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation assigns a feature variation to one given user session. You pass in an <code>entityID</code> that
     * represents the user. Evidently then checks the evaluation rules and assigns the variation.
     * </p>
     * <p>
     * The first rules that are evaluated are the override rules. If the user's <code>entityID</code> matches an
     * override rule, the user is served the variation specified by that rule.
     * </p>
     * <p>
     * If there is a current launch with this feature that uses segment overrides, and if the user session's
     * <code>evaluationContext</code> matches a segment rule defined in a segment override, the configuration in the
     * segment overrides is used. For more information about segments, see <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_CreateSegment.html"
     * >CreateSegment</a> and <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Evidently-segments.html">Use
     * segments to focus your audience</a>.
     * </p>
     * <p>
     * If there is a launch with no segment overrides, the user might be assigned to a variation in the launch. The
     * chance of this depends on the percentage of users that are allocated to that launch. If the user is enrolled in
     * the launch, the variation they are served depends on the allocation of the various feature variations used for
     * the launch.
     * </p>
     * <p>
     * If the user is not assigned to a launch, and there is an ongoing experiment for this feature, the user might be
     * assigned to a variation in the experiment. The chance of this depends on the percentage of users that are
     * allocated to that experiment.
     * </p>
     * <p>
     * If the experiment uses a segment, then only user sessions with <code>evaluationContext</code> values that match
     * the segment rule are used in the experiment.
     * </p>
     * <p>
     * If the user is enrolled in the experiment, the variation they are served depends on the allocation of the various
     * feature variations used for the experiment.
     * </p>
     * <p>
     * If the user is not assigned to a launch or experiment, they are served the default variation.
     * </p>
     *
     * @param evaluateFeatureRequest
     * @return Result of the EvaluateFeature operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.EvaluateFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/EvaluateFeature" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public EvaluateFeatureResponse evaluateFeature(EvaluateFeatureRequest evaluateFeatureRequest) throws ThrottlingException,
            ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EvaluateFeatureResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                EvaluateFeatureResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, evaluateFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EvaluateFeature");
            String hostPrefix = "dataplane.";
            String resolvedHostExpression = "dataplane.";

            return clientHandler.execute(new ClientExecutionParams<EvaluateFeatureRequest, EvaluateFeatureResponse>()
                    .withOperationName("EvaluateFeature").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(evaluateFeatureRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new EvaluateFeatureRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the details about one experiment. You must already know the experiment name. To retrieve a list of
     * experiments in your account, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_ListExperiments.html"
     * >ListExperiments</a>.
     * </p>
     *
     * @param getExperimentRequest
     * @return Result of the GetExperiment operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.GetExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/GetExperiment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetExperimentResponse getExperiment(GetExperimentRequest getExperimentRequest) throws ThrottlingException,
            ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetExperimentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetExperimentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExperiment");

            return clientHandler.execute(new ClientExecutionParams<GetExperimentRequest, GetExperimentResponse>()
                    .withOperationName("GetExperiment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getExperimentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetExperimentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the results of a running or completed experiment. No results are available until there have been 100
     * events for each variation and at least 10 minutes have passed since the start of the experiment. To increase the
     * statistical power, Evidently performs an additional offline p-value analysis at the end of the experiment.
     * Offline p-value analysis can detect statistical significance in some cases where the anytime p-values used during
     * the experiment do not find statistical significance.
     * </p>
     * <p>
     * Experiment results are available up to 63 days after the start of the experiment. They are not available after
     * that because of CloudWatch data retention policies.
     * </p>
     *
     * @param getExperimentResultsRequest
     * @return Result of the GetExperimentResults operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.GetExperimentResults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/GetExperimentResults"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetExperimentResultsResponse getExperimentResults(GetExperimentResultsRequest getExperimentResultsRequest)
            throws ThrottlingException, ValidationException, ConflictException, ResourceNotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetExperimentResultsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetExperimentResultsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExperimentResultsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExperimentResults");

            return clientHandler.execute(new ClientExecutionParams<GetExperimentResultsRequest, GetExperimentResultsResponse>()
                    .withOperationName("GetExperimentResults").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getExperimentResultsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetExperimentResultsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the details about one feature. You must already know the feature name. To retrieve a list of features in
     * your account, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_ListFeatures.html"
     * >ListFeatures</a>.
     * </p>
     *
     * @param getFeatureRequest
     * @return Result of the GetFeature operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.GetFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/GetFeature" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetFeatureResponse getFeature(GetFeatureRequest getFeatureRequest) throws ThrottlingException, ValidationException,
            ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetFeatureResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetFeatureResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFeature");

            return clientHandler
                    .execute(new ClientExecutionParams<GetFeatureRequest, GetFeatureResponse>().withOperationName("GetFeature")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getFeatureRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetFeatureRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the details about one launch. You must already know the launch name. To retrieve a list of launches in
     * your account, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_ListLaunches.html"
     * >ListLaunches</a>.
     * </p>
     *
     * @param getLaunchRequest
     * @return Result of the GetLaunch operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.GetLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/GetLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetLaunchResponse getLaunch(GetLaunchRequest getLaunchRequest) throws ThrottlingException, ValidationException,
            ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetLaunchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLaunch");

            return clientHandler.execute(new ClientExecutionParams<GetLaunchRequest, GetLaunchResponse>()
                    .withOperationName("GetLaunch").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getLaunchRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetLaunchRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the details about one launch. You must already know the project name. To retrieve a list of projects in
     * your account, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_ListProjects.html"
     * >ListProjects</a>.
     * </p>
     *
     * @param getProjectRequest
     * @return Result of the GetProject operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.GetProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/GetProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetProjectResponse getProject(GetProjectRequest getProjectRequest) throws ThrottlingException, ValidationException,
            ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProject");

            return clientHandler
                    .execute(new ClientExecutionParams<GetProjectRequest, GetProjectResponse>().withOperationName("GetProject")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getProjectRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the specified segment. Specify the segment you want to view by specifying its ARN.
     * </p>
     *
     * @param getSegmentRequest
     * @return Result of the GetSegment operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.GetSegment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/GetSegment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSegmentResponse getSegment(GetSegmentRequest getSegmentRequest) throws ThrottlingException, ValidationException,
            ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSegmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSegmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSegmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSegment");

            return clientHandler
                    .execute(new ClientExecutionParams<GetSegmentRequest, GetSegmentResponse>().withOperationName("GetSegment")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getSegmentRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetSegmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns configuration details about all the experiments in the specified project.
     * </p>
     *
     * @param listExperimentsRequest
     * @return Result of the ListExperiments operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.ListExperiments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListExperiments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListExperimentsResponse listExperiments(ListExperimentsRequest listExperimentsRequest) throws ValidationException,
            ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListExperimentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListExperimentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExperimentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExperiments");

            return clientHandler.execute(new ClientExecutionParams<ListExperimentsRequest, ListExperimentsResponse>()
                    .withOperationName("ListExperiments").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listExperimentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListExperimentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns configuration details about all the features in the specified project.
     * </p>
     *
     * @param listFeaturesRequest
     * @return Result of the ListFeatures operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.ListFeatures
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListFeatures" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListFeaturesResponse listFeatures(ListFeaturesRequest listFeaturesRequest) throws ThrottlingException,
            ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFeaturesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListFeaturesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFeaturesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFeatures");

            return clientHandler.execute(new ClientExecutionParams<ListFeaturesRequest, ListFeaturesResponse>()
                    .withOperationName("ListFeatures").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listFeaturesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListFeaturesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns configuration details about all the launches in the specified project.
     * </p>
     *
     * @param listLaunchesRequest
     * @return Result of the ListLaunches operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.ListLaunches
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListLaunches" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListLaunchesResponse listLaunches(ListLaunchesRequest listLaunchesRequest) throws ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLaunchesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListLaunchesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLaunchesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLaunches");

            return clientHandler.execute(new ClientExecutionParams<ListLaunchesRequest, ListLaunchesResponse>()
                    .withOperationName("ListLaunches").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listLaunchesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListLaunchesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns configuration details about all the projects in the current Region in your account.
     * </p>
     *
     * @param listProjectsRequest
     * @return Result of the ListProjects operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.ListProjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListProjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListProjectsResponse listProjects(ListProjectsRequest listProjectsRequest) throws ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListProjectsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProjects");

            return clientHandler.execute(new ClientExecutionParams<ListProjectsRequest, ListProjectsResponse>()
                    .withOperationName("ListProjects").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listProjectsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListProjectsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use this operation to find which experiments or launches are using a specified segment.
     * </p>
     *
     * @param listSegmentReferencesRequest
     * @return Result of the ListSegmentReferences operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.ListSegmentReferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListSegmentReferences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSegmentReferencesResponse listSegmentReferences(ListSegmentReferencesRequest listSegmentReferencesRequest)
            throws ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSegmentReferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSegmentReferencesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSegmentReferencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSegmentReferences");

            return clientHandler.execute(new ClientExecutionParams<ListSegmentReferencesRequest, ListSegmentReferencesResponse>()
                    .withOperationName("ListSegmentReferences").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSegmentReferencesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSegmentReferencesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of audience segments that you have created in your account in this Region.
     * </p>
     *
     * @param listSegmentsRequest
     * @return Result of the ListSegments operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.ListSegments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListSegments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListSegmentsResponse listSegments(ListSegmentsRequest listSegmentsRequest) throws ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSegmentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSegmentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSegmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSegments");

            return clientHandler.execute(new ClientExecutionParams<ListSegmentsRequest, ListSegmentsResponse>()
                    .withOperationName("ListSegments").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSegmentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSegmentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays the tags associated with an Evidently resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, ConflictException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sends performance events to Evidently. These events can be used to evaluate a launch or an experiment.
     * </p>
     *
     * @param putProjectEventsRequest
     * @return Result of the PutProjectEvents operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.PutProjectEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/PutProjectEvents" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutProjectEventsResponse putProjectEvents(PutProjectEventsRequest putProjectEventsRequest) throws ThrottlingException,
            ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutProjectEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutProjectEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putProjectEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutProjectEvents");
            String hostPrefix = "dataplane.";
            String resolvedHostExpression = "dataplane.";

            return clientHandler.execute(new ClientExecutionParams<PutProjectEventsRequest, PutProjectEventsResponse>()
                    .withOperationName("PutProjectEvents").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(putProjectEventsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutProjectEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts an existing experiment. To create an experiment, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_CreateExperiment.html"
     * >CreateExperiment</a>.
     * </p>
     *
     * @param startExperimentRequest
     * @return Result of the StartExperiment operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.StartExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/StartExperiment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartExperimentResponse startExperiment(StartExperimentRequest startExperimentRequest) throws ThrottlingException,
            ValidationException, ConflictException, ServiceQuotaExceededException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartExperimentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartExperimentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartExperiment");

            return clientHandler.execute(new ClientExecutionParams<StartExperimentRequest, StartExperimentResponse>()
                    .withOperationName("StartExperiment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startExperimentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartExperimentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts an existing launch. To create a launch, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_CreateLaunch.html"
     * >CreateLaunch</a>.
     * </p>
     *
     * @param startLaunchRequest
     * @return Result of the StartLaunch operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.StartLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/StartLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartLaunchResponse startLaunch(StartLaunchRequest startLaunchRequest) throws ThrottlingException,
            ValidationException, ConflictException, ServiceQuotaExceededException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartLaunchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartLaunch");

            return clientHandler.execute(new ClientExecutionParams<StartLaunchRequest, StartLaunchResponse>()
                    .withOperationName("StartLaunch").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startLaunchRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartLaunchRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops an experiment that is currently running. If you stop an experiment, you can't resume it or restart it.
     * </p>
     *
     * @param stopExperimentRequest
     * @return Result of the StopExperiment operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.StopExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/StopExperiment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopExperimentResponse stopExperiment(StopExperimentRequest stopExperimentRequest) throws ThrottlingException,
            ValidationException, ConflictException, ServiceQuotaExceededException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopExperimentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopExperimentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopExperiment");

            return clientHandler.execute(new ClientExecutionParams<StopExperimentRequest, StopExperimentResponse>()
                    .withOperationName("StopExperiment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopExperimentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopExperimentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops a launch that is currently running. After you stop a launch, you will not be able to resume it or restart
     * it. Also, it will not be evaluated as a rule for traffic allocation, and the traffic that was allocated to the
     * launch will instead be available to the feature's experiment, if there is one. Otherwise, all traffic will be
     * served the default variation after the launch is stopped.
     * </p>
     *
     * @param stopLaunchRequest
     * @return Result of the StopLaunch operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.StopLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/StopLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopLaunchResponse stopLaunch(StopLaunchRequest stopLaunchRequest) throws ThrottlingException, ValidationException,
            ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopLaunchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopLaunch");

            return clientHandler
                    .execute(new ClientExecutionParams<StopLaunchRequest, StopLaunchResponse>().withOperationName("StopLaunch")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(stopLaunchRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StopLaunchRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified CloudWatch Evidently resource. Projects, features,
     * launches, and experiments can be tagged.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a resource that already has tags. If you specify a new tag
     * key for the resource, this tag is appended to the list of tags associated with the alarm. If you specify a tag
     * key that is already associated with the resource, the new tag value that you specify replaces the previous value
     * for that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a resource.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon
     * Web Services resources</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException, ConflictException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use this operation to test a rules pattern that you plan to use to create an audience segment. For more
     * information about segments, see <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_CreateSegment.html"
     * >CreateSegment</a>.
     * </p>
     *
     * @param testSegmentPatternRequest
     * @return Result of the TestSegmentPattern operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.TestSegmentPattern
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/TestSegmentPattern" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TestSegmentPatternResponse testSegmentPattern(TestSegmentPatternRequest testSegmentPatternRequest)
            throws ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException, SdkClientException,
            EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TestSegmentPatternResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, TestSegmentPatternResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, testSegmentPatternRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TestSegmentPattern");

            return clientHandler.execute(new ClientExecutionParams<TestSegmentPatternRequest, TestSegmentPatternResponse>()
                    .withOperationName("TestSegmentPattern").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(testSegmentPatternRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TestSegmentPatternRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ConflictException, ResourceNotFoundException, AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an Evidently experiment.
     * </p>
     * <p>
     * Don't use this operation to update an experiment's tag. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     *
     * @param updateExperimentRequest
     * @return Result of the UpdateExperiment operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.UpdateExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UpdateExperiment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateExperimentResponse updateExperiment(UpdateExperimentRequest updateExperimentRequest) throws ValidationException,
            ConflictException, ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateExperimentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateExperimentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateExperiment");

            return clientHandler.execute(new ClientExecutionParams<UpdateExperimentRequest, UpdateExperimentResponse>()
                    .withOperationName("UpdateExperiment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateExperimentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateExperimentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing feature.
     * </p>
     * <p>
     * You can't use this operation to update the tags of an existing feature. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     *
     * @param updateFeatureRequest
     * @return Result of the UpdateFeature operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.UpdateFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UpdateFeature" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateFeatureResponse updateFeature(UpdateFeatureRequest updateFeatureRequest) throws ValidationException,
            ConflictException, ServiceQuotaExceededException, ResourceNotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFeatureResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateFeatureResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFeature");

            return clientHandler.execute(new ClientExecutionParams<UpdateFeatureRequest, UpdateFeatureResponse>()
                    .withOperationName("UpdateFeature").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateFeatureRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateFeatureRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a launch of a given feature.
     * </p>
     * <p>
     * Don't use this operation to update the tags of an existing launch. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     *
     * @param updateLaunchRequest
     * @return Result of the UpdateLaunch operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.UpdateLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UpdateLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateLaunchResponse updateLaunch(UpdateLaunchRequest updateLaunchRequest) throws ValidationException,
            ConflictException, ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateLaunchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLaunch");

            return clientHandler.execute(new ClientExecutionParams<UpdateLaunchRequest, UpdateLaunchResponse>()
                    .withOperationName("UpdateLaunch").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateLaunchRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateLaunchRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the description of an existing project.
     * </p>
     * <p>
     * To create a new project, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_CreateProject.html"
     * >CreateProject</a>.
     * </p>
     * <p>
     * Don't use this operation to update the data storage options of a project. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_UpdateProjectDataDelivery.html"
     * >UpdateProjectDataDelivery</a>.
     * </p>
     * <p>
     * Don't use this operation to update the tags of a project. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     *
     * @param updateProjectRequest
     * @return Result of the UpdateProject operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.UpdateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UpdateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateProjectResponse updateProject(UpdateProjectRequest updateProjectRequest) throws ValidationException,
            ConflictException, ServiceQuotaExceededException, ResourceNotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProject");

            return clientHandler.execute(new ClientExecutionParams<UpdateProjectRequest, UpdateProjectResponse>()
                    .withOperationName("UpdateProject").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateProjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the data storage options for this project. If you store evaluation events, you an keep them and analyze
     * them on your own. If you choose not to store evaluation events, Evidently deletes them after using them to
     * produce metrics and other experiment results that you can view.
     * </p>
     * <p>
     * You can't specify both <code>cloudWatchLogs</code> and <code>s3Destination</code> in the same operation.
     * </p>
     *
     * @param updateProjectDataDeliveryRequest
     * @return Result of the UpdateProjectDataDelivery operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EvidentlyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EvidentlyClient.UpdateProjectDataDelivery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UpdateProjectDataDelivery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateProjectDataDeliveryResponse updateProjectDataDelivery(
            UpdateProjectDataDeliveryRequest updateProjectDataDeliveryRequest) throws ValidationException, ConflictException,
            ServiceQuotaExceededException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, EvidentlyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateProjectDataDeliveryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateProjectDataDeliveryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProjectDataDeliveryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProjectDataDelivery");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateProjectDataDeliveryRequest, UpdateProjectDataDeliveryResponse>()
                            .withOperationName("UpdateProjectDataDelivery").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateProjectDataDeliveryRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateProjectDataDeliveryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EvidentlyException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build());
    }

    @Override
    public final EvidentlyServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
