/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * We are retiring EC2-Classic. We recommend that you migrate from EC2-Classic to a VPC. For more information, see <a
 * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/vpc-migrate.html">Migrate from EC2-Classic to a VPC</a> in
 * the <i>Amazon Elastic Compute Cloud User Guide</i>.
 * </p>
 * </note>
 * <p>
 * Describes the VPC peering connection options.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PeeringConnectionOptions implements SdkPojo, Serializable,
        ToCopyableBuilder<PeeringConnectionOptions.Builder, PeeringConnectionOptions> {
    private static final SdkField<Boolean> ALLOW_DNS_RESOLUTION_FROM_REMOTE_VPC_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AllowDnsResolutionFromRemoteVpc")
            .getter(getter(PeeringConnectionOptions::allowDnsResolutionFromRemoteVpc))
            .setter(setter(Builder::allowDnsResolutionFromRemoteVpc))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowDnsResolutionFromRemoteVpc")
                    .unmarshallLocationName("allowDnsResolutionFromRemoteVpc").build()).build();

    private static final SdkField<Boolean> ALLOW_EGRESS_FROM_LOCAL_CLASSIC_LINK_TO_REMOTE_VPC_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AllowEgressFromLocalClassicLinkToRemoteVpc")
            .getter(getter(PeeringConnectionOptions::allowEgressFromLocalClassicLinkToRemoteVpc))
            .setter(setter(Builder::allowEgressFromLocalClassicLinkToRemoteVpc))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("AllowEgressFromLocalClassicLinkToRemoteVpc")
                    .unmarshallLocationName("allowEgressFromLocalClassicLinkToRemoteVpc").build()).build();

    private static final SdkField<Boolean> ALLOW_EGRESS_FROM_LOCAL_VPC_TO_REMOTE_CLASSIC_LINK_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AllowEgressFromLocalVpcToRemoteClassicLink")
            .getter(getter(PeeringConnectionOptions::allowEgressFromLocalVpcToRemoteClassicLink))
            .setter(setter(Builder::allowEgressFromLocalVpcToRemoteClassicLink))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("AllowEgressFromLocalVpcToRemoteClassicLink")
                    .unmarshallLocationName("allowEgressFromLocalVpcToRemoteClassicLink").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            ALLOW_DNS_RESOLUTION_FROM_REMOTE_VPC_FIELD, ALLOW_EGRESS_FROM_LOCAL_CLASSIC_LINK_TO_REMOTE_VPC_FIELD,
            ALLOW_EGRESS_FROM_LOCAL_VPC_TO_REMOTE_CLASSIC_LINK_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean allowDnsResolutionFromRemoteVpc;

    private final Boolean allowEgressFromLocalClassicLinkToRemoteVpc;

    private final Boolean allowEgressFromLocalVpcToRemoteClassicLink;

    private PeeringConnectionOptions(BuilderImpl builder) {
        this.allowDnsResolutionFromRemoteVpc = builder.allowDnsResolutionFromRemoteVpc;
        this.allowEgressFromLocalClassicLinkToRemoteVpc = builder.allowEgressFromLocalClassicLinkToRemoteVpc;
        this.allowEgressFromLocalVpcToRemoteClassicLink = builder.allowEgressFromLocalVpcToRemoteClassicLink;
    }

    /**
     * <p>
     * If true, the public DNS hostnames of instances in the specified VPC resolve to private IP addresses when queried
     * from instances in the peer VPC.
     * </p>
     * 
     * @return If true, the public DNS hostnames of instances in the specified VPC resolve to private IP addresses when
     *         queried from instances in the peer VPC.
     */
    public final Boolean allowDnsResolutionFromRemoteVpc() {
        return allowDnsResolutionFromRemoteVpc;
    }

    /**
     * <p>
     * If true, enables outbound communication from an EC2-Classic instance that's linked to a local VPC using
     * ClassicLink to instances in a peer VPC.
     * </p>
     * 
     * @return If true, enables outbound communication from an EC2-Classic instance that's linked to a local VPC using
     *         ClassicLink to instances in a peer VPC.
     */
    public final Boolean allowEgressFromLocalClassicLinkToRemoteVpc() {
        return allowEgressFromLocalClassicLinkToRemoteVpc;
    }

    /**
     * <p>
     * If true, enables outbound communication from instances in a local VPC to an EC2-Classic instance that's linked to
     * a peer VPC using ClassicLink.
     * </p>
     * 
     * @return If true, enables outbound communication from instances in a local VPC to an EC2-Classic instance that's
     *         linked to a peer VPC using ClassicLink.
     */
    public final Boolean allowEgressFromLocalVpcToRemoteClassicLink() {
        return allowEgressFromLocalVpcToRemoteClassicLink;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(allowDnsResolutionFromRemoteVpc());
        hashCode = 31 * hashCode + Objects.hashCode(allowEgressFromLocalClassicLinkToRemoteVpc());
        hashCode = 31 * hashCode + Objects.hashCode(allowEgressFromLocalVpcToRemoteClassicLink());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PeeringConnectionOptions)) {
            return false;
        }
        PeeringConnectionOptions other = (PeeringConnectionOptions) obj;
        return Objects.equals(allowDnsResolutionFromRemoteVpc(), other.allowDnsResolutionFromRemoteVpc())
                && Objects.equals(allowEgressFromLocalClassicLinkToRemoteVpc(),
                        other.allowEgressFromLocalClassicLinkToRemoteVpc())
                && Objects.equals(allowEgressFromLocalVpcToRemoteClassicLink(),
                        other.allowEgressFromLocalVpcToRemoteClassicLink());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PeeringConnectionOptions")
                .add("AllowDnsResolutionFromRemoteVpc", allowDnsResolutionFromRemoteVpc())
                .add("AllowEgressFromLocalClassicLinkToRemoteVpc", allowEgressFromLocalClassicLinkToRemoteVpc())
                .add("AllowEgressFromLocalVpcToRemoteClassicLink", allowEgressFromLocalVpcToRemoteClassicLink()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AllowDnsResolutionFromRemoteVpc":
            return Optional.ofNullable(clazz.cast(allowDnsResolutionFromRemoteVpc()));
        case "AllowEgressFromLocalClassicLinkToRemoteVpc":
            return Optional.ofNullable(clazz.cast(allowEgressFromLocalClassicLinkToRemoteVpc()));
        case "AllowEgressFromLocalVpcToRemoteClassicLink":
            return Optional.ofNullable(clazz.cast(allowEgressFromLocalVpcToRemoteClassicLink()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PeeringConnectionOptions, T> g) {
        return obj -> g.apply((PeeringConnectionOptions) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PeeringConnectionOptions> {
        /**
         * <p>
         * If true, the public DNS hostnames of instances in the specified VPC resolve to private IP addresses when
         * queried from instances in the peer VPC.
         * </p>
         * 
         * @param allowDnsResolutionFromRemoteVpc
         *        If true, the public DNS hostnames of instances in the specified VPC resolve to private IP addresses
         *        when queried from instances in the peer VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowDnsResolutionFromRemoteVpc(Boolean allowDnsResolutionFromRemoteVpc);

        /**
         * <p>
         * If true, enables outbound communication from an EC2-Classic instance that's linked to a local VPC using
         * ClassicLink to instances in a peer VPC.
         * </p>
         * 
         * @param allowEgressFromLocalClassicLinkToRemoteVpc
         *        If true, enables outbound communication from an EC2-Classic instance that's linked to a local VPC
         *        using ClassicLink to instances in a peer VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowEgressFromLocalClassicLinkToRemoteVpc(Boolean allowEgressFromLocalClassicLinkToRemoteVpc);

        /**
         * <p>
         * If true, enables outbound communication from instances in a local VPC to an EC2-Classic instance that's
         * linked to a peer VPC using ClassicLink.
         * </p>
         * 
         * @param allowEgressFromLocalVpcToRemoteClassicLink
         *        If true, enables outbound communication from instances in a local VPC to an EC2-Classic instance
         *        that's linked to a peer VPC using ClassicLink.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowEgressFromLocalVpcToRemoteClassicLink(Boolean allowEgressFromLocalVpcToRemoteClassicLink);
    }

    static final class BuilderImpl implements Builder {
        private Boolean allowDnsResolutionFromRemoteVpc;

        private Boolean allowEgressFromLocalClassicLinkToRemoteVpc;

        private Boolean allowEgressFromLocalVpcToRemoteClassicLink;

        private BuilderImpl() {
        }

        private BuilderImpl(PeeringConnectionOptions model) {
            allowDnsResolutionFromRemoteVpc(model.allowDnsResolutionFromRemoteVpc);
            allowEgressFromLocalClassicLinkToRemoteVpc(model.allowEgressFromLocalClassicLinkToRemoteVpc);
            allowEgressFromLocalVpcToRemoteClassicLink(model.allowEgressFromLocalVpcToRemoteClassicLink);
        }

        public final Boolean getAllowDnsResolutionFromRemoteVpc() {
            return allowDnsResolutionFromRemoteVpc;
        }

        public final void setAllowDnsResolutionFromRemoteVpc(Boolean allowDnsResolutionFromRemoteVpc) {
            this.allowDnsResolutionFromRemoteVpc = allowDnsResolutionFromRemoteVpc;
        }

        @Override
        public final Builder allowDnsResolutionFromRemoteVpc(Boolean allowDnsResolutionFromRemoteVpc) {
            this.allowDnsResolutionFromRemoteVpc = allowDnsResolutionFromRemoteVpc;
            return this;
        }

        public final Boolean getAllowEgressFromLocalClassicLinkToRemoteVpc() {
            return allowEgressFromLocalClassicLinkToRemoteVpc;
        }

        public final void setAllowEgressFromLocalClassicLinkToRemoteVpc(Boolean allowEgressFromLocalClassicLinkToRemoteVpc) {
            this.allowEgressFromLocalClassicLinkToRemoteVpc = allowEgressFromLocalClassicLinkToRemoteVpc;
        }

        @Override
        public final Builder allowEgressFromLocalClassicLinkToRemoteVpc(Boolean allowEgressFromLocalClassicLinkToRemoteVpc) {
            this.allowEgressFromLocalClassicLinkToRemoteVpc = allowEgressFromLocalClassicLinkToRemoteVpc;
            return this;
        }

        public final Boolean getAllowEgressFromLocalVpcToRemoteClassicLink() {
            return allowEgressFromLocalVpcToRemoteClassicLink;
        }

        public final void setAllowEgressFromLocalVpcToRemoteClassicLink(Boolean allowEgressFromLocalVpcToRemoteClassicLink) {
            this.allowEgressFromLocalVpcToRemoteClassicLink = allowEgressFromLocalVpcToRemoteClassicLink;
        }

        @Override
        public final Builder allowEgressFromLocalVpcToRemoteClassicLink(Boolean allowEgressFromLocalVpcToRemoteClassicLink) {
            this.allowEgressFromLocalVpcToRemoteClassicLink = allowEgressFromLocalVpcToRemoteClassicLink;
            return this;
        }

        @Override
        public PeeringConnectionOptions build() {
            return new PeeringConnectionOptions(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
