/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chime.endpoints.internal;

import java.util.Optional;
import java.util.concurrent.CompletionException;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsExecutionAttribute;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.interceptor.Context;
import software.amazon.awssdk.core.interceptor.ExecutionAttributes;
import software.amazon.awssdk.core.interceptor.ExecutionInterceptor;
import software.amazon.awssdk.core.interceptor.SdkExecutionAttribute;
import software.amazon.awssdk.core.interceptor.SdkInternalExecutionAttribute;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.services.chime.endpoints.ChimeEndpointParams;
import software.amazon.awssdk.services.chime.endpoints.ChimeEndpointProvider;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class ChimeResolveEndpointInterceptor implements ExecutionInterceptor {
    @Override
    public SdkRequest modifyRequest(Context.ModifyRequest context, ExecutionAttributes executionAttributes) {
        if (AwsEndpointProviderUtils.endpointIsDiscovered(executionAttributes)) {
            return context.request();
        }
        ChimeEndpointProvider provider = (ChimeEndpointProvider) executionAttributes
                .getAttribute(SdkInternalExecutionAttribute.ENDPOINT_PROVIDER);
        try {
            Endpoint result = provider.resolveEndpoint(ruleParams(context, executionAttributes)).join();
            if (!AwsEndpointProviderUtils.disableHostPrefixInjection(executionAttributes)) {
                Optional<String> hostPrefix = hostPrefix(executionAttributes.getAttribute(SdkExecutionAttribute.OPERATION_NAME),
                        context.request());
                if (hostPrefix.isPresent()) {
                    result = AwsEndpointProviderUtils.addHostPrefix(result, hostPrefix.get());
                }
            }
            executionAttributes.putAttribute(SdkInternalExecutionAttribute.RESOLVED_ENDPOINT, result);
            return context.request();
        } catch (CompletionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof SdkClientException) {
                throw (SdkClientException) cause;
            } else {
                throw SdkClientException.create("Endpoint resolution failed", cause);
            }
        }
    }

    private static ChimeEndpointParams ruleParams(Context.ModifyRequest context, ExecutionAttributes executionAttributes) {
        ChimeEndpointParams.Builder builder = ChimeEndpointParams.builder();
        builder.region(AwsEndpointProviderUtils.regionBuiltIn(executionAttributes));
        builder.useDualStack(AwsEndpointProviderUtils.dualStackEnabledBuiltIn(executionAttributes));
        builder.useFips(AwsEndpointProviderUtils.fipsEnabledBuiltIn(executionAttributes));
        builder.endpoint(AwsEndpointProviderUtils.endpointBuiltIn(executionAttributes));
        setContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME), context.request());
        setStaticContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME));
        return builder.build();
    }

    private static void setContextParams(ChimeEndpointParams.Builder params, String operationName, SdkRequest request) {
    }

    private static void setStaticContextParams(ChimeEndpointParams.Builder params, String operationName) {
    }

    private static Optional<String> hostPrefix(String operationName, SdkRequest request) {
        switch (operationName) {
        case "BatchCreateChannelMembership": {
            return Optional.of("messaging-");
        }
        case "CreateAppInstance": {
            return Optional.of("identity-");
        }
        case "CreateAppInstanceAdmin": {
            return Optional.of("identity-");
        }
        case "CreateAppInstanceUser": {
            return Optional.of("identity-");
        }
        case "CreateChannel": {
            return Optional.of("messaging-");
        }
        case "CreateChannelBan": {
            return Optional.of("messaging-");
        }
        case "CreateChannelMembership": {
            return Optional.of("messaging-");
        }
        case "CreateChannelModerator": {
            return Optional.of("messaging-");
        }
        case "DeleteAppInstance": {
            return Optional.of("identity-");
        }
        case "DeleteAppInstanceAdmin": {
            return Optional.of("identity-");
        }
        case "DeleteAppInstanceUser": {
            return Optional.of("identity-");
        }
        case "DeleteChannel": {
            return Optional.of("messaging-");
        }
        case "DeleteChannelBan": {
            return Optional.of("messaging-");
        }
        case "DeleteChannelMembership": {
            return Optional.of("messaging-");
        }
        case "DeleteChannelMessage": {
            return Optional.of("messaging-");
        }
        case "DeleteChannelModerator": {
            return Optional.of("messaging-");
        }
        case "DescribeAppInstance": {
            return Optional.of("identity-");
        }
        case "DescribeAppInstanceAdmin": {
            return Optional.of("identity-");
        }
        case "DescribeAppInstanceUser": {
            return Optional.of("identity-");
        }
        case "DescribeChannel": {
            return Optional.of("messaging-");
        }
        case "DescribeChannelBan": {
            return Optional.of("messaging-");
        }
        case "DescribeChannelMembership": {
            return Optional.of("messaging-");
        }
        case "DescribeChannelMembershipForAppInstanceUser": {
            return Optional.of("messaging-");
        }
        case "DescribeChannelModeratedByAppInstanceUser": {
            return Optional.of("messaging-");
        }
        case "DescribeChannelModerator": {
            return Optional.of("messaging-");
        }
        case "GetAppInstanceRetentionSettings": {
            return Optional.of("identity-");
        }
        case "GetChannelMessage": {
            return Optional.of("messaging-");
        }
        case "GetMessagingSessionEndpoint": {
            return Optional.of("messaging-");
        }
        case "ListAppInstanceAdmins": {
            return Optional.of("identity-");
        }
        case "ListAppInstanceUsers": {
            return Optional.of("identity-");
        }
        case "ListAppInstances": {
            return Optional.of("identity-");
        }
        case "ListChannelBans": {
            return Optional.of("messaging-");
        }
        case "ListChannelMemberships": {
            return Optional.of("messaging-");
        }
        case "ListChannelMembershipsForAppInstanceUser": {
            return Optional.of("messaging-");
        }
        case "ListChannelMessages": {
            return Optional.of("messaging-");
        }
        case "ListChannelModerators": {
            return Optional.of("messaging-");
        }
        case "ListChannels": {
            return Optional.of("messaging-");
        }
        case "ListChannelsModeratedByAppInstanceUser": {
            return Optional.of("messaging-");
        }
        case "PutAppInstanceRetentionSettings": {
            return Optional.of("identity-");
        }
        case "RedactChannelMessage": {
            return Optional.of("messaging-");
        }
        case "SendChannelMessage": {
            return Optional.of("messaging-");
        }
        case "UpdateAppInstance": {
            return Optional.of("identity-");
        }
        case "UpdateAppInstanceUser": {
            return Optional.of("identity-");
        }
        case "UpdateChannel": {
            return Optional.of("messaging-");
        }
        case "UpdateChannelMessage": {
            return Optional.of("messaging-");
        }
        case "UpdateChannelReadMarker": {
            return Optional.of("messaging-");
        }
        default:
            return Optional.empty();
        }
    }
}
