/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The type and amount of a resource to assign to a container. The supported resources include <code>GPU</code>,
 * <code>MEMORY</code>, and <code>VCPU</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceRequirement implements SdkPojo, Serializable,
        ToCopyableBuilder<ResourceRequirement.Builder, ResourceRequirement> {
    private static final SdkField<String> VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("value")
            .getter(getter(ResourceRequirement::value)).setter(setter(Builder::value))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("value").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(ResourceRequirement::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VALUE_FIELD, TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String value;

    private final String type;

    private ResourceRequirement(BuilderImpl builder) {
        this.value = builder.value;
        this.type = builder.type;
    }

    /**
     * <p>
     * The quantity of the specified resource to reserve for the container. The values vary based on the
     * <code>type</code> specified.
     * </p>
     * <dl>
     * <dt>type="GPU"</dt>
     * <dd>
     * <p>
     * The number of physical GPUs to reserve for the container. Make sure that the number of GPUs reserved for all
     * containers in a job doesn't exceed the number of available GPUs on the compute resource that the job is launched
     * on.
     * </p>
     * <note>
     * <p>
     * GPUs aren't available for jobs that are running on Fargate resources.
     * </p>
     * </note></dd>
     * <dt>type="MEMORY"</dt>
     * <dd>
     * <p>
     * The memory hard limit (in MiB) present to the container. This parameter is supported for jobs that are running on
     * EC2 resources. If your container attempts to exceed the memory specified, the container is terminated. This
     * parameter maps to <code>Memory</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
     * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--memory</code> option to <a
     * href="https://docs.docker.com/engine/reference/run/">docker run</a>. You must specify at least 4 MiB of memory
     * for a job. This is required but can be specified in several places for multi-node parallel (MNP) jobs. It must be
     * specified for each node at least once. This parameter maps to <code>Memory</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
     * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--memory</code> option to <a
     * href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     * </p>
     * <note>
     * <p>
     * If you're trying to maximize your resource utilization by providing your jobs as much memory as possible for a
     * particular instance type, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/memory-management.html">Memory management</a> in the
     * <i>Batch User Guide</i>.
     * </p>
     * </note>
     * <p>
     * For jobs that are running on Fargate resources, then <code>value</code> is the hard limit (in MiB), and must
     * match one of the supported values and the <code>VCPU</code> values must be one of the values supported for that
     * memory value.
     * </p>
     * <dl>
     * <dt>value = 512</dt>
     * <dd>
     * <p>
     * <code>VCPU</code> = 0.25
     * </p>
     * </dd>
     * <dt>value = 1024</dt>
     * <dd>
     * <p>
     * <code>VCPU</code> = 0.25 or 0.5
     * </p>
     * </dd>
     * <dt>value = 2048</dt>
     * <dd>
     * <p>
     * <code>VCPU</code> = 0.25, 0.5, or 1
     * </p>
     * </dd>
     * <dt>value = 3072</dt>
     * <dd>
     * <p>
     * <code>VCPU</code> = 0.5, or 1
     * </p>
     * </dd>
     * <dt>value = 4096</dt>
     * <dd>
     * <p>
     * <code>VCPU</code> = 0.5, 1, or 2
     * </p>
     * </dd>
     * <dt>value = 5120, 6144, or 7168</dt>
     * <dd>
     * <p>
     * <code>VCPU</code> = 1 or 2
     * </p>
     * </dd>
     * <dt>value = 8192</dt>
     * <dd>
     * <p>
     * <code>VCPU</code> = 1, 2, or 4
     * </p>
     * </dd>
     * <dt>value = 9216, 10240, 11264, 12288, 13312, 14336, or 15360</dt>
     * <dd>
     * <p>
     * <code>VCPU</code> = 2 or 4
     * </p>
     * </dd>
     * <dt>value = 16384</dt>
     * <dd>
     * <p>
     * <code>VCPU</code> = 2, 4, or 8
     * </p>
     * </dd>
     * <dt>value = 17408, 18432, 19456, 21504, 22528, 23552, 25600, 26624, 27648, 29696, or 30720</dt>
     * <dd>
     * <p>
     * <code>VCPU</code> = 4
     * </p>
     * </dd>
     * <dt>value = 20480, 24576, or 28672</dt>
     * <dd>
     * <p>
     * <code>VCPU</code> = 4 or 8
     * </p>
     * </dd>
     * <dt>value = 36864, 45056, 53248, or 61440</dt>
     * <dd>
     * <p>
     * <code>VCPU</code> = 8
     * </p>
     * </dd>
     * <dt>value = 32768, 40960, 49152, or 57344</dt>
     * <dd>
     * <p>
     * <code>VCPU</code> = 8 or 16
     * </p>
     * </dd>
     * <dt>value = 65536, 73728, 81920, 90112, 98304, 106496, 114688, or 122880</dt>
     * <dd>
     * <p>
     * <code>VCPU</code> = 16
     * </p>
     * </dd>
     * </dl>
     * </dd>
     * <dt>type="VCPU"</dt>
     * <dd>
     * <p>
     * The number of vCPUs reserved for the container. This parameter maps to <code>CpuShares</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
     * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--cpu-shares</code> option
     * to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. Each vCPU is equivalent to 1,024 CPU
     * shares. For EC2 resources, you must specify at least one vCPU. This is required but can be specified in several
     * places; it must be specified for each node at least once.
     * </p>
     * <p>
     * The default for the Fargate On-Demand vCPU resource count quota is 6 vCPUs. For more information about Fargate
     * quotas, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/ecs-service.html#service-quotas-fargate">Fargate quotas</a>
     * in the <i>Amazon Web Services General Reference</i>.
     * </p>
     * <p>
     * For jobs that are running on Fargate resources, then <code>value</code> must match one of the supported values
     * and the <code>MEMORY</code> values must be one of the values supported for that <code>VCPU</code> value. The
     * supported values are 0.25, 0.5, 1, 2, 4, 8, and 16
     * </p>
     * <dl>
     * <dt>value = 0.25</dt>
     * <dd>
     * <p>
     * <code>MEMORY</code> = 512, 1024, or 2048
     * </p>
     * </dd>
     * <dt>value = 0.5</dt>
     * <dd>
     * <p>
     * <code>MEMORY</code> = 1024, 2048, 3072, or 4096
     * </p>
     * </dd>
     * <dt>value = 1</dt>
     * <dd>
     * <p>
     * <code>MEMORY</code> = 2048, 3072, 4096, 5120, 6144, 7168, or 8192
     * </p>
     * </dd>
     * <dt>value = 2</dt>
     * <dd>
     * <p>
     * <code>MEMORY</code> = 4096, 5120, 6144, 7168, 8192, 9216, 10240, 11264, 12288, 13312, 14336, 15360, or 16384
     * </p>
     * </dd>
     * <dt>value = 4</dt>
     * <dd>
     * <p>
     * <code>MEMORY</code> = 8192, 9216, 10240, 11264, 12288, 13312, 14336, 15360, 16384, 17408, 18432, 19456, 20480,
     * 21504, 22528, 23552, 24576, 25600, 26624, 27648, 28672, 29696, or 30720
     * </p>
     * </dd>
     * <dt>value = 8</dt>
     * <dd>
     * <p>
     * <code>MEMORY</code> = 16384, 20480, 24576, 28672, 32768, 36864, 40960, 45056, 49152, 53248, 57344, or 61440
     * </p>
     * </dd>
     * <dt>value = 16</dt>
     * <dd>
     * <p>
     * <code>MEMORY</code> = 32768, 40960, 49152, 57344, 65536, 73728, 81920, 90112, 98304, 106496, 114688, or 122880
     * </p>
     * </dd>
     * </dl>
     * </dd>
     * </dl>
     * 
     * @return The quantity of the specified resource to reserve for the container. The values vary based on the
     *         <code>type</code> specified.</p>
     *         <dl>
     *         <dt>type="GPU"</dt>
     *         <dd>
     *         <p>
     *         The number of physical GPUs to reserve for the container. Make sure that the number of GPUs reserved for
     *         all containers in a job doesn't exceed the number of available GPUs on the compute resource that the job
     *         is launched on.
     *         </p>
     *         <note>
     *         <p>
     *         GPUs aren't available for jobs that are running on Fargate resources.
     *         </p>
     *         </note></dd>
     *         <dt>type="MEMORY"</dt>
     *         <dd>
     *         <p>
     *         The memory hard limit (in MiB) present to the container. This parameter is supported for jobs that are
     *         running on EC2 resources. If your container attempts to exceed the memory specified, the container is
     *         terminated. This parameter maps to <code>Memory</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the
     *         <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--memory</code>
     *         option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. You must specify at
     *         least 4 MiB of memory for a job. This is required but can be specified in several places for multi-node
     *         parallel (MNP) jobs. It must be specified for each node at least once. This parameter maps to
     *         <code>Memory</code> in the <a href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create
     *         a container</a> section of the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a>
     *         and the <code>--memory</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a>.
     *         </p>
     *         <note>
     *         <p>
     *         If you're trying to maximize your resource utilization by providing your jobs as much memory as possible
     *         for a particular instance type, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/memory-management.html">Memory management</a> in
     *         the <i>Batch User Guide</i>.
     *         </p>
     *         </note>
     *         <p>
     *         For jobs that are running on Fargate resources, then <code>value</code> is the hard limit (in MiB), and
     *         must match one of the supported values and the <code>VCPU</code> values must be one of the values
     *         supported for that memory value.
     *         </p>
     *         <dl>
     *         <dt>value = 512</dt>
     *         <dd>
     *         <p>
     *         <code>VCPU</code> = 0.25
     *         </p>
     *         </dd>
     *         <dt>value = 1024</dt>
     *         <dd>
     *         <p>
     *         <code>VCPU</code> = 0.25 or 0.5
     *         </p>
     *         </dd>
     *         <dt>value = 2048</dt>
     *         <dd>
     *         <p>
     *         <code>VCPU</code> = 0.25, 0.5, or 1
     *         </p>
     *         </dd>
     *         <dt>value = 3072</dt>
     *         <dd>
     *         <p>
     *         <code>VCPU</code> = 0.5, or 1
     *         </p>
     *         </dd>
     *         <dt>value = 4096</dt>
     *         <dd>
     *         <p>
     *         <code>VCPU</code> = 0.5, 1, or 2
     *         </p>
     *         </dd>
     *         <dt>value = 5120, 6144, or 7168</dt>
     *         <dd>
     *         <p>
     *         <code>VCPU</code> = 1 or 2
     *         </p>
     *         </dd>
     *         <dt>value = 8192</dt>
     *         <dd>
     *         <p>
     *         <code>VCPU</code> = 1, 2, or 4
     *         </p>
     *         </dd>
     *         <dt>value = 9216, 10240, 11264, 12288, 13312, 14336, or 15360</dt>
     *         <dd>
     *         <p>
     *         <code>VCPU</code> = 2 or 4
     *         </p>
     *         </dd>
     *         <dt>value = 16384</dt>
     *         <dd>
     *         <p>
     *         <code>VCPU</code> = 2, 4, or 8
     *         </p>
     *         </dd>
     *         <dt>value = 17408, 18432, 19456, 21504, 22528, 23552, 25600, 26624, 27648, 29696, or 30720</dt>
     *         <dd>
     *         <p>
     *         <code>VCPU</code> = 4
     *         </p>
     *         </dd>
     *         <dt>value = 20480, 24576, or 28672</dt>
     *         <dd>
     *         <p>
     *         <code>VCPU</code> = 4 or 8
     *         </p>
     *         </dd>
     *         <dt>value = 36864, 45056, 53248, or 61440</dt>
     *         <dd>
     *         <p>
     *         <code>VCPU</code> = 8
     *         </p>
     *         </dd>
     *         <dt>value = 32768, 40960, 49152, or 57344</dt>
     *         <dd>
     *         <p>
     *         <code>VCPU</code> = 8 or 16
     *         </p>
     *         </dd>
     *         <dt>value = 65536, 73728, 81920, 90112, 98304, 106496, 114688, or 122880</dt>
     *         <dd>
     *         <p>
     *         <code>VCPU</code> = 16
     *         </p>
     *         </dd>
     *         </dl>
     *         </dd>
     *         <dt>type="VCPU"</dt>
     *         <dd>
     *         <p>
     *         The number of vCPUs reserved for the container. This parameter maps to <code>CpuShares</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the
     *         <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
     *         <code>--cpu-shares</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a>. Each vCPU is equivalent to 1,024 CPU shares. For EC2 resources, you must specify at least one
     *         vCPU. This is required but can be specified in several places; it must be specified for each node at
     *         least once.
     *         </p>
     *         <p>
     *         The default for the Fargate On-Demand vCPU resource count quota is 6 vCPUs. For more information about
     *         Fargate quotas, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/ecs-service.html#service-quotas-fargate">Fargate
     *         quotas</a> in the <i>Amazon Web Services General Reference</i>.
     *         </p>
     *         <p>
     *         For jobs that are running on Fargate resources, then <code>value</code> must match one of the supported
     *         values and the <code>MEMORY</code> values must be one of the values supported for that <code>VCPU</code>
     *         value. The supported values are 0.25, 0.5, 1, 2, 4, 8, and 16
     *         </p>
     *         <dl>
     *         <dt>value = 0.25</dt>
     *         <dd>
     *         <p>
     *         <code>MEMORY</code> = 512, 1024, or 2048
     *         </p>
     *         </dd>
     *         <dt>value = 0.5</dt>
     *         <dd>
     *         <p>
     *         <code>MEMORY</code> = 1024, 2048, 3072, or 4096
     *         </p>
     *         </dd>
     *         <dt>value = 1</dt>
     *         <dd>
     *         <p>
     *         <code>MEMORY</code> = 2048, 3072, 4096, 5120, 6144, 7168, or 8192
     *         </p>
     *         </dd>
     *         <dt>value = 2</dt>
     *         <dd>
     *         <p>
     *         <code>MEMORY</code> = 4096, 5120, 6144, 7168, 8192, 9216, 10240, 11264, 12288, 13312, 14336, 15360, or
     *         16384
     *         </p>
     *         </dd>
     *         <dt>value = 4</dt>
     *         <dd>
     *         <p>
     *         <code>MEMORY</code> = 8192, 9216, 10240, 11264, 12288, 13312, 14336, 15360, 16384, 17408, 18432, 19456,
     *         20480, 21504, 22528, 23552, 24576, 25600, 26624, 27648, 28672, 29696, or 30720
     *         </p>
     *         </dd>
     *         <dt>value = 8</dt>
     *         <dd>
     *         <p>
     *         <code>MEMORY</code> = 16384, 20480, 24576, 28672, 32768, 36864, 40960, 45056, 49152, 53248, 57344, or
     *         61440
     *         </p>
     *         </dd>
     *         <dt>value = 16</dt>
     *         <dd>
     *         <p>
     *         <code>MEMORY</code> = 32768, 40960, 49152, 57344, 65536, 73728, 81920, 90112, 98304, 106496, 114688, or
     *         122880
     *         </p>
     *         </dd>
     *         </dl>
     *         </dd>
     */
    public final String value() {
        return value;
    }

    /**
     * <p>
     * The type of resource to assign to a container. The supported resources include <code>GPU</code>,
     * <code>MEMORY</code>, and <code>VCPU</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of resource to assign to a container. The supported resources include <code>GPU</code>,
     *         <code>MEMORY</code>, and <code>VCPU</code>.
     * @see ResourceType
     */
    public final ResourceType type() {
        return ResourceType.fromValue(type);
    }

    /**
     * <p>
     * The type of resource to assign to a container. The supported resources include <code>GPU</code>,
     * <code>MEMORY</code>, and <code>VCPU</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of resource to assign to a container. The supported resources include <code>GPU</code>,
     *         <code>MEMORY</code>, and <code>VCPU</code>.
     * @see ResourceType
     */
    public final String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(value());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceRequirement)) {
            return false;
        }
        ResourceRequirement other = (ResourceRequirement) obj;
        return Objects.equals(value(), other.value()) && Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResourceRequirement").add("Value", value()).add("Type", typeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "value":
            return Optional.ofNullable(clazz.cast(value()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResourceRequirement, T> g) {
        return obj -> g.apply((ResourceRequirement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceRequirement> {
        /**
         * <p>
         * The quantity of the specified resource to reserve for the container. The values vary based on the
         * <code>type</code> specified.
         * </p>
         * <dl>
         * <dt>type="GPU"</dt>
         * <dd>
         * <p>
         * The number of physical GPUs to reserve for the container. Make sure that the number of GPUs reserved for all
         * containers in a job doesn't exceed the number of available GPUs on the compute resource that the job is
         * launched on.
         * </p>
         * <note>
         * <p>
         * GPUs aren't available for jobs that are running on Fargate resources.
         * </p>
         * </note></dd>
         * <dt>type="MEMORY"</dt>
         * <dd>
         * <p>
         * The memory hard limit (in MiB) present to the container. This parameter is supported for jobs that are
         * running on EC2 resources. If your container attempts to exceed the memory specified, the container is
         * terminated. This parameter maps to <code>Memory</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--memory</code> option
         * to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. You must specify at least 4 MiB of
         * memory for a job. This is required but can be specified in several places for multi-node parallel (MNP) jobs.
         * It must be specified for each node at least once. This parameter maps to <code>Memory</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--memory</code> option
         * to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <note>
         * <p>
         * If you're trying to maximize your resource utilization by providing your jobs as much memory as possible for
         * a particular instance type, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/memory-management.html">Memory management</a> in the
         * <i>Batch User Guide</i>.
         * </p>
         * </note>
         * <p>
         * For jobs that are running on Fargate resources, then <code>value</code> is the hard limit (in MiB), and must
         * match one of the supported values and the <code>VCPU</code> values must be one of the values supported for
         * that memory value.
         * </p>
         * <dl>
         * <dt>value = 512</dt>
         * <dd>
         * <p>
         * <code>VCPU</code> = 0.25
         * </p>
         * </dd>
         * <dt>value = 1024</dt>
         * <dd>
         * <p>
         * <code>VCPU</code> = 0.25 or 0.5
         * </p>
         * </dd>
         * <dt>value = 2048</dt>
         * <dd>
         * <p>
         * <code>VCPU</code> = 0.25, 0.5, or 1
         * </p>
         * </dd>
         * <dt>value = 3072</dt>
         * <dd>
         * <p>
         * <code>VCPU</code> = 0.5, or 1
         * </p>
         * </dd>
         * <dt>value = 4096</dt>
         * <dd>
         * <p>
         * <code>VCPU</code> = 0.5, 1, or 2
         * </p>
         * </dd>
         * <dt>value = 5120, 6144, or 7168</dt>
         * <dd>
         * <p>
         * <code>VCPU</code> = 1 or 2
         * </p>
         * </dd>
         * <dt>value = 8192</dt>
         * <dd>
         * <p>
         * <code>VCPU</code> = 1, 2, or 4
         * </p>
         * </dd>
         * <dt>value = 9216, 10240, 11264, 12288, 13312, 14336, or 15360</dt>
         * <dd>
         * <p>
         * <code>VCPU</code> = 2 or 4
         * </p>
         * </dd>
         * <dt>value = 16384</dt>
         * <dd>
         * <p>
         * <code>VCPU</code> = 2, 4, or 8
         * </p>
         * </dd>
         * <dt>value = 17408, 18432, 19456, 21504, 22528, 23552, 25600, 26624, 27648, 29696, or 30720</dt>
         * <dd>
         * <p>
         * <code>VCPU</code> = 4
         * </p>
         * </dd>
         * <dt>value = 20480, 24576, or 28672</dt>
         * <dd>
         * <p>
         * <code>VCPU</code> = 4 or 8
         * </p>
         * </dd>
         * <dt>value = 36864, 45056, 53248, or 61440</dt>
         * <dd>
         * <p>
         * <code>VCPU</code> = 8
         * </p>
         * </dd>
         * <dt>value = 32768, 40960, 49152, or 57344</dt>
         * <dd>
         * <p>
         * <code>VCPU</code> = 8 or 16
         * </p>
         * </dd>
         * <dt>value = 65536, 73728, 81920, 90112, 98304, 106496, 114688, or 122880</dt>
         * <dd>
         * <p>
         * <code>VCPU</code> = 16
         * </p>
         * </dd>
         * </dl>
         * </dd>
         * <dt>type="VCPU"</dt>
         * <dd>
         * <p>
         * The number of vCPUs reserved for the container. This parameter maps to <code>CpuShares</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--cpu-shares</code>
         * option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. Each vCPU is equivalent to
         * 1,024 CPU shares. For EC2 resources, you must specify at least one vCPU. This is required but can be
         * specified in several places; it must be specified for each node at least once.
         * </p>
         * <p>
         * The default for the Fargate On-Demand vCPU resource count quota is 6 vCPUs. For more information about
         * Fargate quotas, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/ecs-service.html#service-quotas-fargate">Fargate
         * quotas</a> in the <i>Amazon Web Services General Reference</i>.
         * </p>
         * <p>
         * For jobs that are running on Fargate resources, then <code>value</code> must match one of the supported
         * values and the <code>MEMORY</code> values must be one of the values supported for that <code>VCPU</code>
         * value. The supported values are 0.25, 0.5, 1, 2, 4, 8, and 16
         * </p>
         * <dl>
         * <dt>value = 0.25</dt>
         * <dd>
         * <p>
         * <code>MEMORY</code> = 512, 1024, or 2048
         * </p>
         * </dd>
         * <dt>value = 0.5</dt>
         * <dd>
         * <p>
         * <code>MEMORY</code> = 1024, 2048, 3072, or 4096
         * </p>
         * </dd>
         * <dt>value = 1</dt>
         * <dd>
         * <p>
         * <code>MEMORY</code> = 2048, 3072, 4096, 5120, 6144, 7168, or 8192
         * </p>
         * </dd>
         * <dt>value = 2</dt>
         * <dd>
         * <p>
         * <code>MEMORY</code> = 4096, 5120, 6144, 7168, 8192, 9216, 10240, 11264, 12288, 13312, 14336, 15360, or 16384
         * </p>
         * </dd>
         * <dt>value = 4</dt>
         * <dd>
         * <p>
         * <code>MEMORY</code> = 8192, 9216, 10240, 11264, 12288, 13312, 14336, 15360, 16384, 17408, 18432, 19456,
         * 20480, 21504, 22528, 23552, 24576, 25600, 26624, 27648, 28672, 29696, or 30720
         * </p>
         * </dd>
         * <dt>value = 8</dt>
         * <dd>
         * <p>
         * <code>MEMORY</code> = 16384, 20480, 24576, 28672, 32768, 36864, 40960, 45056, 49152, 53248, 57344, or 61440
         * </p>
         * </dd>
         * <dt>value = 16</dt>
         * <dd>
         * <p>
         * <code>MEMORY</code> = 32768, 40960, 49152, 57344, 65536, 73728, 81920, 90112, 98304, 106496, 114688, or
         * 122880
         * </p>
         * </dd>
         * </dl>
         * </dd>
         * </dl>
         * 
         * @param value
         *        The quantity of the specified resource to reserve for the container. The values vary based on the
         *        <code>type</code> specified.</p>
         *        <dl>
         *        <dt>type="GPU"</dt>
         *        <dd>
         *        <p>
         *        The number of physical GPUs to reserve for the container. Make sure that the number of GPUs reserved
         *        for all containers in a job doesn't exceed the number of available GPUs on the compute resource that
         *        the job is launched on.
         *        </p>
         *        <note>
         *        <p>
         *        GPUs aren't available for jobs that are running on Fargate resources.
         *        </p>
         *        </note></dd>
         *        <dt>type="MEMORY"</dt>
         *        <dd>
         *        <p>
         *        The memory hard limit (in MiB) present to the container. This parameter is supported for jobs that are
         *        running on EC2 resources. If your container attempts to exceed the memory specified, the container is
         *        terminated. This parameter maps to <code>Memory</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>--memory</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>. You must specify at least 4 MiB of memory for a job. This is required but can be specified in
         *        several places for multi-node parallel (MNP) jobs. It must be specified for each node at least once.
         *        This parameter maps to <code>Memory</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>--memory</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.
         *        </p>
         *        <note>
         *        <p>
         *        If you're trying to maximize your resource utilization by providing your jobs as much memory as
         *        possible for a particular instance type, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/memory-management.html">Memory management</a>
         *        in the <i>Batch User Guide</i>.
         *        </p>
         *        </note>
         *        <p>
         *        For jobs that are running on Fargate resources, then <code>value</code> is the hard limit (in MiB),
         *        and must match one of the supported values and the <code>VCPU</code> values must be one of the values
         *        supported for that memory value.
         *        </p>
         *        <dl>
         *        <dt>value = 512</dt>
         *        <dd>
         *        <p>
         *        <code>VCPU</code> = 0.25
         *        </p>
         *        </dd>
         *        <dt>value = 1024</dt>
         *        <dd>
         *        <p>
         *        <code>VCPU</code> = 0.25 or 0.5
         *        </p>
         *        </dd>
         *        <dt>value = 2048</dt>
         *        <dd>
         *        <p>
         *        <code>VCPU</code> = 0.25, 0.5, or 1
         *        </p>
         *        </dd>
         *        <dt>value = 3072</dt>
         *        <dd>
         *        <p>
         *        <code>VCPU</code> = 0.5, or 1
         *        </p>
         *        </dd>
         *        <dt>value = 4096</dt>
         *        <dd>
         *        <p>
         *        <code>VCPU</code> = 0.5, 1, or 2
         *        </p>
         *        </dd>
         *        <dt>value = 5120, 6144, or 7168</dt>
         *        <dd>
         *        <p>
         *        <code>VCPU</code> = 1 or 2
         *        </p>
         *        </dd>
         *        <dt>value = 8192</dt>
         *        <dd>
         *        <p>
         *        <code>VCPU</code> = 1, 2, or 4
         *        </p>
         *        </dd>
         *        <dt>value = 9216, 10240, 11264, 12288, 13312, 14336, or 15360</dt>
         *        <dd>
         *        <p>
         *        <code>VCPU</code> = 2 or 4
         *        </p>
         *        </dd>
         *        <dt>value = 16384</dt>
         *        <dd>
         *        <p>
         *        <code>VCPU</code> = 2, 4, or 8
         *        </p>
         *        </dd>
         *        <dt>value = 17408, 18432, 19456, 21504, 22528, 23552, 25600, 26624, 27648, 29696, or 30720</dt>
         *        <dd>
         *        <p>
         *        <code>VCPU</code> = 4
         *        </p>
         *        </dd>
         *        <dt>value = 20480, 24576, or 28672</dt>
         *        <dd>
         *        <p>
         *        <code>VCPU</code> = 4 or 8
         *        </p>
         *        </dd>
         *        <dt>value = 36864, 45056, 53248, or 61440</dt>
         *        <dd>
         *        <p>
         *        <code>VCPU</code> = 8
         *        </p>
         *        </dd>
         *        <dt>value = 32768, 40960, 49152, or 57344</dt>
         *        <dd>
         *        <p>
         *        <code>VCPU</code> = 8 or 16
         *        </p>
         *        </dd>
         *        <dt>value = 65536, 73728, 81920, 90112, 98304, 106496, 114688, or 122880</dt>
         *        <dd>
         *        <p>
         *        <code>VCPU</code> = 16
         *        </p>
         *        </dd>
         *        </dl>
         *        </dd>
         *        <dt>type="VCPU"</dt>
         *        <dd>
         *        <p>
         *        The number of vCPUs reserved for the container. This parameter maps to <code>CpuShares</code> in the
         *        <a href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section
         *        of the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>--cpu-shares</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>. Each vCPU is equivalent to 1,024 CPU shares. For EC2 resources, you must specify at least one
         *        vCPU. This is required but can be specified in several places; it must be specified for each node at
         *        least once.
         *        </p>
         *        <p>
         *        The default for the Fargate On-Demand vCPU resource count quota is 6 vCPUs. For more information about
         *        Fargate quotas, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/ecs-service.html#service-quotas-fargate">Fargate
         *        quotas</a> in the <i>Amazon Web Services General Reference</i>.
         *        </p>
         *        <p>
         *        For jobs that are running on Fargate resources, then <code>value</code> must match one of the
         *        supported values and the <code>MEMORY</code> values must be one of the values supported for that
         *        <code>VCPU</code> value. The supported values are 0.25, 0.5, 1, 2, 4, 8, and 16
         *        </p>
         *        <dl>
         *        <dt>value = 0.25</dt>
         *        <dd>
         *        <p>
         *        <code>MEMORY</code> = 512, 1024, or 2048
         *        </p>
         *        </dd>
         *        <dt>value = 0.5</dt>
         *        <dd>
         *        <p>
         *        <code>MEMORY</code> = 1024, 2048, 3072, or 4096
         *        </p>
         *        </dd>
         *        <dt>value = 1</dt>
         *        <dd>
         *        <p>
         *        <code>MEMORY</code> = 2048, 3072, 4096, 5120, 6144, 7168, or 8192
         *        </p>
         *        </dd>
         *        <dt>value = 2</dt>
         *        <dd>
         *        <p>
         *        <code>MEMORY</code> = 4096, 5120, 6144, 7168, 8192, 9216, 10240, 11264, 12288, 13312, 14336, 15360, or
         *        16384
         *        </p>
         *        </dd>
         *        <dt>value = 4</dt>
         *        <dd>
         *        <p>
         *        <code>MEMORY</code> = 8192, 9216, 10240, 11264, 12288, 13312, 14336, 15360, 16384, 17408, 18432,
         *        19456, 20480, 21504, 22528, 23552, 24576, 25600, 26624, 27648, 28672, 29696, or 30720
         *        </p>
         *        </dd>
         *        <dt>value = 8</dt>
         *        <dd>
         *        <p>
         *        <code>MEMORY</code> = 16384, 20480, 24576, 28672, 32768, 36864, 40960, 45056, 49152, 53248, 57344, or
         *        61440
         *        </p>
         *        </dd>
         *        <dt>value = 16</dt>
         *        <dd>
         *        <p>
         *        <code>MEMORY</code> = 32768, 40960, 49152, 57344, 65536, 73728, 81920, 90112, 98304, 106496, 114688,
         *        or 122880
         *        </p>
         *        </dd>
         *        </dl>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder value(String value);

        /**
         * <p>
         * The type of resource to assign to a container. The supported resources include <code>GPU</code>,
         * <code>MEMORY</code>, and <code>VCPU</code>.
         * </p>
         * 
         * @param type
         *        The type of resource to assign to a container. The supported resources include <code>GPU</code>,
         *        <code>MEMORY</code>, and <code>VCPU</code>.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of resource to assign to a container. The supported resources include <code>GPU</code>,
         * <code>MEMORY</code>, and <code>VCPU</code>.
         * </p>
         * 
         * @param type
         *        The type of resource to assign to a container. The supported resources include <code>GPU</code>,
         *        <code>MEMORY</code>, and <code>VCPU</code>.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder type(ResourceType type);
    }

    static final class BuilderImpl implements Builder {
        private String value;

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceRequirement model) {
            value(model.value);
            type(model.type);
        }

        public final String getValue() {
            return value;
        }

        public final void setValue(String value) {
            this.value = value;
        }

        @Override
        public final Builder value(String value) {
            this.value = value;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ResourceType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        @Override
        public ResourceRequirement build() {
            return new ResourceRequirement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
