/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backupstorage;

import java.nio.file.Path;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.core.async.AsyncRequestBody;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.services.backupstorage.model.DeleteObjectRequest;
import software.amazon.awssdk.services.backupstorage.model.DeleteObjectResponse;
import software.amazon.awssdk.services.backupstorage.model.GetChunkRequest;
import software.amazon.awssdk.services.backupstorage.model.GetChunkResponse;
import software.amazon.awssdk.services.backupstorage.model.GetObjectMetadataRequest;
import software.amazon.awssdk.services.backupstorage.model.GetObjectMetadataResponse;
import software.amazon.awssdk.services.backupstorage.model.ListChunksRequest;
import software.amazon.awssdk.services.backupstorage.model.ListChunksResponse;
import software.amazon.awssdk.services.backupstorage.model.ListObjectsRequest;
import software.amazon.awssdk.services.backupstorage.model.ListObjectsResponse;
import software.amazon.awssdk.services.backupstorage.model.NotifyObjectCompleteRequest;
import software.amazon.awssdk.services.backupstorage.model.NotifyObjectCompleteResponse;
import software.amazon.awssdk.services.backupstorage.model.PutChunkRequest;
import software.amazon.awssdk.services.backupstorage.model.PutChunkResponse;
import software.amazon.awssdk.services.backupstorage.model.PutObjectRequest;
import software.amazon.awssdk.services.backupstorage.model.PutObjectResponse;
import software.amazon.awssdk.services.backupstorage.model.StartObjectRequest;
import software.amazon.awssdk.services.backupstorage.model.StartObjectResponse;
import software.amazon.awssdk.services.backupstorage.paginators.ListChunksPublisher;
import software.amazon.awssdk.services.backupstorage.paginators.ListObjectsPublisher;

/**
 * Service client for accessing AWS Backup Storage asynchronously. This can be created using the static
 * {@link #builder()} method.
 *
 * The frontend service for Cryo Storage.
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface BackupStorageAsyncClient extends AwsClient {
    String SERVICE_NAME = "backup-storage";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "backupstorage";

    /**
     * Delete Object from the incremental base Backup.
     *
     * @param deleteObjectRequest
     * @return A Java Future containing the result of the DeleteObject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.DeleteObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/DeleteObject" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteObjectResponse> deleteObject(DeleteObjectRequest deleteObjectRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Delete Object from the incremental base Backup.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteObjectRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteObjectRequest#builder()}
     * </p>
     *
     * @param deleteObjectRequest
     *        A {@link Consumer} that will call methods on {@link DeleteObjectInput.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteObject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.DeleteObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/DeleteObject" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteObjectResponse> deleteObject(Consumer<DeleteObjectRequest.Builder> deleteObjectRequest) {
        return deleteObject(DeleteObjectRequest.builder().applyMutation(deleteObjectRequest).build());
    }

    /**
     * Gets the specified object's chunk.
     *
     * @param getChunkRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows 'Chunk data'.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.GetChunk
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> getChunk(GetChunkRequest getChunkRequest,
            AsyncResponseTransformer<GetChunkResponse, ReturnT> asyncResponseTransformer) {
        throw new UnsupportedOperationException();
    }

    /**
     * Gets the specified object's chunk.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetChunkRequest.Builder} avoiding the need to
     * create one manually via {@link GetChunkRequest#builder()}
     * </p>
     *
     * @param getChunkRequest
     *        A {@link Consumer} that will call methods on {@link GetChunkInput.Builder} to create a request.
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows 'Chunk data'.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.GetChunk
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> getChunk(Consumer<GetChunkRequest.Builder> getChunkRequest,
            AsyncResponseTransformer<GetChunkResponse, ReturnT> asyncResponseTransformer) {
        return getChunk(GetChunkRequest.builder().applyMutation(getChunkRequest).build(), asyncResponseTransformer);
    }

    /**
     * Gets the specified object's chunk.
     *
     * @param getChunkRequest
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows 'Chunk data'.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.GetChunk
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetChunkResponse> getChunk(GetChunkRequest getChunkRequest, Path destinationPath) {
        return getChunk(getChunkRequest, AsyncResponseTransformer.toFile(destinationPath));
    }

    /**
     * Gets the specified object's chunk.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetChunkRequest.Builder} avoiding the need to
     * create one manually via {@link GetChunkRequest#builder()}
     * </p>
     *
     * @param getChunkRequest
     *        A {@link Consumer} that will call methods on {@link GetChunkInput.Builder} to create a request.
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows 'Chunk data'.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.GetChunk
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetChunkResponse> getChunk(Consumer<GetChunkRequest.Builder> getChunkRequest, Path destinationPath) {
        return getChunk(GetChunkRequest.builder().applyMutation(getChunkRequest).build(), destinationPath);
    }

    /**
     * Get metadata associated with an Object.
     *
     * @param getObjectMetadataRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows 'Metadata blob.'.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.GetObjectMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetObjectMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> getObjectMetadata(GetObjectMetadataRequest getObjectMetadataRequest,
            AsyncResponseTransformer<GetObjectMetadataResponse, ReturnT> asyncResponseTransformer) {
        throw new UnsupportedOperationException();
    }

    /**
     * Get metadata associated with an Object.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetObjectMetadataRequest.Builder} avoiding the need
     * to create one manually via {@link GetObjectMetadataRequest#builder()}
     * </p>
     *
     * @param getObjectMetadataRequest
     *        A {@link Consumer} that will call methods on {@link GetObjectMetadataInput.Builder} to create a request.
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows 'Metadata blob.'.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.GetObjectMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetObjectMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> getObjectMetadata(
            Consumer<GetObjectMetadataRequest.Builder> getObjectMetadataRequest,
            AsyncResponseTransformer<GetObjectMetadataResponse, ReturnT> asyncResponseTransformer) {
        return getObjectMetadata(GetObjectMetadataRequest.builder().applyMutation(getObjectMetadataRequest).build(),
                asyncResponseTransformer);
    }

    /**
     * Get metadata associated with an Object.
     *
     * @param getObjectMetadataRequest
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows 'Metadata blob.'.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.GetObjectMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetObjectMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetObjectMetadataResponse> getObjectMetadata(GetObjectMetadataRequest getObjectMetadataRequest,
            Path destinationPath) {
        return getObjectMetadata(getObjectMetadataRequest, AsyncResponseTransformer.toFile(destinationPath));
    }

    /**
     * Get metadata associated with an Object.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetObjectMetadataRequest.Builder} avoiding the need
     * to create one manually via {@link GetObjectMetadataRequest#builder()}
     * </p>
     *
     * @param getObjectMetadataRequest
     *        A {@link Consumer} that will call methods on {@link GetObjectMetadataInput.Builder} to create a request.
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows 'Metadata blob.'.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.GetObjectMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetObjectMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetObjectMetadataResponse> getObjectMetadata(
            Consumer<GetObjectMetadataRequest.Builder> getObjectMetadataRequest, Path destinationPath) {
        return getObjectMetadata(GetObjectMetadataRequest.builder().applyMutation(getObjectMetadataRequest).build(),
                destinationPath);
    }

    /**
     * List chunks in a given Object
     *
     * @param listChunksRequest
     * @return A Java Future containing the result of the ListChunks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.ListChunks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListChunks" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListChunksResponse> listChunks(ListChunksRequest listChunksRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * List chunks in a given Object<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListChunksRequest.Builder} avoiding the need to
     * create one manually via {@link ListChunksRequest#builder()}
     * </p>
     *
     * @param listChunksRequest
     *        A {@link Consumer} that will call methods on {@link ListChunksInput.Builder} to create a request.
     * @return A Java Future containing the result of the ListChunks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.ListChunks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListChunks" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListChunksResponse> listChunks(Consumer<ListChunksRequest.Builder> listChunksRequest) {
        return listChunks(ListChunksRequest.builder().applyMutation(listChunksRequest).build());
    }

    /**
     * List chunks in a given Object<br/>
     * <p>
     * This is a variant of {@link #listChunks(software.amazon.awssdk.services.backupstorage.model.ListChunksRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupstorage.paginators.ListChunksPublisher publisher = client.listChunksPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupstorage.paginators.ListChunksPublisher publisher = client.listChunksPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.backupstorage.model.ListChunksResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.backupstorage.model.ListChunksResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChunks(software.amazon.awssdk.services.backupstorage.model.ListChunksRequest)} operation.</b>
     * </p>
     *
     * @param listChunksRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.ListChunks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListChunks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChunksPublisher listChunksPaginator(ListChunksRequest listChunksRequest) {
        return new ListChunksPublisher(this, listChunksRequest);
    }

    /**
     * List chunks in a given Object<br/>
     * <p>
     * This is a variant of {@link #listChunks(software.amazon.awssdk.services.backupstorage.model.ListChunksRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupstorage.paginators.ListChunksPublisher publisher = client.listChunksPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupstorage.paginators.ListChunksPublisher publisher = client.listChunksPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.backupstorage.model.ListChunksResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.backupstorage.model.ListChunksResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChunks(software.amazon.awssdk.services.backupstorage.model.ListChunksRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListChunksRequest.Builder} avoiding the need to
     * create one manually via {@link ListChunksRequest#builder()}
     * </p>
     *
     * @param listChunksRequest
     *        A {@link Consumer} that will call methods on {@link ListChunksInput.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.ListChunks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListChunks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChunksPublisher listChunksPaginator(Consumer<ListChunksRequest.Builder> listChunksRequest) {
        return listChunksPaginator(ListChunksRequest.builder().applyMutation(listChunksRequest).build());
    }

    /**
     * List all Objects in a given Backup.
     *
     * @param listObjectsRequest
     * @return A Java Future containing the result of the ListObjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.ListObjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListObjects" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListObjectsResponse> listObjects(ListObjectsRequest listObjectsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * List all Objects in a given Backup.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListObjectsRequest.Builder} avoiding the need to
     * create one manually via {@link ListObjectsRequest#builder()}
     * </p>
     *
     * @param listObjectsRequest
     *        A {@link Consumer} that will call methods on {@link ListObjectsInput.Builder} to create a request.
     * @return A Java Future containing the result of the ListObjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.ListObjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListObjects" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListObjectsResponse> listObjects(Consumer<ListObjectsRequest.Builder> listObjectsRequest) {
        return listObjects(ListObjectsRequest.builder().applyMutation(listObjectsRequest).build());
    }

    /**
     * List all Objects in a given Backup.<br/>
     * <p>
     * This is a variant of {@link #listObjects(software.amazon.awssdk.services.backupstorage.model.ListObjectsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupstorage.paginators.ListObjectsPublisher publisher = client.listObjectsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupstorage.paginators.ListObjectsPublisher publisher = client.listObjectsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.backupstorage.model.ListObjectsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.backupstorage.model.ListObjectsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listObjects(software.amazon.awssdk.services.backupstorage.model.ListObjectsRequest)} operation.</b>
     * </p>
     *
     * @param listObjectsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.ListObjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListObjects" target="_top">AWS API
     *      Documentation</a>
     */
    default ListObjectsPublisher listObjectsPaginator(ListObjectsRequest listObjectsRequest) {
        return new ListObjectsPublisher(this, listObjectsRequest);
    }

    /**
     * List all Objects in a given Backup.<br/>
     * <p>
     * This is a variant of {@link #listObjects(software.amazon.awssdk.services.backupstorage.model.ListObjectsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupstorage.paginators.ListObjectsPublisher publisher = client.listObjectsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupstorage.paginators.ListObjectsPublisher publisher = client.listObjectsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.backupstorage.model.ListObjectsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.backupstorage.model.ListObjectsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listObjects(software.amazon.awssdk.services.backupstorage.model.ListObjectsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListObjectsRequest.Builder} avoiding the need to
     * create one manually via {@link ListObjectsRequest#builder()}
     * </p>
     *
     * @param listObjectsRequest
     *        A {@link Consumer} that will call methods on {@link ListObjectsInput.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.ListObjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListObjects" target="_top">AWS API
     *      Documentation</a>
     */
    default ListObjectsPublisher listObjectsPaginator(Consumer<ListObjectsRequest.Builder> listObjectsRequest) {
        return listObjectsPaginator(ListObjectsRequest.builder().applyMutation(listObjectsRequest).build());
    }

    /**
     * Complete upload
     *
     * @param notifyObjectCompleteRequest
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows 'Optional metadata
     *        associated with an Object. Maximum length is 4MB.'
     * @return A Java Future containing the result of the NotifyObjectComplete operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>NotReadableInputStreamException Retryalble exception. Indicated issues while reading an input stream
     *         due to the networking issues or connection drop on the client side.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.NotifyObjectComplete
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/NotifyObjectComplete"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<NotifyObjectCompleteResponse> notifyObjectComplete(
            NotifyObjectCompleteRequest notifyObjectCompleteRequest, AsyncRequestBody requestBody) {
        throw new UnsupportedOperationException();
    }

    /**
     * Complete upload<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link NotifyObjectCompleteRequest.Builder} avoiding the
     * need to create one manually via {@link NotifyObjectCompleteRequest#builder()}
     * </p>
     *
     * @param notifyObjectCompleteRequest
     *        A {@link Consumer} that will call methods on {@link NotifyObjectCompleteInput.Builder} to create a
     *        request.
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows 'Optional metadata
     *        associated with an Object. Maximum length is 4MB.'
     * @return A Java Future containing the result of the NotifyObjectComplete operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>NotReadableInputStreamException Retryalble exception. Indicated issues while reading an input stream
     *         due to the networking issues or connection drop on the client side.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.NotifyObjectComplete
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/NotifyObjectComplete"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<NotifyObjectCompleteResponse> notifyObjectComplete(
            Consumer<NotifyObjectCompleteRequest.Builder> notifyObjectCompleteRequest, AsyncRequestBody requestBody) {
        return notifyObjectComplete(NotifyObjectCompleteRequest.builder().applyMutation(notifyObjectCompleteRequest).build(),
                requestBody);
    }

    /**
     * Complete upload
     *
     * @param notifyObjectCompleteRequest
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows 'Optional metadata associated with an Object. Maximum length is 4MB.'
     * @return A Java Future containing the result of the NotifyObjectComplete operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>NotReadableInputStreamException Retryalble exception. Indicated issues while reading an input stream
     *         due to the networking issues or connection drop on the client side.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.NotifyObjectComplete
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/NotifyObjectComplete"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<NotifyObjectCompleteResponse> notifyObjectComplete(
            NotifyObjectCompleteRequest notifyObjectCompleteRequest, Path sourcePath) {
        return notifyObjectComplete(notifyObjectCompleteRequest, AsyncRequestBody.fromFile(sourcePath));
    }

    /**
     * Complete upload<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link NotifyObjectCompleteRequest.Builder} avoiding the
     * need to create one manually via {@link NotifyObjectCompleteRequest#builder()}
     * </p>
     *
     * @param notifyObjectCompleteRequest
     *        A {@link Consumer} that will call methods on {@link NotifyObjectCompleteInput.Builder} to create a
     *        request.
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows 'Optional metadata associated with an Object. Maximum length is 4MB.'
     * @return A Java Future containing the result of the NotifyObjectComplete operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>NotReadableInputStreamException Retryalble exception. Indicated issues while reading an input stream
     *         due to the networking issues or connection drop on the client side.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.NotifyObjectComplete
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/NotifyObjectComplete"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<NotifyObjectCompleteResponse> notifyObjectComplete(
            Consumer<NotifyObjectCompleteRequest.Builder> notifyObjectCompleteRequest, Path sourcePath) {
        return notifyObjectComplete(NotifyObjectCompleteRequest.builder().applyMutation(notifyObjectCompleteRequest).build(),
                sourcePath);
    }

    /**
     * Upload chunk.
     *
     * @param putChunkRequest
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows 'Data to be
     *        uploaded'
     * @return A Java Future containing the result of the PutChunk operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>NotReadableInputStreamException Retryalble exception. Indicated issues while reading an input stream
     *         due to the networking issues or connection drop on the client side.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.PutChunk
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutChunkResponse> putChunk(PutChunkRequest putChunkRequest, AsyncRequestBody requestBody) {
        throw new UnsupportedOperationException();
    }

    /**
     * Upload chunk.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutChunkRequest.Builder} avoiding the need to
     * create one manually via {@link PutChunkRequest#builder()}
     * </p>
     *
     * @param putChunkRequest
     *        A {@link Consumer} that will call methods on {@link PutChunkInput.Builder} to create a request.
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows 'Data to be
     *        uploaded'
     * @return A Java Future containing the result of the PutChunk operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>NotReadableInputStreamException Retryalble exception. Indicated issues while reading an input stream
     *         due to the networking issues or connection drop on the client side.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.PutChunk
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutChunkResponse> putChunk(Consumer<PutChunkRequest.Builder> putChunkRequest,
            AsyncRequestBody requestBody) {
        return putChunk(PutChunkRequest.builder().applyMutation(putChunkRequest).build(), requestBody);
    }

    /**
     * Upload chunk.
     *
     * @param putChunkRequest
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows 'Data to be uploaded'
     * @return A Java Future containing the result of the PutChunk operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>NotReadableInputStreamException Retryalble exception. Indicated issues while reading an input stream
     *         due to the networking issues or connection drop on the client side.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.PutChunk
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutChunkResponse> putChunk(PutChunkRequest putChunkRequest, Path sourcePath) {
        return putChunk(putChunkRequest, AsyncRequestBody.fromFile(sourcePath));
    }

    /**
     * Upload chunk.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutChunkRequest.Builder} avoiding the need to
     * create one manually via {@link PutChunkRequest#builder()}
     * </p>
     *
     * @param putChunkRequest
     *        A {@link Consumer} that will call methods on {@link PutChunkInput.Builder} to create a request.
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows 'Data to be uploaded'
     * @return A Java Future containing the result of the PutChunk operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>NotReadableInputStreamException Retryalble exception. Indicated issues while reading an input stream
     *         due to the networking issues or connection drop on the client side.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.PutChunk
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutChunk" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutChunkResponse> putChunk(Consumer<PutChunkRequest.Builder> putChunkRequest, Path sourcePath) {
        return putChunk(PutChunkRequest.builder().applyMutation(putChunkRequest).build(), sourcePath);
    }

    /**
     * Upload object that can store object metadata String and data blob in single API call using inline chunk field.
     *
     * @param putObjectRequest
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows 'Inline chunk data
     *        to be uploaded.'
     * @return A Java Future containing the result of the PutObject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>NotReadableInputStreamException Retryalble exception. Indicated issues while reading an input stream
     *         due to the networking issues or connection drop on the client side.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.PutObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutObject" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutObjectResponse> putObject(PutObjectRequest putObjectRequest, AsyncRequestBody requestBody) {
        throw new UnsupportedOperationException();
    }

    /**
     * Upload object that can store object metadata String and data blob in single API call using inline chunk field.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutObjectRequest.Builder} avoiding the need to
     * create one manually via {@link PutObjectRequest#builder()}
     * </p>
     *
     * @param putObjectRequest
     *        A {@link Consumer} that will call methods on {@link PutObjectInput.Builder} to create a request.
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows 'Inline chunk data
     *        to be uploaded.'
     * @return A Java Future containing the result of the PutObject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>NotReadableInputStreamException Retryalble exception. Indicated issues while reading an input stream
     *         due to the networking issues or connection drop on the client side.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.PutObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutObject" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutObjectResponse> putObject(Consumer<PutObjectRequest.Builder> putObjectRequest,
            AsyncRequestBody requestBody) {
        return putObject(PutObjectRequest.builder().applyMutation(putObjectRequest).build(), requestBody);
    }

    /**
     * Upload object that can store object metadata String and data blob in single API call using inline chunk field.
     *
     * @param putObjectRequest
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows 'Inline chunk data to be uploaded.'
     * @return A Java Future containing the result of the PutObject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>NotReadableInputStreamException Retryalble exception. Indicated issues while reading an input stream
     *         due to the networking issues or connection drop on the client side.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.PutObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutObject" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutObjectResponse> putObject(PutObjectRequest putObjectRequest, Path sourcePath) {
        return putObject(putObjectRequest, AsyncRequestBody.fromFile(sourcePath));
    }

    /**
     * Upload object that can store object metadata String and data blob in single API call using inline chunk field.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutObjectRequest.Builder} avoiding the need to
     * create one manually via {@link PutObjectRequest#builder()}
     * </p>
     *
     * @param putObjectRequest
     *        A {@link Consumer} that will call methods on {@link PutObjectInput.Builder} to create a request.
     * @param sourcePath
     *        {@link Path} to file containing data to send to the service. File will be read entirely and may be read
     *        multiple times in the event of a retry. If the file does not exist or the current user does not have
     *        access to read it then an exception will be thrown. The service documentation for the request content is
     *        as follows 'Inline chunk data to be uploaded.'
     * @return A Java Future containing the result of the PutObject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>NotReadableInputStreamException Retryalble exception. Indicated issues while reading an input stream
     *         due to the networking issues or connection drop on the client side.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.PutObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutObject" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutObjectResponse> putObject(Consumer<PutObjectRequest.Builder> putObjectRequest, Path sourcePath) {
        return putObject(PutObjectRequest.builder().applyMutation(putObjectRequest).build(), sourcePath);
    }

    /**
     * Start upload containing one or many chunks.
     *
     * @param startObjectRequest
     * @return A Java Future containing the result of the StartObject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>DataAlreadyExistsException Non-retryable exception. Attempted to create already existing object or
     *         chunk. This message contains a checksum of already presented data.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.StartObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/StartObject" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<StartObjectResponse> startObject(StartObjectRequest startObjectRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Start upload containing one or many chunks.<br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartObjectRequest.Builder} avoiding the need to
     * create one manually via {@link StartObjectRequest#builder()}
     * </p>
     *
     * @param startObjectRequest
     *        A {@link Consumer} that will call methods on {@link StartObjectInput.Builder} to create a request.
     * @return A Java Future containing the result of the StartObject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>DataAlreadyExistsException Non-retryable exception. Attempted to create already existing object or
     *         chunk. This message contains a checksum of already presented data.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.StartObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/StartObject" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<StartObjectResponse> startObject(Consumer<StartObjectRequest.Builder> startObjectRequest) {
        return startObject(StartObjectRequest.builder().applyMutation(startObjectRequest).build());
    }

    @Override
    default BackupStorageServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link BackupStorageAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static BackupStorageAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link BackupStorageAsyncClient}.
     */
    static BackupStorageAsyncClientBuilder builder() {
        return new DefaultBackupStorageAsyncClientBuilder();
    }
}
