/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appfabric;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.appfabric.internal.AppFabricServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.appfabric.model.AccessDeniedException;
import software.amazon.awssdk.services.appfabric.model.AppFabricException;
import software.amazon.awssdk.services.appfabric.model.BatchGetUserAccessTasksRequest;
import software.amazon.awssdk.services.appfabric.model.BatchGetUserAccessTasksResponse;
import software.amazon.awssdk.services.appfabric.model.ConflictException;
import software.amazon.awssdk.services.appfabric.model.ConnectAppAuthorizationRequest;
import software.amazon.awssdk.services.appfabric.model.ConnectAppAuthorizationResponse;
import software.amazon.awssdk.services.appfabric.model.CreateAppAuthorizationRequest;
import software.amazon.awssdk.services.appfabric.model.CreateAppAuthorizationResponse;
import software.amazon.awssdk.services.appfabric.model.CreateAppBundleRequest;
import software.amazon.awssdk.services.appfabric.model.CreateAppBundleResponse;
import software.amazon.awssdk.services.appfabric.model.CreateIngestionDestinationRequest;
import software.amazon.awssdk.services.appfabric.model.CreateIngestionDestinationResponse;
import software.amazon.awssdk.services.appfabric.model.CreateIngestionRequest;
import software.amazon.awssdk.services.appfabric.model.CreateIngestionResponse;
import software.amazon.awssdk.services.appfabric.model.DeleteAppAuthorizationRequest;
import software.amazon.awssdk.services.appfabric.model.DeleteAppAuthorizationResponse;
import software.amazon.awssdk.services.appfabric.model.DeleteAppBundleRequest;
import software.amazon.awssdk.services.appfabric.model.DeleteAppBundleResponse;
import software.amazon.awssdk.services.appfabric.model.DeleteIngestionDestinationRequest;
import software.amazon.awssdk.services.appfabric.model.DeleteIngestionDestinationResponse;
import software.amazon.awssdk.services.appfabric.model.DeleteIngestionRequest;
import software.amazon.awssdk.services.appfabric.model.DeleteIngestionResponse;
import software.amazon.awssdk.services.appfabric.model.GetAppAuthorizationRequest;
import software.amazon.awssdk.services.appfabric.model.GetAppAuthorizationResponse;
import software.amazon.awssdk.services.appfabric.model.GetAppBundleRequest;
import software.amazon.awssdk.services.appfabric.model.GetAppBundleResponse;
import software.amazon.awssdk.services.appfabric.model.GetIngestionDestinationRequest;
import software.amazon.awssdk.services.appfabric.model.GetIngestionDestinationResponse;
import software.amazon.awssdk.services.appfabric.model.GetIngestionRequest;
import software.amazon.awssdk.services.appfabric.model.GetIngestionResponse;
import software.amazon.awssdk.services.appfabric.model.InternalServerException;
import software.amazon.awssdk.services.appfabric.model.ListAppAuthorizationsRequest;
import software.amazon.awssdk.services.appfabric.model.ListAppAuthorizationsResponse;
import software.amazon.awssdk.services.appfabric.model.ListAppBundlesRequest;
import software.amazon.awssdk.services.appfabric.model.ListAppBundlesResponse;
import software.amazon.awssdk.services.appfabric.model.ListIngestionDestinationsRequest;
import software.amazon.awssdk.services.appfabric.model.ListIngestionDestinationsResponse;
import software.amazon.awssdk.services.appfabric.model.ListIngestionsRequest;
import software.amazon.awssdk.services.appfabric.model.ListIngestionsResponse;
import software.amazon.awssdk.services.appfabric.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.appfabric.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.appfabric.model.ResourceNotFoundException;
import software.amazon.awssdk.services.appfabric.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.appfabric.model.StartIngestionRequest;
import software.amazon.awssdk.services.appfabric.model.StartIngestionResponse;
import software.amazon.awssdk.services.appfabric.model.StartUserAccessTasksRequest;
import software.amazon.awssdk.services.appfabric.model.StartUserAccessTasksResponse;
import software.amazon.awssdk.services.appfabric.model.StopIngestionRequest;
import software.amazon.awssdk.services.appfabric.model.StopIngestionResponse;
import software.amazon.awssdk.services.appfabric.model.TagResourceRequest;
import software.amazon.awssdk.services.appfabric.model.TagResourceResponse;
import software.amazon.awssdk.services.appfabric.model.ThrottlingException;
import software.amazon.awssdk.services.appfabric.model.UntagResourceRequest;
import software.amazon.awssdk.services.appfabric.model.UntagResourceResponse;
import software.amazon.awssdk.services.appfabric.model.UpdateAppAuthorizationRequest;
import software.amazon.awssdk.services.appfabric.model.UpdateAppAuthorizationResponse;
import software.amazon.awssdk.services.appfabric.model.UpdateIngestionDestinationRequest;
import software.amazon.awssdk.services.appfabric.model.UpdateIngestionDestinationResponse;
import software.amazon.awssdk.services.appfabric.model.ValidationException;
import software.amazon.awssdk.services.appfabric.transform.BatchGetUserAccessTasksRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.ConnectAppAuthorizationRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.CreateAppAuthorizationRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.CreateAppBundleRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.CreateIngestionDestinationRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.CreateIngestionRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.DeleteAppAuthorizationRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.DeleteAppBundleRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.DeleteIngestionDestinationRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.DeleteIngestionRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.GetAppAuthorizationRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.GetAppBundleRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.GetIngestionDestinationRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.GetIngestionRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.ListAppAuthorizationsRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.ListAppBundlesRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.ListIngestionDestinationsRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.ListIngestionsRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.StartIngestionRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.StartUserAccessTasksRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.StopIngestionRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.UpdateAppAuthorizationRequestMarshaller;
import software.amazon.awssdk.services.appfabric.transform.UpdateIngestionDestinationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AppFabricAsyncClient}.
 *
 * @see AppFabricAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAppFabricAsyncClient implements AppFabricAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAppFabricAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final AppFabricServiceClientConfiguration serviceClientConfiguration;

    protected DefaultAppFabricAsyncClient(AppFabricServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Gets user access details in a batch request.
     * </p>
     * <p>
     * This action polls data from the tasks that are kicked off by the <code>StartUserAccessTasks</code> action.
     * </p>
     *
     * @param batchGetUserAccessTasksRequest
     * @return A Java Future containing the result of the BatchGetUserAccessTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.BatchGetUserAccessTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/BatchGetUserAccessTasks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetUserAccessTasksResponse> batchGetUserAccessTasks(
            BatchGetUserAccessTasksRequest batchGetUserAccessTasksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetUserAccessTasksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetUserAccessTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetUserAccessTasks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetUserAccessTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetUserAccessTasksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetUserAccessTasksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetUserAccessTasksRequest, BatchGetUserAccessTasksResponse>()
                            .withOperationName("BatchGetUserAccessTasks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchGetUserAccessTasksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchGetUserAccessTasksRequest));
            CompletableFuture<BatchGetUserAccessTasksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Establishes a connection between Amazon Web Services AppFabric and an application, which allows AppFabric to call
     * the APIs of the application.
     * </p>
     *
     * @param connectAppAuthorizationRequest
     * @return A Java Future containing the result of the ConnectAppAuthorization operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.ConnectAppAuthorization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/ConnectAppAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ConnectAppAuthorizationResponse> connectAppAuthorization(
            ConnectAppAuthorizationRequest connectAppAuthorizationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(connectAppAuthorizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, connectAppAuthorizationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ConnectAppAuthorization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ConnectAppAuthorizationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ConnectAppAuthorizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ConnectAppAuthorizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ConnectAppAuthorizationRequest, ConnectAppAuthorizationResponse>()
                            .withOperationName("ConnectAppAuthorization").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ConnectAppAuthorizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(connectAppAuthorizationRequest));
            CompletableFuture<ConnectAppAuthorizationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an app authorization within an app bundle, which allows AppFabric to connect to an application.
     * </p>
     *
     * @param createAppAuthorizationRequest
     * @return A Java Future containing the result of the CreateAppAuthorization operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ServiceQuotaExceededException The request exceeds a service quota.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ConflictException The request has created a conflict. Check the request parameters and try again.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.CreateAppAuthorization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/CreateAppAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAppAuthorizationResponse> createAppAuthorization(
            CreateAppAuthorizationRequest createAppAuthorizationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAppAuthorizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppAuthorizationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAppAuthorization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAppAuthorizationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAppAuthorizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAppAuthorizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAppAuthorizationRequest, CreateAppAuthorizationResponse>()
                            .withOperationName("CreateAppAuthorization").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAppAuthorizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAppAuthorizationRequest));
            CompletableFuture<CreateAppAuthorizationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an app bundle to collect data from an application using AppFabric.
     * </p>
     *
     * @param createAppBundleRequest
     * @return A Java Future containing the result of the CreateAppBundle operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ServiceQuotaExceededException The request exceeds a service quota.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ConflictException The request has created a conflict. Check the request parameters and try again.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.CreateAppBundle
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/CreateAppBundle" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAppBundleResponse> createAppBundle(CreateAppBundleRequest createAppBundleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAppBundleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppBundleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAppBundle");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAppBundleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAppBundleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAppBundleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAppBundleRequest, CreateAppBundleResponse>()
                            .withOperationName("CreateAppBundle").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAppBundleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAppBundleRequest));
            CompletableFuture<CreateAppBundleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a data ingestion for an application.
     * </p>
     *
     * @param createIngestionRequest
     * @return A Java Future containing the result of the CreateIngestion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ServiceQuotaExceededException The request exceeds a service quota.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ConflictException The request has created a conflict. Check the request parameters and try again.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.CreateIngestion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/CreateIngestion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateIngestionResponse> createIngestion(CreateIngestionRequest createIngestionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createIngestionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createIngestionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateIngestion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateIngestionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateIngestionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateIngestionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateIngestionRequest, CreateIngestionResponse>()
                            .withOperationName("CreateIngestion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateIngestionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createIngestionRequest));
            CompletableFuture<CreateIngestionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an ingestion destination, which specifies how an application's ingested data is processed by Amazon Web
     * Services AppFabric and where it's delivered.
     * </p>
     *
     * @param createIngestionDestinationRequest
     * @return A Java Future containing the result of the CreateIngestionDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ServiceQuotaExceededException The request exceeds a service quota.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ConflictException The request has created a conflict. Check the request parameters and try again.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.CreateIngestionDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/CreateIngestionDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateIngestionDestinationResponse> createIngestionDestination(
            CreateIngestionDestinationRequest createIngestionDestinationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createIngestionDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createIngestionDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateIngestionDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateIngestionDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateIngestionDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateIngestionDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateIngestionDestinationRequest, CreateIngestionDestinationResponse>()
                            .withOperationName("CreateIngestionDestination").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateIngestionDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createIngestionDestinationRequest));
            CompletableFuture<CreateIngestionDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an app authorization. You must delete the associated ingestion before you can delete an app
     * authorization.
     * </p>
     *
     * @param deleteAppAuthorizationRequest
     * @return A Java Future containing the result of the DeleteAppAuthorization operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.DeleteAppAuthorization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/DeleteAppAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppAuthorizationResponse> deleteAppAuthorization(
            DeleteAppAuthorizationRequest deleteAppAuthorizationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAppAuthorizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppAuthorizationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppAuthorization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppAuthorizationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAppAuthorizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAppAuthorizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppAuthorizationRequest, DeleteAppAuthorizationResponse>()
                            .withOperationName("DeleteAppAuthorization").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAppAuthorizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAppAuthorizationRequest));
            CompletableFuture<DeleteAppAuthorizationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an app bundle. You must delete all associated app authorizations before you can delete an app bundle.
     * </p>
     *
     * @param deleteAppBundleRequest
     * @return A Java Future containing the result of the DeleteAppBundle operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ConflictException The request has created a conflict. Check the request parameters and try again.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.DeleteAppBundle
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/DeleteAppBundle" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppBundleResponse> deleteAppBundle(DeleteAppBundleRequest deleteAppBundleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAppBundleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppBundleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppBundle");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppBundleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAppBundleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAppBundleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppBundleRequest, DeleteAppBundleResponse>()
                            .withOperationName("DeleteAppBundle").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAppBundleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAppBundleRequest));
            CompletableFuture<DeleteAppBundleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an ingestion. You must stop (disable) the ingestion and you must delete all associated ingestion
     * destinations before you can delete an app ingestion.
     * </p>
     *
     * @param deleteIngestionRequest
     * @return A Java Future containing the result of the DeleteIngestion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.DeleteIngestion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/DeleteIngestion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteIngestionResponse> deleteIngestion(DeleteIngestionRequest deleteIngestionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteIngestionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIngestionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIngestion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteIngestionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteIngestionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteIngestionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteIngestionRequest, DeleteIngestionResponse>()
                            .withOperationName("DeleteIngestion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteIngestionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteIngestionRequest));
            CompletableFuture<DeleteIngestionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an ingestion destination.
     * </p>
     * <p>
     * This deletes the association between an ingestion and it's destination. It doesn't delete previously ingested
     * data or the storage destination, such as the Amazon S3 bucket where the data is delivered. If the ingestion
     * destination is deleted while the associated ingestion is enabled, the ingestion will fail and is eventually
     * disabled.
     * </p>
     *
     * @param deleteIngestionDestinationRequest
     * @return A Java Future containing the result of the DeleteIngestionDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.DeleteIngestionDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/DeleteIngestionDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteIngestionDestinationResponse> deleteIngestionDestination(
            DeleteIngestionDestinationRequest deleteIngestionDestinationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteIngestionDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIngestionDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIngestionDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteIngestionDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteIngestionDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteIngestionDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteIngestionDestinationRequest, DeleteIngestionDestinationResponse>()
                            .withOperationName("DeleteIngestionDestination").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteIngestionDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteIngestionDestinationRequest));
            CompletableFuture<DeleteIngestionDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about an app authorization.
     * </p>
     *
     * @param getAppAuthorizationRequest
     * @return A Java Future containing the result of the GetAppAuthorization operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.GetAppAuthorization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/GetAppAuthorization" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAppAuthorizationResponse> getAppAuthorization(
            GetAppAuthorizationRequest getAppAuthorizationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAppAuthorizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAppAuthorizationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAppAuthorization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAppAuthorizationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAppAuthorizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAppAuthorizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAppAuthorizationRequest, GetAppAuthorizationResponse>()
                            .withOperationName("GetAppAuthorization").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAppAuthorizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAppAuthorizationRequest));
            CompletableFuture<GetAppAuthorizationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about an app bundle.
     * </p>
     *
     * @param getAppBundleRequest
     * @return A Java Future containing the result of the GetAppBundle operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.GetAppBundle
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/GetAppBundle" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAppBundleResponse> getAppBundle(GetAppBundleRequest getAppBundleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAppBundleRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAppBundleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAppBundle");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAppBundleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAppBundleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAppBundleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAppBundleRequest, GetAppBundleResponse>()
                            .withOperationName("GetAppBundle").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAppBundleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAppBundleRequest));
            CompletableFuture<GetAppBundleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about an ingestion.
     * </p>
     *
     * @param getIngestionRequest
     * @return A Java Future containing the result of the GetIngestion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.GetIngestion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/GetIngestion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetIngestionResponse> getIngestion(GetIngestionRequest getIngestionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getIngestionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIngestionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIngestion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetIngestionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetIngestionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetIngestionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetIngestionRequest, GetIngestionResponse>()
                            .withOperationName("GetIngestion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetIngestionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getIngestionRequest));
            CompletableFuture<GetIngestionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about an ingestion destination.
     * </p>
     *
     * @param getIngestionDestinationRequest
     * @return A Java Future containing the result of the GetIngestionDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.GetIngestionDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/GetIngestionDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetIngestionDestinationResponse> getIngestionDestination(
            GetIngestionDestinationRequest getIngestionDestinationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getIngestionDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIngestionDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIngestionDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetIngestionDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetIngestionDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetIngestionDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetIngestionDestinationRequest, GetIngestionDestinationResponse>()
                            .withOperationName("GetIngestionDestination").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetIngestionDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getIngestionDestinationRequest));
            CompletableFuture<GetIngestionDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all app authorizations configured for an app bundle.
     * </p>
     *
     * @param listAppAuthorizationsRequest
     * @return A Java Future containing the result of the ListAppAuthorizations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.ListAppAuthorizations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/ListAppAuthorizations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppAuthorizationsResponse> listAppAuthorizations(
            ListAppAuthorizationsRequest listAppAuthorizationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAppAuthorizationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppAuthorizationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppAuthorizations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppAuthorizationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAppAuthorizationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAppAuthorizationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppAuthorizationsRequest, ListAppAuthorizationsResponse>()
                            .withOperationName("ListAppAuthorizations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAppAuthorizationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAppAuthorizationsRequest));
            CompletableFuture<ListAppAuthorizationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of app bundles.
     * </p>
     *
     * @param listAppBundlesRequest
     * @return A Java Future containing the result of the ListAppBundles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.ListAppBundles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/ListAppBundles" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppBundlesResponse> listAppBundles(ListAppBundlesRequest listAppBundlesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAppBundlesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppBundlesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppBundles");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppBundlesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAppBundlesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAppBundlesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppBundlesRequest, ListAppBundlesResponse>()
                            .withOperationName("ListAppBundles").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAppBundlesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAppBundlesRequest));
            CompletableFuture<ListAppBundlesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all ingestion destinations configured for an ingestion.
     * </p>
     *
     * @param listIngestionDestinationsRequest
     * @return A Java Future containing the result of the ListIngestionDestinations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.ListIngestionDestinations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/ListIngestionDestinations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListIngestionDestinationsResponse> listIngestionDestinations(
            ListIngestionDestinationsRequest listIngestionDestinationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listIngestionDestinationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIngestionDestinationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIngestionDestinations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListIngestionDestinationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListIngestionDestinationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListIngestionDestinationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListIngestionDestinationsRequest, ListIngestionDestinationsResponse>()
                            .withOperationName("ListIngestionDestinations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListIngestionDestinationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listIngestionDestinationsRequest));
            CompletableFuture<ListIngestionDestinationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all ingestions configured for an app bundle.
     * </p>
     *
     * @param listIngestionsRequest
     * @return A Java Future containing the result of the ListIngestions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.ListIngestions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/ListIngestions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListIngestionsResponse> listIngestions(ListIngestionsRequest listIngestionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listIngestionsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIngestionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIngestions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListIngestionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListIngestionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListIngestionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListIngestionsRequest, ListIngestionsResponse>()
                            .withOperationName("ListIngestions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListIngestionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listIngestionsRequest));
            CompletableFuture<ListIngestionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts (enables) an ingestion, which collects data from an application.
     * </p>
     *
     * @param startIngestionRequest
     * @return A Java Future containing the result of the StartIngestion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ConflictException The request has created a conflict. Check the request parameters and try again.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.StartIngestion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/StartIngestion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartIngestionResponse> startIngestion(StartIngestionRequest startIngestionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startIngestionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startIngestionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartIngestion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartIngestionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartIngestionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartIngestionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartIngestionRequest, StartIngestionResponse>()
                            .withOperationName("StartIngestion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartIngestionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startIngestionRequest));
            CompletableFuture<StartIngestionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts the tasks to search user access status for a specific email address.
     * </p>
     * <p>
     * The tasks are stopped when the user access status data is found. The tasks are terminated when the API calls to
     * the application time out.
     * </p>
     *
     * @param startUserAccessTasksRequest
     * @return A Java Future containing the result of the StartUserAccessTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.StartUserAccessTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/StartUserAccessTasks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartUserAccessTasksResponse> startUserAccessTasks(
            StartUserAccessTasksRequest startUserAccessTasksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startUserAccessTasksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startUserAccessTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartUserAccessTasks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartUserAccessTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartUserAccessTasksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartUserAccessTasksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartUserAccessTasksRequest, StartUserAccessTasksResponse>()
                            .withOperationName("StartUserAccessTasks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartUserAccessTasksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startUserAccessTasksRequest));
            CompletableFuture<StartUserAccessTasksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops (disables) an ingestion.
     * </p>
     *
     * @param stopIngestionRequest
     * @return A Java Future containing the result of the StopIngestion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ConflictException The request has created a conflict. Check the request parameters and try again.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.StopIngestion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/StopIngestion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopIngestionResponse> stopIngestion(StopIngestionRequest stopIngestionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopIngestionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopIngestionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopIngestion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopIngestionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopIngestionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopIngestionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopIngestionRequest, StopIngestionResponse>()
                            .withOperationName("StopIngestion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopIngestionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopIngestionRequest));
            CompletableFuture<StopIngestionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag or tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an app authorization within an app bundle, which allows AppFabric to connect to an application.
     * </p>
     * <p>
     * If the app authorization was in a <code>connected</code> state, updating the app authorization will set it back
     * to a <code>PendingConnect</code> state.
     * </p>
     *
     * @param updateAppAuthorizationRequest
     * @return A Java Future containing the result of the UpdateAppAuthorization operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.UpdateAppAuthorization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/UpdateAppAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAppAuthorizationResponse> updateAppAuthorization(
            UpdateAppAuthorizationRequest updateAppAuthorizationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAppAuthorizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAppAuthorizationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAppAuthorization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAppAuthorizationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAppAuthorizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAppAuthorizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAppAuthorizationRequest, UpdateAppAuthorizationResponse>()
                            .withOperationName("UpdateAppAuthorization").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAppAuthorizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAppAuthorizationRequest));
            CompletableFuture<UpdateAppAuthorizationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an ingestion destination, which specifies how an application's ingested data is processed by Amazon Web
     * Services AppFabric and where it's delivered.
     * </p>
     *
     * @param updateIngestionDestinationRequest
     * @return A Java Future containing the result of the UpdateIngestionDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure with an internal server.</li>
     *         <li>ServiceQuotaExceededException The request exceeds a service quota.</li>
     *         <li>ThrottlingException The request rate exceeds the limit.</li>
     *         <li>ConflictException The request has created a conflict. Check the request parameters and try again.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>AccessDeniedException You are not authorized to perform this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppFabricAsyncClient.UpdateIngestionDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appfabric-2023-05-19/UpdateIngestionDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateIngestionDestinationResponse> updateIngestionDestination(
            UpdateIngestionDestinationRequest updateIngestionDestinationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateIngestionDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateIngestionDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateIngestionDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateIngestionDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateIngestionDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateIngestionDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateIngestionDestinationRequest, UpdateIngestionDestinationResponse>()
                            .withOperationName("UpdateIngestionDestination").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateIngestionDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateIngestionDestinationRequest));
            CompletableFuture<UpdateIngestionDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final AppFabricServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AppFabricException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        AppFabricServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = AppFabricServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
