/*
 * (C) Copyright 2006-2010 Nuxeo SA (http://nuxeo.com/) and others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 *     bstefanescu
 */
package org.nuxeo.shell.fs;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

import org.nuxeo.shell.Shell;
import org.nuxeo.shell.ShellFeature;
import org.nuxeo.shell.fs.cmds.FileSystemCommands;

/**
 * @author <a href="mailto:bs@nuxeo.com">Bogdan Stefanescu</a>
 */
public class FileSystem implements ShellFeature {

    protected List<File> wdStack;

    public FileSystem() {
        wdStack = new ArrayList<File>();
        try {
            wdStack.add(new File(".").getCanonicalFile());
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public void install(Shell shell) {
        shell.putContextObject(FileSystem.class, this);
        shell.addValueAdapter(new FileValueAdapter());
        shell.addRegistry(FileSystemCommands.INSTANCE);
    }

    public List<File> getStack() {
        return wdStack;
    }

    public File pwd() {
        return wdStack.get(wdStack.size() - 1);
    }

    public void cd(File dir) {
        wdStack.clear();
        try {
            wdStack.add(dir.getCanonicalFile());
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public File pushd(File dir) {
        File lastWd = pwd();
        try {
            wdStack.add(dir.getCanonicalFile());
        } catch (IOException e) {
            e.printStackTrace();
        }
        return lastWd;
    }

    public File popd() {
        if (wdStack.size() > 1) {
            return wdStack.remove(wdStack.size() - 1);
        }
        return null;
    }

    public File resolveFile(String path) {
        if (path.startsWith("/")) {
            return new File(path);
        } else if (path.startsWith("~/")) {
            return new File(System.getProperty("user.home"), path.substring(2));
        } else {
            return new File(pwd(), path);
        }
    }

    public static void deleteTree(File dir) {
        emptyDirectory(dir);
        dir.delete();
    }

    public static void emptyDirectory(File dir) {
        File[] files = dir.listFiles();
        if (files == null) {
            return;
        }
        int len = files.length;
        for (int i = 0; i < len; i++) {
            File file = files[i];
            if (file.isDirectory()) {
                deleteTree(file);
            } else {
                file.delete();
            }
        }
    }

    public static void copyTree(File src, File dst) throws IOException {
        if (src.isFile()) {
            copyFile(src, dst);
        } else if (src.isDirectory()) {
            if (dst.exists()) {
                dst = new File(dst, src.getName());
                dst.mkdir();
            } else { // allows renaming dest dir
                dst.mkdirs();
            }
            File[] files = src.listFiles();
            for (File file : files) {
                copyTree(file, dst);
            }
        }
    }

    public static void copyFile(File src, File dst) throws IOException {
        if (dst.isDirectory()) {
            dst = new File(dst, src.getName());
        }
        FileInputStream in = null;
        FileOutputStream out = new FileOutputStream(dst);
        try {
            in = new FileInputStream(src);
            copy(in, out);
        } finally {
            if (in != null) {
                in.close();
            }
            out.close();
        }
    }

    public static void copy(InputStream in, OutputStream out) throws IOException {
        byte[] buffer = createBuffer(in.available());
        int read;
        while ((read = in.read(buffer)) != -1) {
            out.write(buffer, 0, read);
        }
    }

    public static String readContent(InputStream in) throws IOException {
        StringBuilder buf = new StringBuilder();
        byte[] bytes = new byte[1024 * 16];
        int r = -1;
        while ((r = in.read(bytes)) > -1) {
            buf.append(new String(bytes, 0, r));
        }
        return buf.toString();
    }

    public static List<String> readLines(InputStream in) throws IOException {
        List<String> lines = new ArrayList<String>();
        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new InputStreamReader(in));
            String line;
            while ((line = reader.readLine()) != null) {
                lines.add(line);
            }
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                }
            }
        }
        return lines;
    }

    public static List<String> readAndMergeLines(InputStream in) throws IOException {
        List<String> lines = readLines(in);
        ArrayList<String> result = new ArrayList<String>();
        StringBuilder lastLine = null;
        for (String line : lines) {
            line = line.trim();
            if (line.length() == 0 || line.startsWith("#")) {
                continue;
            }
            if (line.endsWith("\\")) {
                line = line.substring(0, line.length() - 1);
                if (lastLine != null) {
                    lastLine.append(line);
                } else {
                    lastLine = new StringBuilder(line);
                }
            } else {
                if (lastLine != null) {
                    result.add(lastLine.append(line).toString());
                    lastLine = null;
                } else {
                    result.add(line);
                }
            }
        }
        if (lastLine != null) {
            result.add(lastLine.toString());
        }
        return result;
    }

    public static String readFile(File file) throws IOException {
        FileInputStream in = null;
        try {
            in = new FileInputStream(file);
            return read(in);
        } finally {
            if (in != null) {
                in.close();
            }
        }
    }

    public static String read(InputStream in) throws IOException {
        StringBuilder sb = new StringBuilder();
        byte[] buffer = createBuffer(in.available());
        try {
            int read;
            while ((read = in.read(buffer)) != -1) {
                sb.append(new String(buffer, 0, read));
            }
        } finally {
            in.close();
        }
        return sb.toString();
    }

    public static void writeFile(File file, byte[] buf) throws IOException {
        FileOutputStream fos = null;
        try {
            fos = new FileOutputStream(file);
            fos.write(buf);
        } finally {
            if (fos != null) {
                fos.close();
            }
        }
    }

    public static void writeFile(File file, String buf) throws IOException {
        writeFile(file, buf.getBytes());
    }

    private static final int BUFFER_SIZE = 1024 * 64; // 64K

    private static final int MAX_BUFFER_SIZE = 1024 * 1024; // 64K

    private static final int MIN_BUFFER_SIZE = 1024 * 8; // 64K

    private static byte[] createBuffer(int preferredSize) {
        if (preferredSize < 1) {
            preferredSize = BUFFER_SIZE;
        }
        if (preferredSize > MAX_BUFFER_SIZE) {
            preferredSize = MAX_BUFFER_SIZE;
        } else if (preferredSize < MIN_BUFFER_SIZE) {
            preferredSize = MIN_BUFFER_SIZE;
        }
        return new byte[preferredSize];
    }

}
