/*
 * Decompiled with CFR 0.152.
 */
package org.artofsolving.jodconverter.office;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import org.apache.commons.io.FileUtils;
import org.artofsolving.jodconverter.office.OfficeException;
import org.artofsolving.jodconverter.office.OfficeUtils;
import org.artofsolving.jodconverter.office.OfficeVersionDescriptor;
import org.artofsolving.jodconverter.office.RetryTimeoutException;
import org.artofsolving.jodconverter.office.Retryable;
import org.artofsolving.jodconverter.office.StreamGobbler;
import org.artofsolving.jodconverter.office.TemporaryException;
import org.artofsolving.jodconverter.office.UnoUrl;
import org.artofsolving.jodconverter.process.ProcessManager;
import org.artofsolving.jodconverter.util.PlatformUtils;

class OfficeProcess {
    private final File officeHome;
    private final UnoUrl unoUrl;
    private final File templateProfileDir;
    private final File instanceProfileDir;
    private final File fakeBundlesDir;
    private final ProcessManager processManager;
    private Process process;
    private String pid;
    private List<String> lastCommand;
    private final Logger logger = Logger.getLogger(this.getClass().getName());
    protected static String COMMAND_ARG_PREFIX = "-";
    protected static OfficeVersionDescriptor versionDescriptor = null;
    public static final String STARTUP_WATCH_TIMEOUT = "jod.startup.watch.timeout";
    public static final int DEFAULT_STARTUP_WATCH_TIMEOUT = 7;

    public OfficeProcess(File officeHome, UnoUrl unoUrl, File templateProfileDir, ProcessManager processManager) {
        this(officeHome, unoUrl, templateProfileDir, processManager, false);
    }

    public OfficeProcess(File officeHome, UnoUrl unoUrl, File templateProfileDir, ProcessManager processManager, boolean useGnuStyleLongOptions) {
        this.officeHome = officeHome;
        this.unoUrl = unoUrl;
        this.templateProfileDir = templateProfileDir;
        this.instanceProfileDir = this.getInstanceProfileDir(unoUrl);
        this.fakeBundlesDir = this.getFakeBundlesDir();
        this.processManager = processManager;
        COMMAND_ARG_PREFIX = useGnuStyleLongOptions ? "--" : "-";
    }

    protected void determineOfficeVersion() throws IOException {
        File executable = OfficeUtils.getOfficeExecutable(this.officeHome);
        if (PlatformUtils.isWindows()) {
            versionDescriptor = OfficeVersionDescriptor.parseFromExecutableLocation(executable.getPath());
            return;
        }
        ArrayList<String> command = new ArrayList<String>();
        command.add(executable.getAbsolutePath());
        command.add("-help");
        command.add("-headless");
        command.add("-nocrashreport");
        command.add("-nofirststartwizard");
        command.add("-nolockcheck");
        command.add("-nologo");
        command.add("-norestore");
        command.add("-env:UserInstallation=" + OfficeUtils.toUrl(this.instanceProfileDir));
        ProcessBuilder processBuilder = new ProcessBuilder(command);
        processBuilder.redirectErrorStream(true);
        if (PlatformUtils.isWindows()) {
            this.addBasisAndUrePaths(processBuilder);
        }
        Process checkProcess = processBuilder.start();
        try {
            checkProcess.waitFor();
        }
        catch (InterruptedException e) {
            // empty catch block
        }
        InputStream in = checkProcess.getInputStream();
        String versionCheckOutput = this.read(in);
        versionDescriptor = new OfficeVersionDescriptor(versionCheckOutput);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private String read(InputStream in) throws IOException {
        StringBuilder sb = new StringBuilder();
        if (in.available() > 0) {
            byte[] buffer = new byte[in.available()];
            try {
                int read;
                while ((read = in.read(buffer)) != -1) {
                    sb.append(new String(buffer, 0, read));
                }
            }
            finally {
                in.close();
            }
        }
        return sb.toString();
    }

    public void start() throws IOException {
        if (versionDescriptor == null) {
            this.determineOfficeVersion();
        }
        if (versionDescriptor != null) {
            COMMAND_ARG_PREFIX = versionDescriptor.useGnuStyleLongOptions() ? "--" : "-";
        }
        this.logger.fine("OfficeProcess info:" + versionDescriptor.toString());
        this.doStart(false);
    }

    protected void manageProcessOutputs(Process process) {
        InputStream processOut = process.getInputStream();
        InputStream processError = process.getErrorStream();
        Thread to = new Thread(new StreamGobbler(processOut));
        to.setDaemon(true);
        to.start();
        Thread te = new Thread(new StreamGobbler(processError));
        te.setDaemon(true);
        te.start();
    }

    protected int getStartupWatchTime() {
        String timeOutStr = System.getProperty(STARTUP_WATCH_TIMEOUT, null);
        if (timeOutStr != null) {
            try {
                return new Integer(timeOutStr);
            }
            catch (NumberFormatException e) {
                return 7;
            }
        }
        return 7;
    }

    protected void doStart(boolean retry) throws IOException {
        String processRegex = "soffice.*" + Pattern.quote(this.unoUrl.getAcceptString());
        String existingPid = this.processManager.findPid(processRegex);
        if (existingPid != null) {
            throw new IllegalStateException(String.format("a process with acceptString '%s' is already running; pid %s", this.unoUrl.getAcceptString(), existingPid));
        }
        if (!retry) {
            this.prepareInstanceProfileDir();
            this.prepareFakeBundlesDir();
        }
        ArrayList<String> command = new ArrayList<String>();
        File executable = OfficeUtils.getOfficeExecutable(this.officeHome);
        command.add(executable.getAbsolutePath());
        command.add(COMMAND_ARG_PREFIX + "accept=" + this.unoUrl.getAcceptString() + ";urp;");
        command.add("-env:UserInstallation=" + OfficeUtils.toUrl(this.instanceProfileDir));
        if (!PlatformUtils.isWindows()) {
            command.add("-env:BUNDLED_EXTENSIONS=" + OfficeUtils.toUrl(this.fakeBundlesDir));
        }
        command.add(COMMAND_ARG_PREFIX + "headless");
        command.add(COMMAND_ARG_PREFIX + "nocrashreport");
        command.add(COMMAND_ARG_PREFIX + "nodefault");
        command.add(COMMAND_ARG_PREFIX + "nofirststartwizard");
        command.add(COMMAND_ARG_PREFIX + "nolockcheck");
        command.add(COMMAND_ARG_PREFIX + "nologo");
        command.add(COMMAND_ARG_PREFIX + "norestore");
        ProcessBuilder processBuilder = new ProcessBuilder(command);
        processBuilder.redirectErrorStream(true);
        this.lastCommand = command;
        if (PlatformUtils.isWindows()) {
            this.addBasisAndUrePaths(processBuilder);
        }
        this.logger.info(String.format("starting process with acceptString '%s' and profileDir '%s'", this.unoUrl, this.instanceProfileDir));
        this.process = processBuilder.start();
        int exitValue = 0;
        boolean exited = false;
        for (int i = 0; i < this.getStartupWatchTime() * 2; ++i) {
            try {
                Thread.sleep(500L);
            }
            catch (Exception e) {
                // empty catch block
            }
            try {
                exitValue = this.process.exitValue();
                exited = true;
                break;
            }
            catch (IllegalThreadStateException e) {
                continue;
            }
        }
        if (exited) {
            if (exitValue == 81) {
                this.logger.warning("Restarting OOo after code 81 ...");
                this.process = processBuilder.start();
                try {
                    Thread.sleep(1000L);
                }
                catch (Exception e) {}
            } else {
                this.logger.warning("Process exited with code " + exitValue);
            }
        }
        this.manageProcessOutputs(this.process);
        if (this.processManager.canFindPid()) {
            this.pid = this.processManager.findPid(processRegex);
            if (this.pid == null) {
                throw new IllegalStateException("started process, but can not find the pid, process is probably dead");
            }
            this.logger.info("started process : pid = " + this.pid);
        } else {
            this.logger.info("process started with PureJavaProcessManager - cannot check for pid");
        }
    }

    private File getInstanceProfileDir(UnoUrl unoUrl) {
        String dirName = ".jodconverter_" + unoUrl.getAcceptString().replace(',', '_').replace('=', '-');
        dirName = dirName + "_" + Thread.currentThread().getId();
        return new File(System.getProperty("java.io.tmpdir"), dirName);
    }

    private void prepareInstanceProfileDir() throws OfficeException {
        if (this.instanceProfileDir.exists()) {
            this.logger.warning(String.format("profile dir '%s' already exists; deleting", this.instanceProfileDir));
            this.deleteProfileDir();
        }
        if (this.templateProfileDir != null) {
            try {
                FileUtils.copyDirectory(this.templateProfileDir, this.instanceProfileDir);
            }
            catch (IOException ioException) {
                throw new OfficeException("failed to create profileDir", ioException);
            }
        }
    }

    public void deleteProfileDir() {
        if (this.instanceProfileDir != null) {
            try {
                FileUtils.deleteDirectory(this.instanceProfileDir);
            }
            catch (IOException ioException) {
                this.logger.warning(ioException.getMessage());
            }
        }
    }

    private File getFakeBundlesDir() {
        if (PlatformUtils.isWindows()) {
            return null;
        }
        String dirName = ".jodconverter_bundlesdir";
        dirName = dirName + "_" + Thread.currentThread().getId();
        return new File(System.getProperty("java.io.tmpdir"), dirName);
    }

    private void prepareFakeBundlesDir() throws OfficeException {
        if (this.fakeBundlesDir != null) {
            if (this.fakeBundlesDir.exists()) {
                this.deleteFakeBundlesDir();
            }
            this.fakeBundlesDir.mkdirs();
        }
    }

    public void deleteFakeBundlesDir() {
        if (this.fakeBundlesDir != null) {
            try {
                FileUtils.deleteDirectory(this.fakeBundlesDir);
            }
            catch (IOException ioException) {
                this.logger.warning(ioException.getMessage());
            }
        }
    }

    private void addBasisAndUrePaths(ProcessBuilder processBuilder) throws IOException {
        File ureBin = null;
        File basisProgram = null;
        File basisLink = new File(this.officeHome, "basis-link");
        if (!basisLink.isFile()) {
            File ureLink = new File(this.officeHome, "ure-link");
            if (!ureLink.isFile()) {
                this.logger.fine("no %OFFICE_HOME%/basis-link found; assuming it's OOo 2.x and we don't need to append URE and Basic paths");
                return;
            }
            ureBin = new File(new File(this.officeHome, FileUtils.readFileToString(ureLink).trim()), "bin");
        } else {
            String basisLinkText = FileUtils.readFileToString(basisLink).trim();
            File basisHome = new File(this.officeHome, basisLinkText);
            basisProgram = new File(basisHome, "program");
            File ureLink = new File(basisHome, "ure-link");
            String ureLinkText = FileUtils.readFileToString(ureLink).trim();
            File ureHome = new File(basisHome, ureLinkText);
            ureBin = new File(ureHome, "bin");
        }
        Map<String, String> environment = processBuilder.environment();
        String pathKey = "PATH";
        for (String key : environment.keySet()) {
            if (!"PATH".equalsIgnoreCase(key)) continue;
            pathKey = key;
        }
        String path = environment.get(pathKey) + ";" + ureBin.getAbsolutePath();
        if (basisProgram != null) {
            path = path + ";" + basisProgram.getAbsolutePath();
        }
        this.logger.fine(String.format("setting %s to \"%s\"", pathKey, path));
        environment.put(pathKey, path);
    }

    public boolean isRunning() {
        if (this.process == null) {
            return false;
        }
        try {
            this.process.exitValue();
            return false;
        }
        catch (IllegalThreadStateException exception) {
            return true;
        }
    }

    public int getExitCode(long retryInterval, long retryTimeout) throws RetryTimeoutException {
        try {
            ExitCodeRetryable retryable = new ExitCodeRetryable();
            retryable.execute(retryInterval, retryTimeout);
            return retryable.getExitCode();
        }
        catch (RetryTimeoutException retryTimeoutException) {
            throw retryTimeoutException;
        }
        catch (Exception exception) {
            throw new OfficeException("could not get process exit code", exception);
        }
    }

    public int forciblyTerminate(long retryInterval, long retryTimeout) throws IOException, RetryTimeoutException {
        this.logger.info(String.format("trying to forcibly terminate process: '" + this.unoUrl + "'" + (this.pid != null ? " (pid " + this.pid + ")" : ""), new Object[0]));
        this.processManager.kill(this.process, this.pid);
        return this.getExitCode(retryInterval, retryTimeout);
    }

    public String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append("\nOfficeHome : " + this.officeHome);
        sb.append("\nUnoUrl : " + this.unoUrl);
        if (this.lastCommand != null) {
            sb.append("\nCommand line : \n");
            for (String part : this.lastCommand) {
                sb.append(part + " ");
            }
        }
        sb.append("\nPID : " + this.pid);
        if (this.templateProfileDir == null) {
            sb.append("\ntemplateProfileDir : null");
        } else {
            sb.append("\ntemplateProfileDir : " + this.templateProfileDir.getAbsolutePath());
        }
        if (this.instanceProfileDir == null) {
            sb.append("\ninstanceProfileDir : null");
        } else {
            sb.append("\ninstanceProfileDir : " + this.instanceProfileDir.getAbsolutePath());
        }
        sb.append("\nProcessManager : " + this.processManager.getClass().getSimpleName());
        if (versionDescriptor != null) {
            sb.append("\nversionDescriptor : " + versionDescriptor.toString());
        }
        return sb.toString();
    }

    private class ExitCodeRetryable
    extends Retryable {
        private int exitCode;

        private ExitCodeRetryable() {
        }

        protected void attempt() throws TemporaryException, Exception {
            try {
                this.exitCode = OfficeProcess.this.process.exitValue();
            }
            catch (IllegalThreadStateException illegalThreadStateException) {
                throw new TemporaryException(illegalThreadStateException);
            }
        }

        public int getExitCode() {
            return this.exitCode;
        }
    }
}

