package org.eclipse.milo.opcua.sdk.client.model.nodes.objects;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import org.eclipse.milo.opcua.sdk.client.OpcUaClient;
import org.eclipse.milo.opcua.sdk.client.model.nodes.variables.PropertyTypeNode;
import org.eclipse.milo.opcua.sdk.client.model.types.objects.AuditEventType;
import org.eclipse.milo.opcua.sdk.client.nodes.UaNode;
import org.eclipse.milo.opcua.stack.core.AttributeId;
import org.eclipse.milo.opcua.stack.core.StatusCodes;
import org.eclipse.milo.opcua.stack.core.UaException;
import org.eclipse.milo.opcua.stack.core.types.builtin.DataValue;
import org.eclipse.milo.opcua.stack.core.types.builtin.DateTime;
import org.eclipse.milo.opcua.stack.core.types.builtin.ExpandedNodeId;
import org.eclipse.milo.opcua.stack.core.types.builtin.LocalizedText;
import org.eclipse.milo.opcua.stack.core.types.builtin.NodeId;
import org.eclipse.milo.opcua.stack.core.types.builtin.QualifiedName;
import org.eclipse.milo.opcua.stack.core.types.builtin.StatusCode;
import org.eclipse.milo.opcua.stack.core.types.builtin.Variant;
import org.eclipse.milo.opcua.stack.core.types.builtin.unsigned.UByte;
import org.eclipse.milo.opcua.stack.core.types.builtin.unsigned.UInteger;
import org.eclipse.milo.opcua.stack.core.types.enumerated.NodeClass;

public class AuditEventTypeNode extends BaseEventTypeNode implements AuditEventType {
    public AuditEventTypeNode(OpcUaClient client, NodeId nodeId, NodeClass nodeClass,
                              QualifiedName browseName, LocalizedText displayName, LocalizedText description,
                              UInteger writeMask, UInteger userWriteMask, UByte eventNotifier) {
        super(client, nodeId, nodeClass, browseName, displayName, description, writeMask, userWriteMask, eventNotifier);
    }

    @Override
    public DateTime getActionTimeStamp() throws UaException {
        PropertyTypeNode node = getActionTimeStampNode();
        return (DateTime) node.getValue().getValue().getValue();
    }

    @Override
    public void setActionTimeStamp(DateTime actionTimeStamp) throws UaException {
        PropertyTypeNode node = getActionTimeStampNode();
        node.setValue(new Variant(actionTimeStamp));
    }

    @Override
    public DateTime readActionTimeStamp() throws UaException {
        try {
            return readActionTimeStampAsync().get();
        } catch (ExecutionException | InterruptedException e) {
            throw UaException.extract(e).orElse(new UaException(StatusCodes.Bad_UnexpectedError, e));
        }
    }

    @Override
    public void writeActionTimeStamp(DateTime actionTimeStamp) throws UaException {
        try {
            writeActionTimeStampAsync(actionTimeStamp).get();
        } catch (ExecutionException | InterruptedException e) {
            throw UaException.extract(e).orElse(new UaException(StatusCodes.Bad_UnexpectedError, e));
        }
    }

    @Override
    public CompletableFuture<? extends DateTime> readActionTimeStampAsync() {
        return getActionTimeStampNodeAsync().thenCompose(node -> node.readAttributeAsync(AttributeId.Value)).thenApply(v -> (DateTime) v.getValue().getValue());
    }

    @Override
    public CompletableFuture<StatusCode> writeActionTimeStampAsync(DateTime actionTimeStamp) {
        DataValue value = DataValue.valueOnly(new Variant(actionTimeStamp));
        return getActionTimeStampNodeAsync()
            .thenCompose(node -> node.writeAttributeAsync(AttributeId.Value, value));
    }

    @Override
    public PropertyTypeNode getActionTimeStampNode() throws UaException {
        try {
            return getActionTimeStampNodeAsync().get();
        } catch (ExecutionException | InterruptedException e) {
            throw UaException.extract(e).orElse(new UaException(StatusCodes.Bad_UnexpectedError, e));
        }
    }

    @Override
    public CompletableFuture<? extends PropertyTypeNode> getActionTimeStampNodeAsync() {
        CompletableFuture<UaNode> future = getMemberNodeAsync("http://opcfoundation.org/UA/", "ActionTimeStamp", ExpandedNodeId.parse("nsu=http://opcfoundation.org/UA/;i=46"), false);
        return future.thenApply(node -> (PropertyTypeNode) node);
    }

    @Override
    public Boolean getStatus() throws UaException {
        PropertyTypeNode node = getStatusNode();
        return (Boolean) node.getValue().getValue().getValue();
    }

    @Override
    public void setStatus(Boolean status) throws UaException {
        PropertyTypeNode node = getStatusNode();
        node.setValue(new Variant(status));
    }

    @Override
    public Boolean readStatus() throws UaException {
        try {
            return readStatusAsync().get();
        } catch (ExecutionException | InterruptedException e) {
            throw UaException.extract(e).orElse(new UaException(StatusCodes.Bad_UnexpectedError, e));
        }
    }

    @Override
    public void writeStatus(Boolean status) throws UaException {
        try {
            writeStatusAsync(status).get();
        } catch (ExecutionException | InterruptedException e) {
            throw UaException.extract(e).orElse(new UaException(StatusCodes.Bad_UnexpectedError, e));
        }
    }

    @Override
    public CompletableFuture<? extends Boolean> readStatusAsync() {
        return getStatusNodeAsync().thenCompose(node -> node.readAttributeAsync(AttributeId.Value)).thenApply(v -> (Boolean) v.getValue().getValue());
    }

    @Override
    public CompletableFuture<StatusCode> writeStatusAsync(Boolean status) {
        DataValue value = DataValue.valueOnly(new Variant(status));
        return getStatusNodeAsync()
            .thenCompose(node -> node.writeAttributeAsync(AttributeId.Value, value));
    }

    @Override
    public PropertyTypeNode getStatusNode() throws UaException {
        try {
            return getStatusNodeAsync().get();
        } catch (ExecutionException | InterruptedException e) {
            throw UaException.extract(e).orElse(new UaException(StatusCodes.Bad_UnexpectedError, e));
        }
    }

    @Override
    public CompletableFuture<? extends PropertyTypeNode> getStatusNodeAsync() {
        CompletableFuture<UaNode> future = getMemberNodeAsync("http://opcfoundation.org/UA/", "Status", ExpandedNodeId.parse("nsu=http://opcfoundation.org/UA/;i=46"), false);
        return future.thenApply(node -> (PropertyTypeNode) node);
    }

    @Override
    public String getServerId() throws UaException {
        PropertyTypeNode node = getServerIdNode();
        return (String) node.getValue().getValue().getValue();
    }

    @Override
    public void setServerId(String serverId) throws UaException {
        PropertyTypeNode node = getServerIdNode();
        node.setValue(new Variant(serverId));
    }

    @Override
    public String readServerId() throws UaException {
        try {
            return readServerIdAsync().get();
        } catch (ExecutionException | InterruptedException e) {
            throw UaException.extract(e).orElse(new UaException(StatusCodes.Bad_UnexpectedError, e));
        }
    }

    @Override
    public void writeServerId(String serverId) throws UaException {
        try {
            writeServerIdAsync(serverId).get();
        } catch (ExecutionException | InterruptedException e) {
            throw UaException.extract(e).orElse(new UaException(StatusCodes.Bad_UnexpectedError, e));
        }
    }

    @Override
    public CompletableFuture<? extends String> readServerIdAsync() {
        return getServerIdNodeAsync().thenCompose(node -> node.readAttributeAsync(AttributeId.Value)).thenApply(v -> (String) v.getValue().getValue());
    }

    @Override
    public CompletableFuture<StatusCode> writeServerIdAsync(String serverId) {
        DataValue value = DataValue.valueOnly(new Variant(serverId));
        return getServerIdNodeAsync()
            .thenCompose(node -> node.writeAttributeAsync(AttributeId.Value, value));
    }

    @Override
    public PropertyTypeNode getServerIdNode() throws UaException {
        try {
            return getServerIdNodeAsync().get();
        } catch (ExecutionException | InterruptedException e) {
            throw UaException.extract(e).orElse(new UaException(StatusCodes.Bad_UnexpectedError, e));
        }
    }

    @Override
    public CompletableFuture<? extends PropertyTypeNode> getServerIdNodeAsync() {
        CompletableFuture<UaNode> future = getMemberNodeAsync("http://opcfoundation.org/UA/", "ServerId", ExpandedNodeId.parse("nsu=http://opcfoundation.org/UA/;i=46"), false);
        return future.thenApply(node -> (PropertyTypeNode) node);
    }

    @Override
    public String getClientAuditEntryId() throws UaException {
        PropertyTypeNode node = getClientAuditEntryIdNode();
        return (String) node.getValue().getValue().getValue();
    }

    @Override
    public void setClientAuditEntryId(String clientAuditEntryId) throws UaException {
        PropertyTypeNode node = getClientAuditEntryIdNode();
        node.setValue(new Variant(clientAuditEntryId));
    }

    @Override
    public String readClientAuditEntryId() throws UaException {
        try {
            return readClientAuditEntryIdAsync().get();
        } catch (ExecutionException | InterruptedException e) {
            throw UaException.extract(e).orElse(new UaException(StatusCodes.Bad_UnexpectedError, e));
        }
    }

    @Override
    public void writeClientAuditEntryId(String clientAuditEntryId) throws UaException {
        try {
            writeClientAuditEntryIdAsync(clientAuditEntryId).get();
        } catch (ExecutionException | InterruptedException e) {
            throw UaException.extract(e).orElse(new UaException(StatusCodes.Bad_UnexpectedError, e));
        }
    }

    @Override
    public CompletableFuture<? extends String> readClientAuditEntryIdAsync() {
        return getClientAuditEntryIdNodeAsync().thenCompose(node -> node.readAttributeAsync(AttributeId.Value)).thenApply(v -> (String) v.getValue().getValue());
    }

    @Override
    public CompletableFuture<StatusCode> writeClientAuditEntryIdAsync(String clientAuditEntryId) {
        DataValue value = DataValue.valueOnly(new Variant(clientAuditEntryId));
        return getClientAuditEntryIdNodeAsync()
            .thenCompose(node -> node.writeAttributeAsync(AttributeId.Value, value));
    }

    @Override
    public PropertyTypeNode getClientAuditEntryIdNode() throws UaException {
        try {
            return getClientAuditEntryIdNodeAsync().get();
        } catch (ExecutionException | InterruptedException e) {
            throw UaException.extract(e).orElse(new UaException(StatusCodes.Bad_UnexpectedError, e));
        }
    }

    @Override
    public CompletableFuture<? extends PropertyTypeNode> getClientAuditEntryIdNodeAsync() {
        CompletableFuture<UaNode> future = getMemberNodeAsync("http://opcfoundation.org/UA/", "ClientAuditEntryId", ExpandedNodeId.parse("nsu=http://opcfoundation.org/UA/;i=46"), false);
        return future.thenApply(node -> (PropertyTypeNode) node);
    }

    @Override
    public String getClientUserId() throws UaException {
        PropertyTypeNode node = getClientUserIdNode();
        return (String) node.getValue().getValue().getValue();
    }

    @Override
    public void setClientUserId(String clientUserId) throws UaException {
        PropertyTypeNode node = getClientUserIdNode();
        node.setValue(new Variant(clientUserId));
    }

    @Override
    public String readClientUserId() throws UaException {
        try {
            return readClientUserIdAsync().get();
        } catch (ExecutionException | InterruptedException e) {
            throw UaException.extract(e).orElse(new UaException(StatusCodes.Bad_UnexpectedError, e));
        }
    }

    @Override
    public void writeClientUserId(String clientUserId) throws UaException {
        try {
            writeClientUserIdAsync(clientUserId).get();
        } catch (ExecutionException | InterruptedException e) {
            throw UaException.extract(e).orElse(new UaException(StatusCodes.Bad_UnexpectedError, e));
        }
    }

    @Override
    public CompletableFuture<? extends String> readClientUserIdAsync() {
        return getClientUserIdNodeAsync().thenCompose(node -> node.readAttributeAsync(AttributeId.Value)).thenApply(v -> (String) v.getValue().getValue());
    }

    @Override
    public CompletableFuture<StatusCode> writeClientUserIdAsync(String clientUserId) {
        DataValue value = DataValue.valueOnly(new Variant(clientUserId));
        return getClientUserIdNodeAsync()
            .thenCompose(node -> node.writeAttributeAsync(AttributeId.Value, value));
    }

    @Override
    public PropertyTypeNode getClientUserIdNode() throws UaException {
        try {
            return getClientUserIdNodeAsync().get();
        } catch (ExecutionException | InterruptedException e) {
            throw UaException.extract(e).orElse(new UaException(StatusCodes.Bad_UnexpectedError, e));
        }
    }

    @Override
    public CompletableFuture<? extends PropertyTypeNode> getClientUserIdNodeAsync() {
        CompletableFuture<UaNode> future = getMemberNodeAsync("http://opcfoundation.org/UA/", "ClientUserId", ExpandedNodeId.parse("nsu=http://opcfoundation.org/UA/;i=46"), false);
        return future.thenApply(node -> (PropertyTypeNode) node);
    }
}
