package org.bouncycastle.pqc.legacy.math.linearalgebra;

import java.math.BigInteger;


/**
 * This interface defines a finite field element. It is implemented by the
 * class {@link GF2nElement}.
 *
 * @see GF2nElement
 */
public interface GFElement
{

    /**
     * @return a copy of this GFElement
     */
    Object clone();

    // /////////////////////////////////////////////////////////////////
    // comparison
    // /////////////////////////////////////////////////////////////////

    /**
     * Compare this curve with another object.
     *
     * @param other the other object
     * @return the result of the comparison
     */
    boolean equals(Object other);

    /**
     * @return the hash code of this element
     */
    int hashCode();

    /**
     * Checks whether this element is zero.
     *
     * @return  true</tt> if  this</tt> is the zero element
     */
    boolean isZero();

    /**
     * Checks whether this element is one.
     *
     * @return  true</tt> if  this</tt> is the one element
     */
    boolean isOne();

    // /////////////////////////////////////////////////////////////////////
    // arithmetic
    // /////////////////////////////////////////////////////////////////////

    /**
     * Compute the sum of this element and the addend.
     *
     * @param addend the addend
     * @return  this + other</tt> (newly created)
     */
    GFElement add(GFElement addend)
        throws RuntimeException;

    /**
     * Compute the sum of this element and the addend, overwriting this element.
     *
     * @param addend the addend
     */
    void addToThis(GFElement addend)
        throws RuntimeException;

    /**
     * Compute the difference of this element and  minuend</tt>.
     *
     * @param minuend the minuend
     * @return  this - minuend</tt> (newly created)
     */
    GFElement subtract(GFElement minuend)
        throws RuntimeException;

    /**
     * Compute the difference of this element and  minuend</tt>,
     * overwriting this element.
     *
     * @param minuend the minuend
     */
    void subtractFromThis(GFElement minuend);

    /**
     * Compute the product of this element and  factor</tt>.
     *
     * @param factor the factor
     * @return  this * factor</tt> (newly created)
     */
    GFElement multiply(GFElement factor)
        throws RuntimeException;

    /**
     * Compute  this * factor</tt> (overwrite  this</tt>).
     *
     * @param factor the factor
     */
    void multiplyThisBy(GFElement factor)
        throws RuntimeException;

    /**
     * Compute the multiplicative inverse of this element.
     *
     * @return  this -1</sup></tt> (newly created)
     * @throws ArithmeticException if  this</tt> is the zero element.
     */
    GFElement invert()
        throws ArithmeticException;

    // /////////////////////////////////////////////////////////////////////
    // conversion
    // /////////////////////////////////////////////////////////////////////

    /**
     * Returns this element as FlexiBigInt. The conversion is <a
     * href="https://grouper.ieee.org/groups/1363/">P1363</a>-conform.
     *
     * @return this element as BigInt
     */
    BigInteger toFlexiBigInt();

    /**
     * Returns this element as byte array. The conversion is <a href =
     * "https://grouper.ieee.org/groups/1363/">P1363</a>-conform.
     *
     * @return this element as byte array
     */
    byte[] toByteArray();

    /**
     * Return a String representation of this element.
     *
     * @return String representation of this element
     */
    String toString();

    /**
     * Return a String representation of this element.  radix</tt>
     * specifies the radix of the String representation.
     *
     * @param radix specifies the radix of the String representation
     * @return String representation of this element with the specified radix
     */
    String toString(int radix);

}
